# Copyright (C) Evan Goetz (2021)
#
# This file is part of pyDARM.
#
# pyDARM is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# pyDARM is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# pyDARM. If not, see <https://www.gnu.org/licenses/>.

# Coefficients copied from lines 184 to 248 of RCG 3.0.2 tagged release of
# <https://redoubt.ligo-wa.caltech.edu/svn/advLigoRTS/tags/advLigoRTS-3.0.2/\
# src/fe/controller.c>

from scipy import signal
import numpy as np


def iopdownsamplingfilters(sample_freq_str, filter_calc_method, rcg_ver):
    """
    Compute the transfer function for the DAQ downsampling filters

    Parameters
    ----------
    sample_freq_str : `str`
        Number of samples per second input
    filter_calc_method : `str`
        Filtering method. Choose between 'biquad' and 'df2'
    rcg_ver : `str`
        RCG version. Choose between 'v3' and 'v2' (only needed when also
        specifying '16k' and 'biquad')

    Returns
    -------
    iopfilter : :obj:`scipy.signal.StateSpace`
        State space object

    """

    if filter_calc_method == 'biquad':
        # /* Recalculated filters in biquad form */
        # /* Oversamping base rate is 64K */
        if sample_freq_str == '32k':
            # /* Coeffs for the 2x downsampling (32K system) filter */
            # static double __attribute__ ((unused)) feCoeff2x[9] =
            #        {0.053628649721183,
            #         0.2568759660371100, -0.3225906481359000,
            #         1.2568801238621801, 1.6774135096891700,
            #         -0.2061764045745400, -1.0941543149527400,
            #         2.0846376586498803, 2.1966597482716801};
            gain = 0.053628649721183
            sos = np.array([[0.2568759660371100, -0.3225906481359000,
                             1.2568801238621801, 1.6774135096891700],
                            [-0.2061764045745400, -1.0941543149527400,
                             2.0846376586498803, 2.1966597482716801]])
        elif sample_freq_str == '16k':
            if rcg_ver == 'v2':
                # Coefficients copied from lines 289 to 354 of RCG 2.6.2 tagged
                # release of
                # <https://redoubt.ligo-wa.caltech.edu/svn/advLigoRTS/tags/\
                # advLigoRTS-2.6.2/src/fe/controller.c>
                # /* Coeffs for the 4x downsampling (16K system) filter */
                # static double __attribute__ ((unused)) feCoeff4x[9] =
                #        {0.014805052402446,
                #         0.7166258547451800, -0.0683289874517300,
                #         0.3031629575762000, 0.5171469569032900,
                #         0.6838596423885499, -0.2534855521841101,
                #         1.6838609161411500, 1.7447155374502499};
                gain = 0.014805052402446
                sos = np.array([[0.7166258547451800, -0.0683289874517300,
                                 0.3031629575762000, 0.5171469569032900],
                                [0.6838596423885499, -0.2534855521841101,
                                 1.6838609161411500, 1.7447155374502499]])
            else:
                # /* RCG V3.0 Coeffs for the 4x downsampling (16K system)
                #    filter */
                # static double __attribute__ ((unused)) feCoeff4x[9] =
                #        {0.054285975,
                #         0.3890221, -0.17645085, -0.0417771600000001,
                #         0.41775916, 0.52191125, -0.37884382,
                #         1.52190741336686, 1.69347541336686};
                gain = 0.054285975
                sos = np.array([[0.3890221, -0.17645085,
                                 -0.0417771600000001, 0.41775916],
                                [0.52191125, -0.37884382,
                                 1.52190741336686, 1.69347541336686]])
        elif sample_freq_str == '4k':
            # // New Brian Lantz 4k decimation filter
            # static double __attribute__ ((unused)) feCoeff16x[9] =
            #       {0.010203728365,
            #        0.8052941009065100, -0.0241751519071000,
            #        0.3920490703701900, 0.5612099784288400,
            #        0.8339678987936501, -0.0376022631287799,
            #        -0.0131581721533700, 0.1145865116421301};
            gain = 0.010203728365
            sos = np.array([[0.8052941009065100, -0.0241751519071000,
                             0.3920490703701900, 0.5612099784288400],
                            [0.8339678987936501, -0.0376022631287799,
                             -0.0131581721533700, 0.1145865116421301]])
        elif sample_freq_str == '2k':
            # /* Coeffs for the 32x downsampling filter (2K system) per Brian Lantz May 5,
            # * 2009 */
            # static double __attribute__( ( unused ) ) feCoeff32x[ 9 ] =
            #   {0.010581064947739,
            #    0.90444302586137004, -0.0063413204375699639,
            #    -0.056459743474659874, 0.032075154877300172,
            #    0.92390910024681006, -0.0097523655540199261,
            #    0.077383808424050127, 0.14238741130302013};
            gain = 0.010581064947739
            sos = np.array([[0.90444302586137004, -0.0063413204375699639,
                             -0.056459743474659874, 0.032075154877300172],
                            [0.92390910024681006, -0.0097523655540199261,
                             0.077383808424050127, 0.14238741130302013]])
        elif sample_freq_str == '2k_rana':
            # /* Coeffs for the 32x downsampling filter (2K system) */
            # /* Original Rana coeffs from 40m lab elog */
            # static double __attribute__ ((unused)) feCoeff32x[9] =
            #        {0.0001104130574447,
            #         0.9701834961388200, -0.0010837026165800,
            #         -0.0200761119821899, 0.0085463156103800,
            #         0.9871502388637901, -0.0039246182095299,
            #         3.9871502388637898, 3.9960753817904697};
            gain = 0.0001104130574447
            sos = np.array([[0.9701834961388200, -0.0010837026165800,
                             -0.0200761119821899, 0.0085463156103800],
                            [0.9871502388637901, -0.0039246182095299,
                             3.9871502388637898, 3.9960753817904697]])

        # Put it into standard form

        # BiQuad coefficients are reported in rows of [a11, a12, c1, and c2]
        # instead of rows of [a1, a2, b1, b2] for DF2. So, we need to
        # compute a1, a2, b1, and b2. The relation is derived from pg 11 of
        # G0900928-v1
        rows = len(sos[:, 0])
        b0 = np.ones((rows, 1))
        b1 = -np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) + \
            np.reshape(sos[:, 2], (len(sos[:, 2]), 1)) - 1
        b2 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - \
            np.reshape(sos[:, 1], (len(sos[:, 1]), 1)) - \
            np.reshape(sos[:, 2], (len(sos[:, 2]), 1)) + \
            np.reshape(sos[:, 3], (len(sos[:, 3]), 1))
        a0 = np.ones((rows, 1))
        a1 = -np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - 1
        a2 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - \
            np.reshape(sos[:, 1], (len(sos[:, 1]), 1))

    if filter_calc_method == 'df2':
        # /* Oversamping base rate is 64K */
        if sample_freq_str == '32k':
            # /* Coeffs for the 2x downsampling (32K system) filter */
            # static double __attribute__ ((unused)) feCoeff2x[9] =
            #        {0.053628649721183,
            #        -1.25687596603711,    0.57946661417301,
            #         0.00000415782507,    1.00000000000000,
            #        -0.79382359542546,    0.88797791037820,
            #         1.29081406322442,    1.00000000000000};
            gain = 0.053628649721183
            sos = np.array([[-1.25687596603711,    0.57946661417301,
                             0.00000415782507,    1.00000000000000],
                            [-0.79382359542546,    0.88797791037820,
                             1.29081406322442,    1.00000000000000]])
        elif sample_freq_str == '16k':
            # /* Coeffs for the 4x downsampling (16K system) filter */
            # static double __attribute__ ((unused)) feCoeff4x[9] =
            #        {0.014805052402446,
            #        -1.71662585474518,    0.78495484219691,
            #        -1.41346289716898,   0.99893884152400,
            #        -1.68385964238855,    0.93734519457266,
            #         0.00000127375260,   0.99819981588176};
            gain = 0.014805052402446
            sos = np.array([[-1.71662585474518,    0.78495484219691,
                             -1.41346289716898,   0.99893884152400],
                            [-1.68385964238855,    0.93734519457266,
                             0.00000127375260,   0.99819981588176]])
        elif sample_freq_str == '4k':
            # // New Brian Lantz 4k decimation filter
            # static double __attribute__ ((unused)) feCoeff16x[9] =
            #        {0.010203728365,
            #        -1.80529410090651,   0.82946925281361,
            #        -1.41324503053632,   0.99863016087226,
            #        -1.83396789879365,   0.87157016192243,
            #        -1.84712607094702,   0.99931484571793};
            gain = 0.010203728365
            sos = np.array([[-1.80529410090651,   0.82946925281361,
                             -1.41324503053632,   0.99863016087226],
                            [-1.83396789879365,   0.87157016192243,
                             -1.84712607094702,   0.99931484571793]])
        elif sample_freq_str == '2k':
            # /* Coeffs for the 32x downsampling filter (2K system)
            #    per Brian Lantz May 5, 2009 */
            # static double __attribute__ ((unused)) feCoeff32x[9] =
            #        {0.010581064947739,
            #        -1.90444302586137,    0.91078434629894,
            #        -1.96090276933603,    0.99931924465090,
            #        -1.92390910024681,    0.93366146580083,
            #        -1.84652529182276,    0.99866506867980};
            gain = 0.010581064947739
            sos = np.array([[-1.90444302586137,    0.91078434629894,
                             -1.96090276933603,    0.99931924465090],
                            [-1.92390910024681,    0.93366146580083,
                             -1.84652529182276,    0.99866506867980]])

        # Put it into standard form
        #
        # The coefficients stored in the C code assume a form of the second
        # order section where the leading coefficient in the numerator and the
        # denominator are both 1. Here we insert those two 1's into the matrix,
        # and also swap the numerator and denominator.
        rows = len(sos[:, 0])
        b0 = np.ones((rows, 1))
        b1 = np.reshape(sos[:, 2], (len(sos[:, 2]), 1))
        b2 = np.reshape(sos[:, 3], (len(sos[:, 3]), 1))
        a0 = np.ones((rows, 1))
        a1 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1))
        a2 = np.reshape(sos[:, 1], (len(sos[:, 1]), 1))

    soscoef = np.concatenate((b0, b1, b2, a0, a1, a2), axis=1)

    [z, p, k] = signal.sos2zpk(soscoef)
    [A, B, C, D] = signal.zpk2ss(z, p, k*gain)
    iopfilter = signal.StateSpace(A, B, C, D, dt=1.0/2**16)
    return iopfilter


def daqdownsamplingfilters(from_freq, to_freq, filter_method, rcg_ver):
    """
    Compute the transfer function for the DAQ downsampling filters

    Parameters
    ----------
    from_freq : `int`
        Number of samples per second input
    to_freq : `int`
        Number of samples per second output
    filter_method : `str`
        Filtering method. Choose between 'biquad' and 'df2'
    rcg_ver : `str`
        RCG version. Choose between 'v3' and 'v2'

    Returns
    -------
    daq_filter : :obj:`scipy.signal.StateSpace`
        State space object

    """

    sampleFreqRatio = from_freq / to_freq

    if filter_method == 'biquad':
        if sampleFreqRatio == 2:
            if rcg_ver == 'v3':
                # static double dCoeff2x[13] = {0.02717257186578,
                #         -0.1159055409088, -0.40753832312918,
                #          2.66236735378793, 3.37073457156755,
                #         -0.49505157452475, -1.10461941102831,
                #          1.40184470311617, 1.79227686661261,
                #         -0.74143396593712, -1.62740819248313,
                #          0.72188475979666, 0.83591053325065};
                gain = 0.02717257186578
                sos = np.array([[-0.1159055409088, -0.40753832312918,
                                 2.66236735378793, 3.37073457156755],
                                [-0.49505157452475, -1.10461941102831,
                                 1.40184470311617, 1.79227686661261],
                                [-0.74143396593712, -1.62740819248313,
                                 0.72188475979666, 0.83591053325065]])
            else:
                # static double dCoeff2x[13] =
                #                 { 0.014605318489015,
                #                   0.0061330534633199, -0.3067718521410701,
                #                   1.0061297523961799, 1.6898970546512500,
                #                  -0.1416634327119900, -0.7218542081231900,
                #                   1.1639392962956800, 1.5641813375750999,
                #                  -0.2223002987598800, -1.1002072247518702,
                #                   2.4322961493728101, 2.5543892233808201};
                gain = 0.014605318489015
                sos = np.array([[0.0061330534633199, -0.3067718521410701,
                                 1.0061297523961799, 1.6898970546512500],
                                [-0.1416634327119900, -0.7218542081231900,
                                 1.1639392962956800, 1.5641813375750999],
                                [-0.2223002987598800, -1.1002072247518702,
                                 2.4322961493728101, 2.5543892233808201]])
        if sampleFreqRatio == 4:
            if rcg_ver == 'v3':
                # static double dCoeff4x[13] = {0.00426219526013,
                #         0.46640482430571, -0.10620935923005,
                #         2.50932081620118, 2.93670663266542,
                #         0.43602772908265, -0.31016854747127,
                #         0.75143527373544, 1.00523899718152,
                #         0.44571894955428, -0.47692045639835,
                #         0.36664098129003, 0.4440015753374};
                gain = 0.00426219526013
                sos = np.array([[0.46640482430571, -0.10620935923005,
                                 2.50932081620118, 2.93670663266542],
                                [0.43602772908265, -0.31016854747127,
                                 0.75143527373544, 1.00523899718152],
                                [0.44571894955428, -0.47692045639835,
                                 0.36664098129003, 0.4440015753374]])
            else:
                # static double dCoeff4x[13] =
                #         {0.0032897561126272,
                #         0.5262606025434300, -0.0761411615790100,
                #         0.1130468884239699, 0.5092319101840799,
                #         0.5730930806734700, -0.1812069602474000,
                #         0.4535162982982299, 0.6837579627174200,
                #         0.6560226277436600, -0.2732748286521299,
                #         1.9218491283142201, 1.9903219392643201};
                gain = 0.0032897561126272
                sos = np.array([[0.5262606025434300, -0.0761411615790100,
                                 0.1130468884239699, 0.5092319101840799],
                                [0.5730930806734700, -0.1812069602474000,
                                 0.4535162982982299, 0.6837579627174200],
                                [0.6560226277436600, -0.2732748286521299,
                                 1.9218491283142201, 1.9903219392643201]])
        if sampleFreqRatio == 8:
            if rcg_ver == 'v3':
                # static double dCoeff8x[13] = {0.00162185538923,
                #         0.73342532779703, -0.02862365091314,
                #         1.44110125961504, 1.67905228090487,
                #         0.77657563380963, -0.08304311675394,
                #         0.18851328163424, 0.32889453107067,
                #         0.83213081484618, -0.12573495191273,
                #         0.0940911979108501, 0.13622543115194};
                gain = 0.00162185538923
                sos = np.array([[0.73342532779703, -0.02862365091314,
                                 1.44110125961504, 1.67905228090487],
                                [0.77657563380963, -0.08304311675394,
                                 0.18851328163424, 0.3288945310706],
                                [0.83213081484618, -0.12573495191273,
                                 0.0940911979108501, 0.13622543115194]])
            else:
                # static double dCoeff8x[13] =
                #         {0.0019451746049824,
                #         0.7581968768203300, -0.0208065724495400,
                #         -0.0885007357744900, 0.1313472736383900,
                #         0.8177667403664499, -0.0484926355391700,
                #         0.0804638221493899, 0.1881713856561398,
                #         0.8916285940607900, -0.0710046059171400,
                #         1.0789961400704899, 0.9517899355931499};
                gain = 0.0019451746049824
                sos = np.array([[0.7581968768203300, -0.0208065724495400,
                                 -0.0885007357744900, 0.1313472736383900],
                                [0.8177667403664499, -0.0484926355391700,
                                 0.0804638221493899, 0.1881713856561398],
                                [0.8916285940607900, -0.0710046059171400,
                                 1.0789961400704899, 0.9517899355931499]])
        if sampleFreqRatio == 16:
            if rcg_ver == 'v3':
                # static double dCoeff16x[13] = {0.00112590539483,
                #         0.86616831686611, -0.00753654634986012,
                #         0.48586805026482, 0.61216318704885,
                #         0.90508101474565, -0.0215349711544799,
                #         0.0149886842581499, 0.08837269835802,
                #         0.94631370100442, -0.0320417955561,
                #         0.0141281606027401, 0.0357726640422202};
                gain = 0.00112590539483
                sos = np.array([[0.86616831686611, -0.00753654634986012,
                                 0.48586805026482, 0.61216318704885],
                                [0.90508101474565, -0.0215349711544799,
                                 0.0149886842581499, 0.08837269835802],
                                [0.94631370100442, -0.0320417955561,
                                 0.0141281606027401, 0.0357726640422202]])
            else:
                # static double dCoeff16x[13] =
                #         {0.0010292496296221,
                #         0.8821795273490400, -0.0050811632845900,
                #         -0.0793910648843901, 0.0333482444819799,
                #         0.9213891226266799, -0.0116267556958600,
                #         -0.0147463176669200, 0.0522378040105400,
                #         0.9631231120650900, -0.0166459584776699,
                #         0.3524249070977401, 0.3726558365549801};
                gain = 0.0010292496296221
                sos = np.array([[0.8821795273490400, -0.0050811632845900,
                                 -0.0793910648843901, 0.0333482444819799],
                                [0.9213891226266799, -0.0116267556958600,
                                 -0.0147463176669200, 0.0522378040105400],
                                [0.9631231120650900, -0.0166459584776699,
                                 0.3524249070977401, 0.3726558365549801]])
        if sampleFreqRatio == 32:
            if rcg_ver == 'v3':
                # static double dCoeff32x[13] = {0.00102945292275,
                #   0.93288074072411, -0.00194092797014001,
                #   0.10751900551591, 0.17269733682166,
                #   0.9570539169953, -0.00548573773340011,
                #   -0.0149997987966302, 0.0224605464746699,
                #   0.98100244642901, -0.00807148639261013,
                #   -0.00189235941040011, 0.00903370776797985};
                gain = 0.00102945292275
                sos = np.array([[0.93288074072411, -0.00194092797014001,
                                 0.10751900551591, 0.17269733682166],
                                [0.9570539169953, -0.00548573773340011,
                                 -0.0149997987966302, 0.0224605464746699],
                                [0.98100244642901, -0.00807148639261013,
                                 -0.00189235941040011, 0.00903370776797985]])
            else:
                # static double dCoeff32x[13] =
                #         {0.00099066651652901,
                #         0.9407723671890900, -0.0013021996687700,
                #         -0.0495970976842000, 0.0083283354579400,
                #         0.9629941014830901, -0.0029486095232200,
                #         -0.0209238527730700, 0.0131334362206199,
                #         0.9856499106827501, -0.0041756491626799,
                #         0.0901459629393901, 0.1003204030939602};
                gain = 0.00099066651652901
                sos = np.array([[0.9407723671890900, -0.0013021996687700,
                                 -0.0495970976842000, 0.0083283354579400],
                                [0.9629941014830901, -0.0029486095232200,
                                 -0.0209238527730700, 0.0131334362206199],
                                [0.9856499106827501, -0.0041756491626799,
                                 0.0901459629393901, 0.1003204030939602]])
        if sampleFreqRatio == 64:
            if rcg_ver == 'v3':
                # static double dCoeff64x[13] = {0.00101894798776,
                #         0.96638168022541, -0.000492974627960052,
                #         0.01147570619135, 0.04460105133798,
                #         0.97969775930388, -0.00138449271550001,
                #         -0.0132857101503898, 0.00563203783023014,
                #         0.99249184543014, -0.0020244997813601,
                #         -0.00322227927422025, 0.00226137551427974};
                gain = 0.00101894798776
                sos = np.array([[0.96638168022541, -0.000492974627960052,
                                 0.01147570619135, 0.04460105133798],
                                [0.97969775930388, -0.00138449271550001,
                                 -0.0132857101503898, 0.00563203783023014],
                                [0.99249184543014, -0.0020244997813601,
                                 -0.00322227927422025, 0.00226137551427974]])
            else:
                # static double dCoeff64x[13] =
                #         {9.117708813402705e-05,
                #         0.9884411147668981, -0.0002966277054097,
                #         -0.0097345801570645, 0.0015276773706276,
                #         0.9958880030565560, -0.0005974359311846,
                #         0.0001273101399277, 0.0036418711521875,
                #         -0.0079580196372524, -0.0079580196372524,
                #         1.9920419803627476, 1.9920419803627476};
                gain = 9.117708813402705e-05
                sos = np.array([[0.9884411147668981, -0.0002966277054097,
                                 -0.0097345801570645, 0.0015276773706276],
                                [0.9958880030565560, -0.0005974359311846,
                                 0.0001273101399277, 0.0036418711521875],
                                [-0.0079580196372524, -0.0079580196372524,
                                 1.9920419803627476, 1.9920419803627476]])
        if sampleFreqRatio == 128:
            if rcg_ver == 'v3':
                # static double dCoeff128x[13] = {0.00102359688929,
                #         0.98317523053482, -0.000124254191099959,
                #         -0.00545789721985002, 0.01124261805423,
                #         0.9901470788001, -0.000347773996469902,
                #         -0.00809690612593994, 0.00140824107749005,
                #         0.9967468102523, -0.000506888877139899,
                #         -0.00218112074794985, 0.000565180122610309};
                gain = 0.00102359688929
                sos = np.array([[0.98317523053482, -0.000124254191099959,
                                 -0.00545789721985002, 0.01124261805423],
                                [0.9901470788001, -0.000347773996469902,
                                 -0.00809690612593994, 0.00140824107749005],
                                [0.9967468102523, -0.000506888877139899,
                                 -0.00218112074794985, 0.000565180122610309]])
            else:
                # static double dCoeff128x[13] =
                #         {4.580254440937838e-05,
                #         0.9942785038368447, -0.0000743648654229,
                #         -0.0052652981867765, 0.0003818331109557,
                #         0.9980915472248755, -0.0001494958409671,
                #         -0.0008478627563595, 0.0009110941777979,
                #         -0.0039867920479109, -0.0039867920479109,
                #         1.9960132079520891, 1.9960132079520891};
                gain = 4.580254440937838e-05
                sos = np.array([[0.9942785038368447, -0.0000743648654229,
                                 -0.0052652981867765, 0.0003818331109557],
                                [0.9980915472248755, -0.0001494958409671,
                                 -0.0008478627563595, 0.0009110941777979],
                                [-0.0039867920479109, -0.0039867920479109,
                                 1.9960132079520891, 1.9960132079520891]])
        if sampleFreqRatio == 256:
            if rcg_ver == 'v3':
                # static double dCoeff256x[13] = {0.00102849104272,
                #         0.99158359864769, -3.11926170200039e-05,
                #         -0.00556878740432998, 0.00281642133096005,
                #         0.99514878857652, -8.7150981279982e-05,
                #         -0.00441208984599983, 0.000351970596200069,
                #         0.99849900371282, -0.000126813612729926,
                #         -0.00123294150072994, 0.000141241173720053};
                gain = 0.00102849104272
                sos = np.array([[0.99158359864769, -3.11926170200039e-05,
                                 -0.00556878740432998, 0.00281642133096005],
                                [0.99514878857652, -8.7150981279982e-05,
                                 -0.00441208984599983, 0.000351970596200069],
                                [0.99849900371282, -0.000126813612729926,
                                 -0.00123294150072994, 0.000141241173720053]])
            else:
                # static double dCoeff256x[13] =
                #         {2.296084727953743e-05,
                #         0.9971538121386385, -0.0000186174099586,
                #         -0.0027321307580492, 0.0000954396933539,
                #         0.9990827274780281, -0.0000373907471043,
                #         -0.0006520772714058, 0.0002278045034618,
                #         -0.0019953660573223, -0.0019953660573223,
                #         1.9980046339426778, 1.9980046339426778};
                gain = 2.296084727953743e-05
                sos = np.array([[0.9971538121386385, -0.0000186174099586,
                                 -0.0027321307580492, 0.0000954396933539],
                                [0.9990827274780281, -0.0000373907471043,
                                 -0.0006520772714058, 0.0002278045034618],
                                [-0.0019953660573223, -0.0019953660573223,
                                 1.9980046339426778, 1.9980046339426778]])

        # Put it into standard form

        # BiQuad coefficients are reported in rows of [a11, a12, c1, and c2]
        # instead of rows of [a1, a2, b1, b2] for DF2. So, we need to
        # compute a1, a2, b1, and b2. The relation is derived from pg 11 of
        # G0900928-v1
        rows = len(sos[:, 0])
        b0 = np.ones((rows, 1))
        b1 = -np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) + \
            np.reshape(sos[:, 2], (len(sos[:, 2]), 1)) - 1
        b2 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - \
            np.reshape(sos[:, 1], (len(sos[:, 1]), 1)) - \
            np.reshape(sos[:, 2], (len(sos[:, 2]), 1)) + \
            np.reshape(sos[:, 3], (len(sos[:, 3]), 1))
        a0 = np.ones((rows, 1))
        a1 = -np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - 1
        a2 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1)) - \
            np.reshape(sos[:, 1], (len(sos[:, 1]), 1))

    if filter_method == 'df2':
        if sampleFreqRatio == 2:
            if rcg_ver == 'v3':
                # static double dCoeff2x[13] = {0.02717257186578,
                #   -0.8840944590912, 0.29163278222038, 1.77827289469673, 1,
                #   -0.50494842547525, 0.60956783650356, 0.89689627764092, 1,
                #   -0.25856603406288, 0.88597422654601, 0.46331872573378, 1};
                gain = 0.02717257186578
                sos = np.array([[-0.8840944590912, 0.29163278222038,
                                 1.77827289469673, 1],
                                [-0.50494842547525, 0.60956783650356,
                                 0.89689627764092, 1],
                                [-0.25856603406288, 0.88597422654601,
                                 0.46331872573378, 1]])
            else:
                # static double dCoeff2x[13] =
                #   {0.014605318489015,
                #    -1.00613305346332, 0.31290490560439,
                #    -0.00000330106714, 0.99667220785946,
                #    -0.85833656728801, 0.58019077541120,
                #    0.30560272900767, 0.98043281669062,
                #    -0.77769970124012, 0.87790692599199,
                #    1.65459644813269, 1.00000000000000};
                gain = 0.014605318489015
                sos = np.array([[-1.00613305346332, 0.31290490560439,
                                 -0.00000330106714, 0.99667220785946],
                                [-0.85833656728801, 0.58019077541120,
                                 0.30560272900767, 0.98043281669062],
                                [-0.77769970124012, 0.87790692599199,
                                 1.65459644813269, 1.00000000000000]])
        if sampleFreqRatio == 4:
            if rcg_ver == 'v3':
                # static double dCoeff4x[13] = {0.00426219526013,
                #   -1.46640482430571, 0.57261418353576, 1.04291599189547, 1,
                #   -1.43602772908265, 0.74619627655392, -0.68459245534721, 1,
                #   -1.44571894955428, 0.92263940595263, -1.07907796826425, 1};
                gain = 0.00426219526013
                sos = np.array([[-1.46640482430571, 0.57261418353576,
                                 1.04291599189547, 1],
                                [-1.43602772908265, 0.74619627655392,
                                 -0.68459245534721, 1],
                                [-1.44571894955428, 0.92263940595263,
                                 -1.07907796826425, 1]])
            else:
                # static double dCoeff4x[13] =
                #   {0.0032897561126272,
                #    -1.52626060254343, 0.60240176412244,
                #    -1.41321371411946, 0.99858678588255,
                #    -1.57309308067347, 0.75430004092087,
                #    -1.11957678237524, 0.98454170534006,
                #    -1.65602262774366, 0.92929745639579,
                #    0.26582650057056, 0.99777026734589};
                gain = 0.0032897561126272
                sos = np.array([[-1.52626060254343, 0.60240176412244,
                                 -1.41321371411946, 0.99858678588255],
                                [-1.57309308067347, 0.75430004092087,
                                 -1.11957678237524, 0.98454170534006],
                                [-1.65602262774366, 0.92929745639579,
                                 0.26582650057056, 0.99777026734589]])
        if sampleFreqRatio == 8:
            if rcg_ver == 'v3':
                # static double dCoeff8x[13] = {0.00162185538923,
                #   -1.73342532779703, 0.76204897871017, -0.29232406818199, 1,
                #   -1.77657563380963, 0.85961875056357, -1.58806235217539, 1,
                #   -1.83213081484618, 0.95786576675891, -1.73803961693533, 1};
                gain = 0.00162185538923
                sos = np.array([[-1.73342532779703, 0.76204897871017,
                                 -0.29232406818199, 1],
                                [-1.77657563380963, 0.85961875056357,
                                 -1.58806235217539, 1],
                                [-1.83213081484618, 0.95786576675891,
                                 -1.73803961693533, 1]])
            else:
                # static double dCoeff8x[13] =
                #   {0.0019451746049824,
                #    -1.75819687682033, 0.77900344926987,
                #    -1.84669761259482, 0.99885145868275,
                #    -1.81776674036645, 0.86625937590562,
                #    -1.73730291821706, 0.97396693941237,
                #    -1.89162859406079, 0.96263319997793,
                #    -0.81263245399030, 0.83542699550059};
                gain = 0.0019451746049824
                sos = np.array([[-1.75819687682033, 0.77900344926987,
                                 -1.84669761259482, 0.99885145868275],
                                [-1.81776674036645, 0.86625937590562,
                                 -1.73730291821706, 0.97396693941237],
                                [-1.89162859406079, 0.96263319997793,
                                 -0.81263245399030, 0.83542699550059]])
        if sampleFreqRatio == 16:
            if rcg_ver == 'v3':
                # static double dCoeff16x[13] = {0.00112590539483,
                #   -1.86616831686611, 0.87370486321597, -1.38030026660129, 1,
                #   -1.90508101474565, 0.92661598590013, -1.8900923304875, 1,
                #   -1.94631370100442, 0.97835549656052, -1.93218554040168, 1};
                gain = 0.00112590539483
                sos = np.array([[-1.86616831686611, 0.87370486321597,
                                 -1.38030026660129, 1],
                                [-1.90508101474565, 0.92661598590013,
                                 -1.8900923304875, 1],
                                [-1.94631370100442, 0.97835549656052,
                                 -1.93218554040168, 1]])
            else:
                # static double dCoeff16x[13] =
                #   {0.0010292496296221,
                #    -1.88217952734904, 0.88726069063363,
                #    -1.96157059223343, 1.00000000000000,
                #    -1.92138912262668, 0.93301587832254,
                #    -1.93613544029360, 1.00000000000000,
                #    -1.96312311206509, 0.97976907054276,
                #    -1.61069820496735, 1.00000000000000};
                gain = 0.0010292496296221
                sos = np.array([[-1.88217952734904, 0.88726069063363,
                                 -1.96157059223343, 1.00000000000000],
                                [-1.92138912262668, 0.93301587832254,
                                 -1.93613544029360, 1.00000000000000],
                                [-1.96312311206509, 0.97976907054276,
                                 -1.61069820496735, 1.00000000000000]])
        if sampleFreqRatio == 32:
            if rcg_ver == 'v3':
                # static double dCoeff32x[13] = {0.00102945292275,
                #   -1.93288074072411, 0.93482166869425, -1.8253617352082, 1,
                #   -1.9570539169953, 0.9625396547287, -1.97205371579193, 1,
                #   -1.98100244642901, 0.98907393282162, -1.98289480583941, 1};
                gain = 0.00102945292275
                sos = np.array([[-1.93288074072411, 0.93482166869425,
                                 -1.8253617352082, 1],
                                [-1.9570539169953, 0.9625396547287,
                                 -1.97205371579193, 1],
                                [-1.98100244642901, 0.98907393282162,
                                 -1.98289480583941, 1]])
            else:
                # static double dCoeff32x[13] =
                #   {0.00099066651652901, -1.94077236718909, 0.94207456685786,
                #    -1.99036946487329, 1.00000000000000, -1.96299410148309,
                #    0.96594271100631, -1.98391795425616, 1.00000000000000,
                #    -1.98564991068275, 0.98982555984543, -1.89550394774336,
                #    1.00000000000000};
                gain = 0.00099066651652901
                sos = np.array([[-1.94077236718909, 0.94207456685786,
                                 -1.99036946487329, 1.00000000000000],
                                [-1.96299410148309, 0.96594271100631,
                                 -1.98391795425616, 1.00000000000000],
                                [-1.98564991068275, 0.98982555984543,
                                 -1.89550394774336, 1.00000000000000]])
        if sampleFreqRatio == 64:
            if rcg_ver == 'v3':
                # static double dCoeff64x[13] = {0.00101894798776,
                #   -1.96638168022541, 0.96687465485337, -1.95490597403406, 1,
                #   -1.97969775930388, 0.98108225201938, -1.99298346945427, 1,
                #   -1.99249184543014, 0.9945163452115, -1.99571412470436, 1};
                gain = 0.00101894798776
                sos = np.array([[-1.96638168022541, 0.96687465485337,
                                 -1.95490597403406, 1],
                                [-1.97969775930388, 0.98108225201938,
                                 -1.99298346945427, 1],
                                [-1.99249184543014, 0.9945163452115,
                                 -1.99571412470436, 1]])
            else:
                # static double dCoeff64x[13] =
                #   {9.117708813402705e-05, -1.9884411147668981,
                #    0.9887377424723078, -1.9981756949239626,
                #    1.0000000000000000, -1.9958880030565560,
                #    0.9964854389877406, -1.9957606929166283,
                #    1.0000000000000002, -0.9920419803627476,
                #    0.0000000000000000, 1.0000000000000000,
                #    0.0000000000000000};
                gain = 9.117708813402705e-05
                sos = np.array([[-1.9884411147668981, 0.9887377424723078,
                                 -1.9981756949239626, 1.0000000000000000],
                                [-1.9958880030565560, 0.9964854389877406,
                                 -1.9957606929166283, 1.0000000000000002],
                                [-0.9920419803627476, 0.0000000000000000,
                                 1.0000000000000000, 0.0000000000000000]])
        if sampleFreqRatio == 128:
            if rcg_ver == 'v3':
                # static double dCoeff128x[13] = {0.00102359688929,
                #   -1.98317523053482, 0.98329948472592, -1.98863312775467, 1,
                #   -1.9901470788001, 0.99049485279657, -1.99824398492604, 1,
                #   -1.9967468102523, 0.99725369912944, -1.99892793100025, 1};
                gain = 0.00102359688929
                sos = np.array([[-1.98317523053482, 0.98329948472592,
                                 -1.98863312775467, 1],
                                [-1.9901470788001, 0.99049485279657,
                                 -1.99824398492604, 1],
                                [-1.9967468102523, 0.99725369912944,
                                 -1.99892793100025, 1]])
            else:
                # static double dCoeff128x[13] =
                #   {4.580254440937838e-05, -1.9942785038368447,
                #    0.9943528687022676, -1.9995438020236211,
                #    0.9999999999999998, -1.9980915472248755,
                #    0.9982410430658426, -1.9989394099812350,
                #    1.0000000000000000, -0.9960132079520891,
                #    0.0000000000000000,  1.0000000000000000,
                #    0.0000000000000000};
                gain = 4.580254440937838e-05
                sos = np.array([[-1.9942785038368447, 0.9943528687022676,
                                 -1.9995438020236211, 0.9999999999999998],
                                [-1.9980915472248755, 0.9982410430658426,
                                 -1.9989394099812350, 1.0000000000000000],
                                [-0.9960132079520891, 0.0000000000000000,
                                 1.0000000000000000, 0.0000000000000000]])
        if sampleFreqRatio == 256:
            if rcg_ver == 'v3':
                # static double dCoeff256x[13] = {0.00102849104272,
                #   -1.99158359864769, 0.99161479126471, -1.99715238605202, 1,
                #   -1.99514878857652, 0.9952359395578, -1.99956087842252, 1,
                #   -1.99849900371282, 0.99862581732555, -1.99973194521355, 1};
                gain = 0.00102849104272
                sos = np.array([[-1.99158359864769, 0.99161479126471,
                                 -1.99715238605202, 1],
                                [-1.99514878857652, 0.9952359395578,
                                 -1.99956087842252, 1],
                                [-1.99849900371282, 0.99862581732555,
                                 -1.99973194521355, 1]])
            else:
                # static double dCoeff256x[13] = {2.296084727953743e-05,
                #   -1.9971538121386385, 0.9971724295485971,
                #   -1.9998859428966878, 1.0000000000000002,
                #   -1.9990827274780281, 0.9991201182251324,
                #   -1.9997348047494339,  0.9999999999999999,
                #   -0.9980046339426777, 0.0000000000000000,
                #   1.0000000000000000,  0.0000000000000000};
                gain = 2.296084727953743e-05
                sos = np.array([[-1.9971538121386385, 0.9971724295485971,
                                 -1.9998859428966878, 1.0000000000000002],
                                [-1.9990827274780281, 0.9991201182251324,
                                 -1.9997348047494339, 0.9999999999999999],
                                [-0.9980046339426777, 0.0000000000000000,
                                 1.0000000000000000, 0.0000000000000000]])

        # Put it into standard form
        #
        # The coefficients stored in the C code assume a form of the second
        # order section where the leading coefficient in the numerator and the
        # denominator are both 1. Here we insert those two 1's into the matrix,
        # and also swap the numerator and denominator.
        rows = len(sos[:, 0])
        b0 = np.ones((rows, 1))
        b1 = np.reshape(sos[:, 2], (len(sos[:, 2]), 1))
        b2 = np.reshape(sos[:, 3], (len(sos[:, 3]), 1))
        a0 = np.ones((rows, 1))
        a1 = np.reshape(sos[:, 0], (len(sos[:, 0]), 1))
        a2 = np.reshape(sos[:, 1], (len(sos[:, 1]), 1))

    soscoef = np.concatenate((b0, b1, b2, a0, a1, a2), axis=1)

    [z, p, k] = signal.sos2zpk(soscoef)
    [A, B, C, D] = signal.zpk2ss(z, p, k*gain)
    daq_filter = signal.StateSpace(A, B, C, D, dt=1.0/from_freq)
    return daq_filter
