"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
/**
 * Property to specify if the compute environment
 * uses On-Demand, SpotFleet, Fargate, or Fargate Spot compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
    /**
     * Resources will be Fargate resources.
     */
    ComputeResourceType["FARGATE"] = "FARGATE";
    /**
     * Resources will be Fargate Spot resources.
     *
     * Fargate Spot uses spare capacity in the AWS cloud to run your fault-tolerant,
     * time-flexible jobs at up to a 70% discount. If AWS needs the resources back,
     * jobs running on Fargate Spot will be interrupted with two minutes of notification.
     */
    ComputeResourceType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_ComputeEnvironmentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ComputeEnvironment);
            }
            throw error;
        }
        const isFargate = ComputeResourceType.FARGATE === props.computeResources?.type
            || ComputeResourceType.FARGATE_SPOT === props.computeResources?.type;
        ;
        this.validateProps(props, isFargate);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        const useLaunchTemplateNetworkInterface = props.computeResources?.launchTemplate?.useNetworkInterfaceSecurityGroups ? true : false;
        this.connections = this.buildConnections(useLaunchTemplateNetworkInterface, props.computeResources?.vpc, props.computeResources?.securityGroups);
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.getSecurityGroupIds(useLaunchTemplateNetworkInterface),
                spotIamFleetRole: spotFleetRole?.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
                ...(!isFargate ? {
                    allocationStrategy: props.computeResources.allocationStrategy
                        || (props.computeResources.type === ComputeResourceType.SPOT
                            ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                            : AllocationStrategy.BEST_FIT),
                    instanceRole: props.computeResources.instanceRole
                        ? props.computeResources.instanceRole
                        : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                            roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                    managedPolicies: [
                                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                    ],
                                }).roleName],
                        }).attrArn,
                    instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                    minvCpus: props.computeResources.minvCpus || 0,
                } : {}),
            };
        }
        const computeEnvironment = new aws_batch_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole?.roleArn
                ?? new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const computeEnvironmentName = stack.splitArn(computeEnvironmentArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed ?? true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props, isFargate) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        if (props.computeResources) {
            if (isFargate) {
                // VALIDATE FOR FARGATE
                // Bid percentage cannot be set for Fargate evnvironments
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Bid percentage must not be set for Fargate compute environments');
                }
                // Allocation strategy cannot be set for Fargate evnvironments
                if (props.computeResources.allocationStrategy !== undefined) {
                    throw new Error('Allocation strategy must not be set for Fargate compute environments');
                }
                // Desired vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.desiredvCpus !== undefined) {
                    throw new Error('Desired vCPUs must not be set for Fargate compute environments');
                }
                // Image ID cannot be set for Fargate evnvironments
                if (props.computeResources.image !== undefined) {
                    throw new Error('Image must not be set for Fargate compute environments');
                }
                // Instance types cannot be set for Fargate evnvironments
                if (props.computeResources.instanceTypes !== undefined) {
                    throw new Error('Instance types must not be set for Fargate compute environments');
                }
                // EC2 key pair cannot be set for Fargate evnvironments
                if (props.computeResources.ec2KeyPair !== undefined) {
                    throw new Error('EC2 key pair must not be set for Fargate compute environments');
                }
                // Instance role cannot be set for Fargate evnvironments
                if (props.computeResources.instanceRole !== undefined) {
                    throw new Error('Instance role must not be set for Fargate compute environments');
                }
                // Launch template cannot be set for Fargate evnvironments
                if (props.computeResources.launchTemplate !== undefined) {
                    throw new Error('Launch template must not be set for Fargate compute environments');
                }
                // Min vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.minvCpus !== undefined) {
                    throw new Error('Min vCPUs must not be set for Fargate compute environments');
                }
                // Placement group cannot be set for Fargate evnvironments
                if (props.computeResources.placementGroup !== undefined) {
                    throw new Error('Placement group must not be set for Fargate compute environments');
                }
                // Spot fleet role cannot be set for Fargate evnvironments
                if (props.computeResources.spotFleetRole !== undefined) {
                    throw new Error('Spot fleet role must not be set for Fargate compute environments');
                }
            }
            else {
                // VALIDATE FOR ON_DEMAND AND SPOT
                if (props.computeResources.minvCpus) {
                    // minvCpus cannot be less than 0
                    if (props.computeResources.minvCpus < 0) {
                        throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                    }
                    // minvCpus cannot exceed max vCpus
                    if (props.computeResources.maxvCpus &&
                        props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                        throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                    }
                }
                // Check if both launchTemplateId and launchTemplateName are provided
                if (props.computeResources.launchTemplate &&
                    (props.computeResources.launchTemplate.launchTemplateId && props.computeResources.launchTemplate.launchTemplateName)) {
                    throw new Error('You must specify either the launch template ID or launch template name in the request, but not both.');
                }
                // Check if both launchTemplateId and launchTemplateName are missing
                if (props.computeResources.launchTemplate &&
                    (!props.computeResources.launchTemplate.launchTemplateId && !props.computeResources.launchTemplate.launchTemplateName)) {
                    throw new Error('You must specify either the launch template ID or launch template name in the request.');
                }
                // useNetworkInterfaceSecurityGroups cannot have securityGroups defined
                if (props.computeResources.launchTemplate?.useNetworkInterfaceSecurityGroups &&
                    props.computeResources.securityGroups) {
                    throw new Error('securityGroups cannot be specified if launchTemplate useNetworkInterfaceSecurityGroups is active');
                }
                // Setting a bid percentage is only allowed on SPOT resources +
                // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
                if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                    // VALIDATE FOR ON_DEMAND
                    // Bid percentage is not allowed
                    if (props.computeResources.bidPercentage !== undefined) {
                        throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                    }
                    // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                    if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                        throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                    }
                }
                else if (props.computeResources.type === ComputeResourceType.SPOT) {
                    // VALIDATE FOR SPOT
                    // Bid percentage must be from 0 - 100
                    if (props.computeResources.bidPercentage !== undefined &&
                        (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                        throw new Error('Bid percentage can only be a value between 0 and 100');
                    }
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildConnections(useLaunchTemplateNetworkInterface, vpc, securityGroups) {
        if (vpc === undefined || useLaunchTemplateNetworkInterface) {
            return new ec2.Connections({});
        }
        if (securityGroups === undefined) {
            return new ec2.Connections({
                securityGroups: [
                    new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }),
                ],
            });
        }
        return new ec2.Connections({ securityGroups });
    }
    ;
    getSecurityGroupIds(useLaunchTemplateInterface) {
        if (this.connections === undefined ||
            useLaunchTemplateInterface) {
            return undefined;
        }
        return this.connections.securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        if (props.computeResources?.allocationStrategy && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.env.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "@aws-cdk/aws-batch-alpha.ComputeEnvironment", version: "2.66.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbXB1dGUtZW52aXJvbm1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsMkNBQTJDO0FBQzNDLDJDQUEyQztBQUMzQyw2Q0FBb0U7QUFFcEUscURBQThEO0FBRTlEOzs7R0FHRztBQUNILElBQVksbUJBd0JYO0FBeEJELFdBQVksbUJBQW1CO0lBQzdCOztPQUVHO0lBQ0gsd0NBQWlCLENBQUE7SUFFakI7O09BRUc7SUFDSCxvQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCwwQ0FBbUIsQ0FBQTtJQUVuQjs7Ozs7O09BTUc7SUFDSCxvREFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBeEJXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBd0I5QjtBQUVEOzs7R0FHRztBQUNILElBQVksa0JBcUJYO0FBckJELFdBQVksa0JBQWtCO0lBQzVCOzs7T0FHRztJQUNILDJDQUFxQixDQUFBO0lBRXJCOzs7O09BSUc7SUFDSCxtRUFBNkMsQ0FBQTtJQUU3Qzs7Ozs7T0FLRztJQUNILHlFQUFtRCxDQUFBO0FBQ3JELENBQUMsRUFyQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFxQjdCO0FBa1JEOzs7O0dBSUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLHNCQUFRO0lBdUM5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQWlDLEVBQUUsT0FBTyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFO1FBQ3pHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxzQkFBc0I7U0FDM0MsQ0FBQyxDQUFDOzs7Ozs7K0NBMUNNLGtCQUFrQjs7OztRQTRDM0IsTUFBTSxTQUFTLEdBQUcsbUJBQW1CLENBQUMsT0FBTyxLQUFLLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxJQUFJO2VBQ3pFLG1CQUFtQixDQUFDLFlBQVksS0FBSyxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDO1FBQUEsQ0FBQztRQUV4RSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUVyQyxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbkQsSUFBSSxnQkFBNEUsQ0FBQztRQUVqRixNQUFNLGlDQUFpQyxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxjQUFjLEVBQUUsaUNBQWlDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1FBRW5JLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGlDQUFpQyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRWpKLGlFQUFpRTtRQUNqRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBRW5ELGdCQUFnQixHQUFHO2dCQUNqQixhQUFhLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWE7Z0JBQ25ELFlBQVksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsWUFBWTtnQkFDakQsVUFBVSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUM3QyxPQUFPLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPO2dCQUM1RixjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWM7Z0JBQ3JELFFBQVEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxJQUFJLEdBQUc7Z0JBQ2hELGNBQWMsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYztnQkFDckQsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGlDQUFpQyxDQUFDO2dCQUM3RSxnQkFBZ0IsRUFBRSxhQUFhLEVBQUUsT0FBTztnQkFDeEMsT0FBTyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQyxTQUFTO2dCQUM5RixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG9CQUFvQjtnQkFDakQsSUFBSSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksbUJBQW1CLENBQUMsU0FBUztnQkFDbEUsR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztvQkFDZixrQkFBa0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCOzJCQUN4RCxDQUNELEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSTs0QkFDdEQsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHVCQUF1Qjs0QkFDNUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FDaEM7b0JBQ0gsWUFBWSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO3dCQUMvQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVk7d0JBQ3JDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7NEJBQ3JELEtBQUssRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7b0NBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztvQ0FDeEQsZUFBZSxFQUFFO3dDQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsa0RBQWtELENBQUM7cUNBQy9GO2lDQUNGLENBQUMsQ0FBQyxRQUFRLENBQUM7eUJBQ2IsQ0FBQyxDQUFDLE9BQU87b0JBQ1osYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDO29CQUM1RSxRQUFRLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsSUFBSSxDQUFDO2lCQUMvQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDUixDQUFDO1NBQ0g7UUFFRCxNQUFNLGtCQUFrQixHQUFHLElBQUksaUNBQXFCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNyRSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN6QyxnQkFBZ0I7WUFDaEIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXLEVBQUUsT0FBTzttQkFDbEMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxnQ0FBZ0MsRUFBRTtvQkFDdEQsZUFBZSxFQUFFO3dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsa0NBQWtDLENBQUM7cUJBQy9FO29CQUNELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsQ0FBQztpQkFDM0QsQ0FBQyxDQUFDLE9BQU87WUFDWixLQUFLLEVBQUUsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztZQUN6RixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxXQUFXO1NBQ3RELENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUU7WUFDeEQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUU7WUFDaEYsT0FBTyxFQUFFLE9BQU87WUFDaEIsUUFBUSxFQUFFLHFCQUFxQjtZQUMvQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUNyRjtJQXRIRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMseUJBQXlCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUscUJBQTZCO1FBQ2pHLE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sc0JBQXNCLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQixDQUFDLENBQUMsWUFBYSxDQUFDO1FBRWxILE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0IsMEJBQXFCLEdBQUcscUJBQXFCLENBQUM7Z0JBQzlDLDJCQUFzQixHQUFHLHNCQUFzQixDQUFDO1lBQ2xFLENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBdUdPLFNBQVMsQ0FBQyxLQUE4QjtRQUM5QyxPQUFPLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDO0tBQzlCO0lBRUQ7O09BRUc7SUFDSyxhQUFhLENBQUMsS0FBOEIsRUFBRSxTQUFrQjtRQUN0RSxJQUFJLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDdkIsT0FBTztTQUNSO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLG1GQUFtRixDQUFDLENBQUM7U0FDdEc7UUFFRCxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNqRSxNQUFNLElBQUksS0FBSyxDQUFDLDJFQUEyRSxDQUFDLENBQUM7U0FDOUY7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLFNBQVMsRUFBRTtnQkFDYix1QkFBdUI7Z0JBRXZCLHlEQUF5RDtnQkFDekQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO2lCQUNwRjtnQkFFRCw4REFBOEQ7Z0JBQzlELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixLQUFLLFNBQVMsRUFBRTtvQkFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO2lCQUN6RjtnQkFFRCx3REFBd0Q7Z0JBQ3hELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7b0JBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztpQkFDbkY7Z0JBRUQsbURBQW1EO2dCQUNuRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEtBQUssU0FBUyxFQUFFO29CQUM5QyxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7aUJBQzNFO2dCQUVELHlEQUF5RDtnQkFDekQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxpRUFBaUUsQ0FBQyxDQUFDO2lCQUNwRjtnQkFFRCx1REFBdUQ7Z0JBQ3ZELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFVBQVUsS0FBSyxTQUFTLEVBQUU7b0JBQ25ELE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQztpQkFDbEY7Z0JBRUQsd0RBQXdEO2dCQUN4RCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLEtBQUssU0FBUyxFQUFFO29CQUNyRCxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7aUJBQ25GO2dCQUVELDBEQUEwRDtnQkFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxLQUFLLFNBQVMsRUFBRTtvQkFDdkQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO2lCQUNyRjtnQkFFRCxvREFBb0Q7Z0JBQ3BELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsS0FBSyxTQUFTLEVBQUU7b0JBQ2pELE1BQU0sSUFBSSxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztpQkFDL0U7Z0JBRUQsMERBQTBEO2dCQUMxRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEtBQUssU0FBUyxFQUFFO29CQUN2RCxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUM7aUJBQ3JGO2dCQUVELDBEQUEwRDtnQkFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTtvQkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsQ0FBQyxDQUFDO2lCQUNyRjthQUNGO2lCQUFNO2dCQUNMLGtDQUFrQztnQkFDbEMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFO29CQUNuQyxpQ0FBaUM7b0JBQ2pDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsR0FBRyxDQUFDLEVBQUU7d0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztxQkFDeEY7b0JBRUQsbUNBQW1DO29CQUNuQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO3dCQUNqQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUU7d0JBQ25FLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztxQkFDM0U7aUJBQ0Y7Z0JBRUQscUVBQXFFO2dCQUNyRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjO29CQUN2QyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO29CQUN0SCxNQUFNLElBQUksS0FBSyxDQUFDLHNHQUFzRyxDQUFDLENBQUM7aUJBQ3pIO2dCQUVELG9FQUFvRTtnQkFDcEUsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYztvQkFDdkMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLGtCQUFrQixDQUFDLEVBQUU7b0JBQ3hILE1BQU0sSUFBSSxLQUFLLENBQUMsd0ZBQXdGLENBQUMsQ0FBQztpQkFDM0c7Z0JBRUQsdUVBQXVFO2dCQUN2RSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsaUNBQWlDO29CQUN4RSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxFQUFHO29CQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLGtHQUFrRyxDQUFDLENBQUM7aUJBQ3JIO2dCQUVELCtEQUErRDtnQkFDL0QsMERBQTBEO2dCQUMxRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsU0FBUyxFQUFFO29CQUNqRSx5QkFBeUI7b0JBRXpCLGdDQUFnQztvQkFDaEMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVMsRUFBRTt3QkFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxtR0FBbUcsQ0FBQyxDQUFDO3FCQUN0SDtvQkFFRCxvREFBb0Q7b0JBQ3BELElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsS0FBSyxrQkFBa0IsQ0FBQyx1QkFBdUIsRUFBRTt3QkFDekksTUFBTSxJQUFJLEtBQUssQ0FBQyx1SEFBdUgsQ0FBQyxDQUFDO3FCQUMxSTtpQkFDRjtxQkFBTSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSSxFQUFFO29CQUNuRSxvQkFBb0I7b0JBRXBCLHNDQUFzQztvQkFDdEMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVM7d0JBQ3BELENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsR0FBRyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsR0FBRyxHQUFHLENBQUMsRUFBRTt3QkFDMUYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO3FCQUN6RTtpQkFDRjthQUNGO1NBQ0Y7S0FDRjtJQUVPLGtCQUFrQixDQUFDLGFBQWtDO1FBQzNELElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUMvQixPQUFPO2dCQUNMLFNBQVM7YUFDVixDQUFDO1NBQ0g7UUFFRCxPQUFPLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFzQixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztLQUN2RTtJQUVPLGdCQUFnQixDQUFDLGlDQUEwQyxFQUFFLEdBQWMsRUFBRSxjQUFvQztRQUV2SCxJQUFJLEdBQUcsS0FBSyxTQUFTLElBQUksaUNBQWlDLEVBQUc7WUFDM0QsT0FBTyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDaEM7UUFFRCxJQUFJLGNBQWMsS0FBSyxTQUFTLEVBQUU7WUFDaEMsT0FBTyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUM7Z0JBQ3pCLGNBQWMsRUFBRTtvQkFDZCxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLHlCQUF5QixFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUM7aUJBQ2hFO2FBQ0YsQ0FBQyxDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7S0FDaEQ7SUFBQSxDQUFDO0lBRU0sbUJBQW1CLENBQUMsMEJBQW1DO1FBQzdELElBQUksSUFBSSxDQUFDLFdBQVcsS0FBSyxTQUFTO1lBQ2hDLDBCQUEwQixFQUFHO1lBQzdCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUF5QixFQUFFLEVBQUUsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDLENBQUM7S0FDbEc7SUFFRDs7Ozs7T0FLRztJQUNLLGdCQUFnQixDQUFDLEtBQThCO1FBQ3JELElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsS0FBSyxrQkFBa0IsQ0FBQyxRQUFRLEVBQUU7WUFDM0gsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLEVBQUU7Z0JBQ3hDLE9BQU8sS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQzthQUM3QztpQkFBTSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSSxFQUFFO2dCQUNuRSxPQUFPLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSx5QkFBeUIsRUFDekQsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsU0FBUyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQU8sOEVBQThFLENBQUMsQ0FBQzthQUN2STtTQUNGO1FBRUQsT0FBTyxTQUFTLENBQUM7S0FDbEI7O0FBM1RILGdEQTRUQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IEFybkZvcm1hdCwgSVJlc291cmNlLCBSZXNvdXJjZSwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbkNvbXB1dGVFbnZpcm9ubWVudCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1iYXRjaCc7XG5cbi8qKlxuICogUHJvcGVydHkgdG8gc3BlY2lmeSBpZiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudFxuICogdXNlcyBPbi1EZW1hbmQsIFNwb3RGbGVldCwgRmFyZ2F0ZSwgb3IgRmFyZ2F0ZSBTcG90IGNvbXB1dGUgcmVzb3VyY2VzLlxuICovXG5leHBvcnQgZW51bSBDb21wdXRlUmVzb3VyY2VUeXBlIHtcbiAgLyoqXG4gICAqIFJlc291cmNlcyB3aWxsIGJlIEVDMiBPbi1EZW1hbmQgcmVzb3VyY2VzLlxuICAgKi9cbiAgT05fREVNQU5EID0gJ0VDMicsXG5cbiAgLyoqXG4gICAqIFJlc291cmNlcyB3aWxsIGJlIEVDMiBTcG90RmxlZXQgcmVzb3VyY2VzLlxuICAgKi9cbiAgU1BPVCA9ICdTUE9UJyxcblxuICAvKipcbiAgICogUmVzb3VyY2VzIHdpbGwgYmUgRmFyZ2F0ZSByZXNvdXJjZXMuXG4gICAqL1xuICBGQVJHQVRFID0gJ0ZBUkdBVEUnLFxuXG4gIC8qKlxuICAgKiBSZXNvdXJjZXMgd2lsbCBiZSBGYXJnYXRlIFNwb3QgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBGYXJnYXRlIFNwb3QgdXNlcyBzcGFyZSBjYXBhY2l0eSBpbiB0aGUgQVdTIGNsb3VkIHRvIHJ1biB5b3VyIGZhdWx0LXRvbGVyYW50LFxuICAgKiB0aW1lLWZsZXhpYmxlIGpvYnMgYXQgdXAgdG8gYSA3MCUgZGlzY291bnQuIElmIEFXUyBuZWVkcyB0aGUgcmVzb3VyY2VzIGJhY2ssXG4gICAqIGpvYnMgcnVubmluZyBvbiBGYXJnYXRlIFNwb3Qgd2lsbCBiZSBpbnRlcnJ1cHRlZCB3aXRoIHR3byBtaW51dGVzIG9mIG5vdGlmaWNhdGlvbi5cbiAgICovXG4gIEZBUkdBVEVfU1BPVCA9ICdGQVJHQVRFX1NQT1QnLFxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGhvdyB0byBwcmVwYXJlIGNvbXB1dGUgcmVzb3VyY2VzXG4gKiB0aGF0IGFyZSBwcm92aXNpb25lZCBmb3IgYSBjb21wdXRlIGVudmlyb25tZW50LlxuICovXG5leHBvcnQgZW51bSBBbGxvY2F0aW9uU3RyYXRlZ3kge1xuICAvKipcbiAgICogQmF0Y2ggd2lsbCB1c2UgdGhlIGJlc3QgZml0dGluZyBpbnN0YW5jZSB0eXBlIHdpbGwgYmUgdXNlZFxuICAgKiB3aGVuIGFzc2lnbmluZyBhIGJhdGNoIGpvYiBpbiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqL1xuICBCRVNUX0ZJVCA9ICdCRVNUX0ZJVCcsXG5cbiAgLyoqXG4gICAqIEJhdGNoIHdpbGwgc2VsZWN0IGFkZGl0aW9uYWwgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGFyZ2UgZW5vdWdoIHRvXG4gICAqIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvclxuICAgKiBpbnN0YW5jZSB0eXBlcyB3aXRoIGEgbG93ZXIgY29zdCBwZXIgdW5pdCB2Q1BVLlxuICAgKi9cbiAgQkVTVF9GSVRfUFJPR1JFU1NJVkUgPSAnQkVTVF9GSVRfUFJPR1JFU1NJVkUnLFxuXG4gIC8qKlxuICAgKiBUaGlzIGlzIG9ubHkgYXZhaWxhYmxlIGZvciBTcG90IEluc3RhbmNlIGNvbXB1dGUgcmVzb3VyY2VzIGFuZCB3aWxsIHNlbGVjdFxuICAgKiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGVzIHRoYXQgYXJlIGxhcmdlIGVub3VnaCB0byBtZWV0IHRoZSByZXF1aXJlbWVudHMgb2ZcbiAgICogdGhlIGpvYnMgaW4gdGhlIHF1ZXVlLCB3aXRoIGEgcHJlZmVyZW5jZSBmb3IgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGVzc1xuICAgKiBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqL1xuICBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCA9ICdTUE9UX0NBUEFDSVRZX09QVElNSVpFRCcsXG59XG5cbi8qKlxuICogTGF1bmNoIHRlbXBsYXRlIHByb3BlcnR5IHNwZWNpZmljYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb24ge1xuICAvKipcbiAgICogVGhlIExhdW5jaCB0ZW1wbGF0ZSBJRC4gTXV0dWFsbHkgZXhjbHVzaXZlIHdpdGggYGxhdW5jaFRlbXBsYXRlTmFtZWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbGF1bmNoIHRlbXBsYXRlIGlkIHByb3ZpZGVkXG4gICAqL1xuICByZWFkb25seSBsYXVuY2hUZW1wbGF0ZUlkPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIExhdW5jaCB0ZW1wbGF0ZSBuYW1lLiBNdXR1YWxseSBleGNsdXNpdmUgd2l0aCBgbGF1bmNoVGVtcGxhdGVJZGBcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBsYXVuY2ggdGVtcGxhdGUgbmFtZSBwcm92aWRlZFxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGVOYW1lPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGxhdW5jaCB0ZW1wbGF0ZSB2ZXJzaW9uIHRvIGJlIHVzZWQgKG9wdGlvbmFsKS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgZGVmYXVsdCB2ZXJzaW9uIG9mIHRoZSBsYXVuY2ggdGVtcGxhdGVcbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb24/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBVc2Ugc2VjdXJpdHkgZ3JvdXBzIGRlZmluZWQgaW4gdGhlIGxhdW5jaCB0ZW1wbGF0ZSBuZXR3b3JrIGludGVyZmFjZXNcbiAgICpcbiAgICogSW4gc29tZSBjYXNlcywgc3VjaCBhcyBzcGVjaWZ5aW5nIEVsYXN0aWMgRmFicmljIEFkYXB0ZXJzLFxuICAgKiBuZXR3b3JrIGludGVyZmFjZXMgbXVzdCBiZSB1c2VkIHRvIHNwZWNpZnkgc2VjdXJpdHkgZ3JvdXBzLiAgVGhpc1xuICAgKiBwYXJhbWV0ZXIgdGVsbHMgdGhlIENvbXB1dGUgRW52aXJvbm1lbnQgY29uc3RydWN0IHRoYXQgdGhpcyBpcyB5b3VyXG4gICAqIGludGVudGlvbiwgYW5kIHN0b3BzIGl0IGZyb20gY3JlYXRpbmcgaXRzIG93biBzZWN1cml0eSBncm91cHMuICBUaGlzXG4gICAqIHBhcmFtZXRlciBpcyBtdXR1YWxseSBleGNsdXNpdmUgd2l0aCBzZWN1cml0eUdyb3VwcyBpbiB0aGUgQ29tcHV0ZVxuICAgKiBFbnZpcm9ubWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSB1c2VOZXR3b3JrSW50ZXJmYWNlU2VjdXJpdHlHcm91cHM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIHRoZSBzdHJ1Y3R1cmUgb2YgdGhlIGJhdGNoIGNvbXB1dGUgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21wdXRlUmVzb3VyY2VzIHtcbiAgLyoqXG4gICAqIFRoZSBhbGxvY2F0aW9uIHN0cmF0ZWd5IHRvIHVzZSBmb3IgdGhlIGNvbXB1dGUgcmVzb3VyY2UgaW4gY2FzZSBub3QgZW5vdWdoIGluc3RhbmNlcyBvZiB0aGUgYmVzdFxuICAgKiBmaXR0aW5nIGluc3RhbmNlIHR5cGUgY2FuIGJlIGFsbG9jYXRlZC4gVGhpcyBjb3VsZCBiZSBkdWUgdG8gYXZhaWxhYmlsaXR5IG9mIHRoZSBpbnN0YW5jZSB0eXBlIGluXG4gICAqIHRoZSByZWdpb24gb3IgQW1hem9uIEVDMiBzZXJ2aWNlIGxpbWl0cy4gSWYgdGhpcyBpcyBub3Qgc3BlY2lmaWVkLCB0aGUgZGVmYXVsdCBmb3IgdGhlIEVDMlxuICAgKiBDb21wdXRlUmVzb3VyY2VUeXBlIGlzIEJFU1RfRklULCB3aGljaCB3aWxsIHVzZSBvbmx5IHRoZSBiZXN0IGZpdHRpbmcgaW5zdGFuY2UgdHlwZSwgd2FpdGluZyBmb3JcbiAgICogYWRkaXRpb25hbCBjYXBhY2l0eSBpZiBpdCdzIG5vdCBhdmFpbGFibGUuIFRoaXMgYWxsb2NhdGlvbiBzdHJhdGVneSBrZWVwcyBjb3N0cyBsb3dlciBidXQgY2FuIGxpbWl0XG4gICAqIHNjYWxpbmcuIElmIHlvdSBhcmUgdXNpbmcgU3BvdCBGbGVldHMgd2l0aCBCRVNUX0ZJVCB0aGVuIHRoZSBTcG90IEZsZWV0IElBTSBSb2xlIG11c3QgYmUgc3BlY2lmaWVkLlxuICAgKiBCRVNUX0ZJVF9QUk9HUkVTU0lWRSB3aWxsIHNlbGVjdCBhbiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGUgdGhhdCBpcyBsYXJnZSBlbm91Z2ggdG8gbWVldCB0aGVcbiAgICogcmVxdWlyZW1lbnRzIG9mIHRoZSBqb2JzIGluIHRoZSBxdWV1ZSwgd2l0aCBhIHByZWZlcmVuY2UgZm9yIGFuIGluc3RhbmNlIHR5cGUgd2l0aCBhIGxvd2VyIGNvc3QuXG4gICAqIFRoZSBkZWZhdWx0IHZhbHVlIGZvciB0aGUgU1BPVCBpbnN0YW5jZSB0eXBlIGlzIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVELCB3aGljaCBpcyBvbmx5IGF2YWlsYWJsZSBmb3JcbiAgICogZm9yIHRoaXMgdHlwZSBvZiBjb21wdXRlIHJlc291cmNlcyBhbmQgd2lsbCBzZWxlY3QgYW4gYWRkaXRpb25hbCBpbnN0YW5jZSB0eXBlIHRoYXQgaXMgbGFyZ2UgZW5vdWdoXG4gICAqIHRvIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvciBhbiBpbnN0YW5jZSB0eXBlIHRoYXQgaXNcbiAgICogbGVzcyBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVFxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb2NhdGlvblN0cmF0ZWd5PzogQWxsb2NhdGlvblN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDUyBpbnN0YW5jZSBwcm9maWxlIGFwcGxpZWQgdG8gQW1hem9uIEVDMiBpbnN0YW5jZXMgaW4gYSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnlcbiAgICogdGhlIHNob3J0IG5hbWUgb3IgZnVsbCBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiBhbiBpbnN0YW5jZSBwcm9maWxlLiBGb3IgZXhhbXBsZSwgZWNzSW5zdGFuY2VSb2xlIG9yXG4gICAqIGFybjphd3M6aWFtOjo8YXdzX2FjY291bnRfaWQ+Omluc3RhbmNlLXByb2ZpbGUvZWNzSW5zdGFuY2VSb2xlIC4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBBbWF6b24gRUNTXG4gICAqIEluc3RhbmNlIFJvbGUgaW4gdGhlIEFXUyBCYXRjaCBVc2VyIEd1aWRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9pbnN0YW5jZV9JQU1fcm9sZS5odG1sXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVJvbGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIGxhdW5jaCB0ZW1wbGF0ZSB0byBhc3NvY2lhdGUgd2l0aCB5b3VyIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFJFQURNRSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGN1c3RvbSBsYXVuY2ggdGVtcGxhdGUgd2lsbCBiZSB1c2VkXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2xhdW5jaC10ZW1wbGF0ZXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGU/OiBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlcyBvZiBFQzIgaW5zdGFuY2VzIHRoYXQgbWF5IGJlIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnkgaW5zdGFuY2VcbiAgICogZmFtaWxpZXMgdG8gbGF1bmNoIGFueSBpbnN0YW5jZSB0eXBlIHdpdGhpbiB0aG9zZSBmYW1pbGllcyAoZm9yIGV4YW1wbGUsIGM0IG9yIHAzKSwgb3IgeW91IGNhbiBzcGVjaWZ5XG4gICAqIHNwZWNpZmljIHNpemVzIHdpdGhpbiBhIGZhbWlseSAoc3VjaCBhcyBjNC44eGxhcmdlKS4gWW91IGNhbiBhbHNvIGNob29zZSBvcHRpbWFsIHRvIHBpY2sgaW5zdGFuY2UgdHlwZXNcbiAgICogKGZyb20gdGhlIEMsIE0sIGFuZCBSIGluc3RhbmNlIGZhbWlsaWVzKSBvbiB0aGUgZmx5IHRoYXQgbWF0Y2ggdGhlIGRlbWFuZCBvZiB5b3VyIGpvYiBxdWV1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG9wdGltYWxcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZXM/OiBlYzIuSW5zdGFuY2VUeXBlW107XG5cbiAgLyoqXG4gICAqIFVwIHRvIDUgRUMyIHNlY3VyaXR5IGdyb3VwKHMpIGFzc29jaWF0ZWQgd2l0aCBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIFRoaXMgcGFyYW1ldGVyIGlzIG11dHVhbGx5IGV4Y2x1c2l2ZSB3aXRoIGxhdW5jaFRlbXBsYXRlLnVzZU5ldHdvcmtJbnRlcmZhY2VTZWN1cml0eUdyb3Vwc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIENyZWF0ZSBhIHNpbmdsZSBkZWZhdWx0IHNlY3VyaXR5IGdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXTtcblxuICAvKipcbiAgICogVGhlIFZQQyBuZXR3b3JrIHRoYXQgYWxsIGNvbXB1dGUgcmVzb3VyY2VzIHdpbGwgYmUgY29ubmVjdGVkIHRvLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVGhlIFZQQyBzdWJuZXRzIGludG8gd2hpY2ggdGhlIGNvbXB1dGUgcmVzb3VyY2VzIGFyZSBsYXVuY2hlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBwcml2YXRlIHN1Ym5ldHMgb2YgdGhlIHN1cHBsaWVkIFZQQy5cbiAgICovXG4gIHJlYWRvbmx5IHZwY1N1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBjb21wdXRlIGVudmlyb25tZW50OiBPTl9ERU1BTkQsIFNQT1QsIEZBUkdBVEUsIG9yIEZBUkdBVEVfU1BPVC5cbiAgICpcbiAgICogQGRlZmF1bHQgT05fREVNQU5EXG4gICAqL1xuICByZWFkb25seSB0eXBlPzogQ29tcHV0ZVJlc291cmNlVHlwZTtcblxuICAvKipcbiAgICogVGhpcyBwcm9wZXJ0eSB3aWxsIGJlIGlnbm9yZWQgaWYgeW91IHNldCB0aGUgZW52aXJvbm1lbnQgdHlwZSB0byBPTl9ERU1BTkQuXG4gICAqXG4gICAqIFRoZSBtYXhpbXVtIHBlcmNlbnRhZ2UgdGhhdCBhIFNwb3QgSW5zdGFuY2UgcHJpY2UgY2FuIGJlIHdoZW4gY29tcGFyZWQgd2l0aCB0aGUgT24tRGVtYW5kIHByaWNlIGZvclxuICAgKiB0aGF0IGluc3RhbmNlIHR5cGUgYmVmb3JlIGluc3RhbmNlcyBhcmUgbGF1bmNoZWQuIEZvciBleGFtcGxlLCBpZiB5b3VyIG1heGltdW0gcGVyY2VudGFnZSBpcyAyMCUsXG4gICAqIHRoZW4gdGhlIFNwb3QgcHJpY2UgbXVzdCBiZSBiZWxvdyAyMCUgb2YgdGhlIGN1cnJlbnQgT24tRGVtYW5kIHByaWNlIGZvciB0aGF0IEVDMiBpbnN0YW5jZS4gWW91IGFsd2F5c1xuICAgKiBwYXkgdGhlIGxvd2VzdCAobWFya2V0KSBwcmljZSBhbmQgbmV2ZXIgbW9yZSB0aGFuIHlvdXIgbWF4aW11bSBwZXJjZW50YWdlLiBJZiB5b3UgbGVhdmUgdGhpcyBmaWVsZCBlbXB0eSxcbiAgICogdGhlIGRlZmF1bHQgdmFsdWUgaXMgMTAwJSBvZiB0aGUgT24tRGVtYW5kIHByaWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAxMDBcbiAgICovXG4gIHJlYWRvbmx5IGJpZFBlcmNlbnRhZ2U/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBkZXNpcmVkIG51bWJlciBvZiBFQzIgdkNQVVMgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gZGVzaXJlZCB2Y3B1IHZhbHVlIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2lyZWR2Q3B1cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIEVDMiB2Q1BVcyB0aGF0IGFuIGVudmlyb25tZW50IGNhbiByZWFjaC4gRWFjaCB2Q1BVIGlzIGVxdWl2YWxlbnQgdG9cbiAgICogMSwwMjQgQ1BVIHNoYXJlcy4gWW91IG11c3Qgc3BlY2lmeSBhdCBsZWFzdCBvbmUgdkNQVS5cbiAgICpcbiAgICogQGRlZmF1bHQgMjU2XG4gICAqL1xuICByZWFkb25seSBtYXh2Q3B1cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1pbmltdW0gbnVtYmVyIG9mIEVDMiB2Q1BVcyB0aGF0IGFuIGVudmlyb25tZW50IHNob3VsZCBtYWludGFpbiAoZXZlbiBpZiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudCBzdGF0ZSBpcyBESVNBQkxFRCkuXG4gICAqIEVhY2ggdkNQVSBpcyBlcXVpdmFsZW50IHRvIDEsMDI0IENQVSBzaGFyZXMuIEJ5IGtlZXBpbmcgdGhpcyBzZXQgdG8gMCB5b3Ugd2lsbCBub3QgaGF2ZSBpbnN0YW5jZSB0aW1lIHdhc3RlZCB3aGVuXG4gICAqIHRoZXJlIGlzIG5vIHdvcmsgdG8gYmUgcnVuLiBJZiB5b3Ugc2V0IHRoaXMgYWJvdmUgemVybyB5b3Ugd2lsbCBtYWludGFpbiB0aGF0IG51bWJlciBvZiB2Q1BVcyBhdCBhbGwgdGltZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IDBcbiAgICovXG4gIHJlYWRvbmx5IG1pbnZDcHVzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDMiBwbGFjZW1lbnQgZ3JvdXAgdG8gYXNzb2NpYXRlIHdpdGggeW91ciBjb21wdXRlIHJlc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwbGFjZW1lbnQgZ3JvdXAgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgcGxhY2VtZW50R3JvdXA/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIga2V5IHBhaXIgdGhhdCBpcyB1c2VkIGZvciBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqIElmIG5vIGtleSBpcyBkZWZpbmVkLCB0aGVuIFNTSCBhY2Nlc3MgaXMgbm90IGFsbG93ZWQgdG8gcHJvdmlzaW9uZWQgY29tcHV0ZSByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gU1NIIGFjY2VzcyB3aWxsIGJlIHBvc3NpYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgZWMyS2V5UGFpcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFtYXpvbiBNYWNoaW5lIEltYWdlIChBTUkpIElEIHVzZWQgZm9yIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBpbWFnZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBpbWFnZT86IGVjMi5JTWFjaGluZUltYWdlO1xuXG4gIC8qKlxuICAgKiBUaGlzIHByb3BlcnR5IHdpbGwgYmUgaWdub3JlZCBpZiB5b3Ugc2V0IHRoZSBlbnZpcm9ubWVudCB0eXBlIHRvIE9OX0RFTUFORC5cbiAgICpcbiAgICogVGhlIEFtYXpvbiBSZXNvdXJjZSBOYW1lIChBUk4pIG9mIHRoZSBBbWF6b24gRUMyIFNwb3QgRmxlZXQgSUFNIHJvbGUgYXBwbGllZCB0byBhIFNQT1QgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBBbWF6b24gRUMyIFNwb3QgRmxlZXQgUm9sZSBpbiB0aGUgQVdTIEJhdGNoIFVzZXIgR3VpZGUuXG4gICAqXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL3Nwb3RfZmxlZXRfSUFNX3JvbGUuaHRtbFxuICAgKiBAZGVmYXVsdCAtIG5vIGZsZWV0IHJvbGUgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgc3BvdEZsZWV0Um9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogS2V5LXZhbHVlIHBhaXIgdGFncyB0byBiZSBhcHBsaWVkIHRvIHJlc291cmNlcyB0aGF0IGFyZSBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICogRm9yIEFXUyBCYXRjaCwgdGhlc2UgdGFrZSB0aGUgZm9ybSBvZiBcIlN0cmluZzFcIjogXCJTdHJpbmcyXCIsIHdoZXJlIFN0cmluZzEgaXMgdGhlIHRhZyBrZXkgYW5kXG4gICAqIFN0cmluZzIgaXMgdGhlIHRhZyB2YWx1ZeKAlGZvciBleGFtcGxlLCB7IFwiTmFtZVwiOiBcIkFXUyBCYXRjaCBJbnN0YW5jZSAtIEM0T25EZW1hbmRcIiB9LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHRhZ3Mgd2lsbCBiZSBhc3NpZ25lZCBvbiBjb21wdXRlIHJlc291cmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVSZXNvdXJjZXNUYWdzPzoge1xuICAgIFtrZXk6IHN0cmluZ106IHN0cmluZ1xuICB9O1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGEgbmV3IENvbXB1dGUgRW52aXJvbm1lbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21wdXRlRW52aXJvbm1lbnRQcm9wcyB7XG4gIC8qKlxuICAgKiBBIG5hbWUgZm9yIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBVcCB0byAxMjggbGV0dGVycyAodXBwZXJjYXNlIGFuZCBsb3dlcmNhc2UpLCBudW1iZXJzLCBoeXBoZW5zLCBhbmQgdW5kZXJzY29yZXMgYXJlIGFsbG93ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ2xvdWRGb3JtYXRpb24tZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkZXRhaWxzIG9mIHRoZSByZXF1aXJlZCBjb21wdXRlIHJlc291cmNlcyBmb3IgdGhlIG1hbmFnZWQgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogSWYgc3BlY2lmaWVkLCBhbmQgdGhpcyBpcyBhbiB1bm1hbmFnZWQgY29tcHV0ZSBlbnZpcm9ubWVudCwgd2lsbCB0aHJvdyBhbiBlcnJvci5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwgQVdTIEJhdGNoIG1hbmFnZWQgY29tcHV0ZSBlbnZpcm9ubWVudHMgdXNlIGEgcmVjZW50LCBhcHByb3ZlZCB2ZXJzaW9uIG9mIHRoZVxuICAgKiBBbWF6b24gRUNTLW9wdGltaXplZCBBTUkgZm9yIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENsb3VkRm9ybWF0aW9uIGRlZmF1bHRzXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1wcm9wZXJ0aWVzLWJhdGNoLWNvbXB1dGVlbnZpcm9ubWVudC1jb21wdXRlcmVzb3VyY2VzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVSZXNvdXJjZXM/OiBDb21wdXRlUmVzb3VyY2VzO1xuXG4gIC8qKlxuICAgKiBUaGUgc3RhdGUgb2YgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuIElmIHRoZSBzdGF0ZSBpcyBzZXQgdG8gdHJ1ZSwgdGhlbiB0aGUgY29tcHV0ZVxuICAgKiBlbnZpcm9ubWVudCBhY2NlcHRzIGpvYnMgZnJvbSBhIHF1ZXVlIGFuZCBjYW4gc2NhbGUgb3V0IGF1dG9tYXRpY2FsbHkgYmFzZWQgb24gcXVldWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIHVzZWQgYnkgQmF0Y2ggdG8gbWFrZSBjYWxscyB0byBvdGhlciBBV1Mgc2VydmljZXMgb24geW91ciBiZWhhbGYgZm9yIG1hbmFnaW5nXG4gICAqIHRoZSByZXNvdXJjZXMgdGhhdCB5b3UgdXNlIHdpdGggdGhlIHNlcnZpY2UuIEJ5IGRlZmF1bHQsIHRoaXMgcm9sZSBpcyBjcmVhdGVkIGZvciB5b3UgdXNpbmdcbiAgICogdGhlIEFXUyBtYW5hZ2VkIHNlcnZpY2UgcG9saWN5IGZvciBCYXRjaC5cbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvc2VydmljZV9JQU1fcm9sZS5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUm9sZSB1c2luZyB0aGUgJ3NlcnZpY2Utcm9sZS9BV1NCYXRjaFNlcnZpY2VSb2xlJyBwb2xpY3kuXG4gICAqL1xuICByZWFkb25seSBzZXJ2aWNlUm9sZT86IGlhbS5JUm9sZSxcblxuICAvKipcbiAgICogRGV0ZXJtaW5lcyBpZiBBV1Mgc2hvdWxkIG1hbmFnZSB0aGUgYWxsb2NhdGlvbiBvZiBjb21wdXRlIHJlc291cmNlcyBmb3IgcHJvY2Vzc2luZyBqb2JzLlxuICAgKiBJZiBzZXQgdG8gZmFsc2UsIHRoZW4geW91IGFyZSBpbiBjaGFyZ2Ugb2YgcHJvdmlkaW5nIHRoZSBjb21wdXRlIHJlc291cmNlIGRldGFpbHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IG1hbmFnZWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgb2YgYSBjb21wdXRlIGVudmlyb25tZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElDb21wdXRlRW52aXJvbm1lbnQgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudEFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudE5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBCYXRjaCBDb21wdXRlIEVudmlyb25tZW50LlxuICpcbiAqIERlZmluZXMgYSBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50IHRvIHJ1biBiYXRjaCBqb2JzIG9uLlxuICovXG5leHBvcnQgY2xhc3MgQ29tcHV0ZUVudmlyb25tZW50IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ29tcHV0ZUVudmlyb25tZW50LCBlYzIuSUNvbm5lY3RhYmxlIHtcbiAgLyoqXG4gICAqIEZldGNoZXMgYW4gZXhpc3RpbmcgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudCBieSBpdHMgYW1hem9uIHJlc291cmNlIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZVxuICAgKiBAcGFyYW0gaWRcbiAgICogQHBhcmFtIGNvbXB1dGVFbnZpcm9ubWVudEFyblxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQ29tcHV0ZUVudmlyb25tZW50QXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNvbXB1dGVFbnZpcm9ubWVudEFybjogc3RyaW5nKTogSUNvbXB1dGVFbnZpcm9ubWVudCB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG4gICAgY29uc3QgY29tcHV0ZUVudmlyb25tZW50TmFtZSA9IHN0YWNrLnNwbGl0QXJuKGNvbXB1dGVFbnZpcm9ubWVudEFybiwgQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZSE7XG5cbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDb21wdXRlRW52aXJvbm1lbnQge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudEFybiA9IGNvbXB1dGVFbnZpcm9ubWVudEFybjtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnROYW1lID0gY29tcHV0ZUVudmlyb25tZW50TmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhpcyBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50QXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudE5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogQ29ubmVjdGlvbnMgZm9yIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb25uZWN0aW9uczogZWMyLkNvbm5lY3Rpb25zO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDb21wdXRlRW52aXJvbm1lbnRQcm9wcyA9IHsgZW5hYmxlZDogdHJ1ZSwgbWFuYWdlZDogdHJ1ZSB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmNvbXB1dGVFbnZpcm9ubWVudE5hbWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBpc0ZhcmdhdGUgPSBDb21wdXRlUmVzb3VyY2VUeXBlLkZBUkdBVEUgPT09IHByb3BzLmNvbXB1dGVSZXNvdXJjZXM/LnR5cGVcbiAgICAgIHx8IENvbXB1dGVSZXNvdXJjZVR5cGUuRkFSR0FURV9TUE9UID09PSBwcm9wcy5jb21wdXRlUmVzb3VyY2VzPy50eXBlOztcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcywgaXNGYXJnYXRlKTtcblxuICAgIGNvbnN0IHNwb3RGbGVldFJvbGUgPSB0aGlzLmdldFNwb3RGbGVldFJvbGUocHJvcHMpO1xuICAgIGxldCBjb21wdXRlUmVzb3VyY2VzOiBDZm5Db21wdXRlRW52aXJvbm1lbnQuQ29tcHV0ZVJlc291cmNlc1Byb3BlcnR5IHwgdW5kZWZpbmVkO1xuXG4gICAgY29uc3QgdXNlTGF1bmNoVGVtcGxhdGVOZXR3b3JrSW50ZXJmYWNlID0gcHJvcHMuY29tcHV0ZVJlc291cmNlcz8ubGF1bmNoVGVtcGxhdGU/LnVzZU5ldHdvcmtJbnRlcmZhY2VTZWN1cml0eUdyb3VwcyA/IHRydWUgOiBmYWxzZTtcblxuICAgIHRoaXMuY29ubmVjdGlvbnMgPSB0aGlzLmJ1aWxkQ29ubmVjdGlvbnModXNlTGF1bmNoVGVtcGxhdGVOZXR3b3JrSW50ZXJmYWNlLCBwcm9wcy5jb21wdXRlUmVzb3VyY2VzPy52cGMsIHByb3BzLmNvbXB1dGVSZXNvdXJjZXM/LnNlY3VyaXR5R3JvdXBzKTtcblxuICAgIC8vIE9ubHkgYWxsb3cgY29tcHV0ZSByZXNvdXJjZXMgdG8gYmUgc2V0IHdoZW4gdXNpbmcgTUFOQUdFRCB0eXBlXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgJiYgdGhpcy5pc01hbmFnZWQocHJvcHMpKSB7XG5cbiAgICAgIGNvbXB1dGVSZXNvdXJjZXMgPSB7XG4gICAgICAgIGJpZFBlcmNlbnRhZ2U6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSxcbiAgICAgICAgZGVzaXJlZHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmRlc2lyZWR2Q3B1cyxcbiAgICAgICAgZWMyS2V5UGFpcjogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5lYzJLZXlQYWlyLFxuICAgICAgICBpbWFnZUlkOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmltYWdlICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW1hZ2UuZ2V0SW1hZ2UodGhpcykuaW1hZ2VJZCxcbiAgICAgICAgbGF1bmNoVGVtcGxhdGU6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubGF1bmNoVGVtcGxhdGUsXG4gICAgICAgIG1heHZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1heHZDcHVzIHx8IDI1NixcbiAgICAgICAgcGxhY2VtZW50R3JvdXA6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMucGxhY2VtZW50R3JvdXAsXG4gICAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IHRoaXMuZ2V0U2VjdXJpdHlHcm91cElkcyh1c2VMYXVuY2hUZW1wbGF0ZU5ldHdvcmtJbnRlcmZhY2UpLFxuICAgICAgICBzcG90SWFtRmxlZXRSb2xlOiBzcG90RmxlZXRSb2xlPy5yb2xlQXJuLFxuICAgICAgICBzdWJuZXRzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwYy5zZWxlY3RTdWJuZXRzKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjU3VibmV0cykuc3VibmV0SWRzLFxuICAgICAgICB0YWdzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmNvbXB1dGVSZXNvdXJjZXNUYWdzLFxuICAgICAgICB0eXBlOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnR5cGUgfHwgQ29tcHV0ZVJlc291cmNlVHlwZS5PTl9ERU1BTkQsXG4gICAgICAgIC4uLighaXNGYXJnYXRlID8ge1xuICAgICAgICAgIGFsbG9jYXRpb25TdHJhdGVneTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3lcbiAgICAgICAgICAgIHx8IChcbiAgICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1RcbiAgICAgICAgICAgICAgICA/IEFsbG9jYXRpb25TdHJhdGVneS5TUE9UX0NBUEFDSVRZX09QVElNSVpFRFxuICAgICAgICAgICAgICAgIDogQWxsb2NhdGlvblN0cmF0ZWd5LkJFU1RfRklUXG4gICAgICAgICAgICApLFxuICAgICAgICAgIGluc3RhbmNlUm9sZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVJvbGVcbiAgICAgICAgICAgID8gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVJvbGVcbiAgICAgICAgICAgIDogbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlLVByb2ZpbGUnLCB7XG4gICAgICAgICAgICAgIHJvbGVzOiBbbmV3IGlhbS5Sb2xlKHRoaXMsICdFY3MtSW5zdGFuY2UtUm9sZScsIHtcbiAgICAgICAgICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FtYXpvbkVDMkNvbnRhaW5lclNlcnZpY2Vmb3JFQzJSb2xlJyksXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSkucm9sZU5hbWVdLFxuICAgICAgICAgICAgfSkuYXR0ckFybixcbiAgICAgICAgICBpbnN0YW5jZVR5cGVzOiB0aGlzLmJ1aWxkSW5zdGFuY2VUeXBlcyhwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmluc3RhbmNlVHlwZXMpLFxuICAgICAgICAgIG1pbnZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIHx8IDAsXG4gICAgICAgIH0gOiB7fSksXG4gICAgICB9O1xuICAgIH1cblxuICAgIGNvbnN0IGNvbXB1dGVFbnZpcm9ubWVudCA9IG5ldyBDZm5Db21wdXRlRW52aXJvbm1lbnQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgY29tcHV0ZUVudmlyb25tZW50TmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBjb21wdXRlUmVzb3VyY2VzLFxuICAgICAgc2VydmljZVJvbGU6IHByb3BzLnNlcnZpY2VSb2xlPy5yb2xlQXJuXG4gICAgICAgID8/IG5ldyBpYW0uUm9sZSh0aGlzLCAnUmVzb3VyY2UtU2VydmljZS1JbnN0YW5jZS1Sb2xlJywge1xuICAgICAgICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTQmF0Y2hTZXJ2aWNlUm9sZScpLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2JhdGNoLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgICAgfSkucm9sZUFybixcbiAgICAgIHN0YXRlOiBwcm9wcy5lbmFibGVkID09PSB1bmRlZmluZWQgPyAnRU5BQkxFRCcgOiAocHJvcHMuZW5hYmxlZCA/ICdFTkFCTEVEJyA6ICdESVNBQkxFRCcpLFxuICAgICAgdHlwZTogdGhpcy5pc01hbmFnZWQocHJvcHMpID8gJ01BTkFHRUQnIDogJ1VOTUFOQUdFRCcsXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcyAmJiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwYykge1xuICAgICAgdGhpcy5ub2RlLmFkZERlcGVuZGVuY3kocHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMpO1xuICAgIH1cblxuICAgIHRoaXMuY29tcHV0ZUVudmlyb25tZW50QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShjb21wdXRlRW52aXJvbm1lbnQucmVmLCB7XG4gICAgICBzZXJ2aWNlOiAnYmF0Y2gnLFxuICAgICAgcmVzb3VyY2U6ICdjb21wdXRlLWVudmlyb25tZW50JyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgfSk7XG4gICAgdGhpcy5jb21wdXRlRW52aXJvbm1lbnROYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoY29tcHV0ZUVudmlyb25tZW50LnJlZik7XG4gIH1cblxuICBwcml2YXRlIGlzTWFuYWdlZChwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMpOiBib29sZWFuIHtcbiAgICByZXR1cm4gcHJvcHMubWFuYWdlZCA/PyB0cnVlO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgcHJvcGVydGllcyBwcm92aWRlZCBmb3IgYSBuZXcgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMsIGlzRmFyZ2F0ZTogYm9vbGVhbikge1xuICAgIGlmIChwcm9wcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmlzTWFuYWdlZChwcm9wcykgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0l0IGlzIG5vdCBhbGxvd2VkIHRvIHNldCBjb21wdXRlUmVzb3VyY2VzIG9uIGFuIEFXUyB1bm1hbmFnZWQgY29tcHV0ZSBlbnZpcm9ubWVudCcpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmlzTWFuYWdlZChwcm9wcykgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2NvbXB1dGVSZXNvdXJjZXMgaXMgbWlzc2luZyBidXQgcmVxdWlyZWQgb24gYSBtYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcykge1xuICAgICAgaWYgKGlzRmFyZ2F0ZSkge1xuICAgICAgICAvLyBWQUxJREFURSBGT1IgRkFSR0FURVxuXG4gICAgICAgIC8vIEJpZCBwZXJjZW50YWdlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0JpZCBwZXJjZW50YWdlIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gQWxsb2NhdGlvbiBzdHJhdGVneSBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FsbG9jYXRpb24gc3RyYXRlZ3kgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBEZXNpcmVkIHZDUFVzIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5kZXNpcmVkdkNwdXMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignRGVzaXJlZCB2Q1BVcyBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEltYWdlIElEIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbWFnZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbWFnZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEluc3RhbmNlIHR5cGVzIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbnN0YW5jZVR5cGVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0luc3RhbmNlIHR5cGVzIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gRUMyIGtleSBwYWlyIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5lYzJLZXlQYWlyICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VDMiBrZXkgcGFpciBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIEluc3RhbmNlIHJvbGUgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmluc3RhbmNlUm9sZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbnN0YW5jZSByb2xlIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gTGF1bmNoIHRlbXBsYXRlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5sYXVuY2hUZW1wbGF0ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdMYXVuY2ggdGVtcGxhdGUgbXVzdCBub3QgYmUgc2V0IGZvciBGYXJnYXRlIGNvbXB1dGUgZW52aXJvbm1lbnRzJyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBNaW4gdkNQVXMgY2Fubm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBldm52aXJvbm1lbnRzXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ01pbiB2Q1BVcyBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFBsYWNlbWVudCBncm91cCBjYW5ub3QgYmUgc2V0IGZvciBGYXJnYXRlIGV2bnZpcm9ubWVudHNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMucGxhY2VtZW50R3JvdXAgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGxhY2VtZW50IGdyb3VwIG11c3Qgbm90IGJlIHNldCBmb3IgRmFyZ2F0ZSBjb21wdXRlIGVudmlyb25tZW50cycpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gU3BvdCBmbGVldCByb2xlIGNhbm5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgZXZudmlyb25tZW50c1xuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5zcG90RmxlZXRSb2xlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Nwb3QgZmxlZXQgcm9sZSBtdXN0IG5vdCBiZSBzZXQgZm9yIEZhcmdhdGUgY29tcHV0ZSBlbnZpcm9ubWVudHMnKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gVkFMSURBVEUgRk9SIE9OX0RFTUFORCBBTkQgU1BPVFxuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cykge1xuICAgICAgICAgIC8vIG1pbnZDcHVzIGNhbm5vdCBiZSBsZXNzIHRoYW4gMFxuICAgICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIDwgMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdNaW5pbXVtIHZDcHVzIGZvciBhIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgY2Fubm90IGJlIGxlc3MgdGhhbiAwJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgLy8gbWludkNwdXMgY2Fubm90IGV4Y2VlZCBtYXggdkNwdXNcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5tYXh2Q3B1cyAmJlxuICAgICAgICAgICAgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5taW52Q3B1cyA+IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWF4dkNwdXMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWluaW11bSB2Q3B1cyBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIHRoZSBtYXhpbXVtIHZDcHVzJyk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG5cbiAgICAgICAgLy8gQ2hlY2sgaWYgYm90aCBsYXVuY2hUZW1wbGF0ZUlkIGFuZCBsYXVuY2hUZW1wbGF0ZU5hbWUgYXJlIHByb3ZpZGVkXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlICYmXG4gICAgICAgICAgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubGF1bmNoVGVtcGxhdGUubGF1bmNoVGVtcGxhdGVJZCAmJiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlTmFtZSkpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBtdXN0IHNwZWNpZnkgZWl0aGVyIHRoZSBsYXVuY2ggdGVtcGxhdGUgSUQgb3IgbGF1bmNoIHRlbXBsYXRlIG5hbWUgaW4gdGhlIHJlcXVlc3QsIGJ1dCBub3QgYm90aC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIENoZWNrIGlmIGJvdGggbGF1bmNoVGVtcGxhdGVJZCBhbmQgbGF1bmNoVGVtcGxhdGVOYW1lIGFyZSBtaXNzaW5nXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlICYmXG4gICAgICAgICAgKCFwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlLmxhdW5jaFRlbXBsYXRlSWQgJiYgIXByb3BzLmNvbXB1dGVSZXNvdXJjZXMubGF1bmNoVGVtcGxhdGUubGF1bmNoVGVtcGxhdGVOYW1lKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG11c3Qgc3BlY2lmeSBlaXRoZXIgdGhlIGxhdW5jaCB0ZW1wbGF0ZSBJRCBvciBsYXVuY2ggdGVtcGxhdGUgbmFtZSBpbiB0aGUgcmVxdWVzdC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIHVzZU5ldHdvcmtJbnRlcmZhY2VTZWN1cml0eUdyb3VwcyBjYW5ub3QgaGF2ZSBzZWN1cml0eUdyb3VwcyBkZWZpbmVkXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlPy51c2VOZXR3b3JrSW50ZXJmYWNlU2VjdXJpdHlHcm91cHMgJiZcbiAgICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc2VjdXJpdHlHcm91cHMgKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdzZWN1cml0eUdyb3VwcyBjYW5ub3QgYmUgc3BlY2lmaWVkIGlmIGxhdW5jaFRlbXBsYXRlIHVzZU5ldHdvcmtJbnRlcmZhY2VTZWN1cml0eUdyb3VwcyBpcyBhY3RpdmUnKTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFNldHRpbmcgYSBiaWQgcGVyY2VudGFnZSBpcyBvbmx5IGFsbG93ZWQgb24gU1BPVCByZXNvdXJjZXMgK1xuICAgICAgICAvLyBDYW5ub3QgdXNlIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEIHdoZW4gdXNpbmcgT05fREVNQU5EXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnR5cGUgPT09IENvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5EKSB7XG4gICAgICAgICAgLy8gVkFMSURBVEUgRk9SIE9OX0RFTUFORFxuXG4gICAgICAgICAgLy8gQmlkIHBlcmNlbnRhZ2UgaXMgbm90IGFsbG93ZWRcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignU2V0dGluZyB0aGUgYmlkIHBlcmNlbnRhZ2UgaXMgb25seSBhbGxvd2VkIGZvciBTUE9UIHR5cGUgcmVzb3VyY2VzIG9uIGEgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudCcpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIC8vIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEIGFsbG9jYXRpb24gaXMgbm90IGFsbG93ZWRcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3kgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5hbGxvY2F0aW9uU3RyYXRlZ3kgPT09IEFsbG9jYXRpb25TdHJhdGVneS5TUE9UX0NBUEFDSVRZX09QVElNSVpFRCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBvbmx5IGFsbG93ZWQgaWYgdGhlIGVudmlyb25tZW50IGlzIGEgU1BPVCB0eXBlIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1QpIHtcbiAgICAgICAgICAvLyBWQUxJREFURSBGT1IgU1BPVFxuXG4gICAgICAgICAgLy8gQmlkIHBlcmNlbnRhZ2UgbXVzdCBiZSBmcm9tIDAgLSAxMDBcbiAgICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQgJiZcbiAgICAgICAgICAgIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmJpZFBlcmNlbnRhZ2UgPCAwIHx8IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSA+IDEwMCkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQmlkIHBlcmNlbnRhZ2UgY2FuIG9ubHkgYmUgYSB2YWx1ZSBiZXR3ZWVuIDAgYW5kIDEwMCcpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYnVpbGRJbnN0YW5jZVR5cGVzKGluc3RhbmNlVHlwZXM/OiBlYzIuSW5zdGFuY2VUeXBlW10pOiBzdHJpbmdbXSB7XG4gICAgaWYgKGluc3RhbmNlVHlwZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgJ29wdGltYWwnLFxuICAgICAgXTtcbiAgICB9XG5cbiAgICByZXR1cm4gaW5zdGFuY2VUeXBlcy5tYXAoKHR5cGU6IGVjMi5JbnN0YW5jZVR5cGUpID0+IHR5cGUudG9TdHJpbmcoKSk7XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkQ29ubmVjdGlvbnModXNlTGF1bmNoVGVtcGxhdGVOZXR3b3JrSW50ZXJmYWNlOiBib29sZWFuLCB2cGM/OiBlYzIuSVZwYywgc2VjdXJpdHlHcm91cHM/OmVjMi5JU2VjdXJpdHlHcm91cFtdKTogZWMyLkNvbm5lY3Rpb25zIHtcblxuICAgIGlmICh2cGMgPT09IHVuZGVmaW5lZCB8fCB1c2VMYXVuY2hUZW1wbGF0ZU5ldHdvcmtJbnRlcmZhY2UgKSB7XG4gICAgICByZXR1cm4gbmV3IGVjMi5Db25uZWN0aW9ucyh7fSk7XG4gICAgfVxuXG4gICAgaWYgKHNlY3VyaXR5R3JvdXBzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBuZXcgZWMyLkNvbm5lY3Rpb25zKHtcbiAgICAgICAgc2VjdXJpdHlHcm91cHM6IFtcbiAgICAgICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAodGhpcywgJ1Jlc291cmNlLVNlY3VyaXR5LUdyb3VwJywgeyB2cGMgfSksXG4gICAgICAgIF0sXG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBlYzIuQ29ubmVjdGlvbnMoeyBzZWN1cml0eUdyb3VwcyB9KTtcbiAgfTtcblxuICBwcml2YXRlIGdldFNlY3VyaXR5R3JvdXBJZHModXNlTGF1bmNoVGVtcGxhdGVJbnRlcmZhY2U6IGJvb2xlYW4pOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHRoaXMuY29ubmVjdGlvbnMgPT09IHVuZGVmaW5lZCB8fFxuICAgICAgdXNlTGF1bmNoVGVtcGxhdGVJbnRlcmZhY2UgKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLmNvbm5lY3Rpb25zLnNlY3VyaXR5R3JvdXBzLm1hcCgoZ3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cCkgPT4gZ3JvdXAuc2VjdXJpdHlHcm91cElkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgYW4gQVdTIElBTSByb2xlIGZvciBwcm92aXNpb25pbmcgc3BvdGZsZWV0IHJlc291cmNlc1xuICAgKiBpZiB0aGUgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBzZXQgdG8gQkVTVF9GSVQgb3Igbm90IGRlZmluZWQuXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wcyAtIHRoZSBjb21wdXRlIGVudmlyb25tZW50IGNvbnN0cnVjdCBwcm9wZXJ0aWVzXG4gICAqL1xuICBwcml2YXRlIGdldFNwb3RGbGVldFJvbGUocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzKTogaWFtLklSb2xlIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcz8uYWxsb2NhdGlvblN0cmF0ZWd5ICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICE9PSBBbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMpIHtcbiAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnNwb3RGbGVldFJvbGUpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc3BvdEZsZWV0Um9sZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1QpIHtcbiAgICAgICAgcmV0dXJuIGlhbS5Sb2xlLmZyb21Sb2xlQXJuKHRoaXMsICdSZXNvdXJjZS1TcG90RmxlZXQtUm9sZScsXG4gICAgICAgICAgYGFybjoke3RoaXMuc3RhY2sucGFydGl0aW9ufTppYW06OiR7dGhpcy5lbnYuYWNjb3VudH06cm9sZS9hd3Mtc2VydmljZS1yb2xlL3Nwb3RmbGVldC5hbWF6b25hd3MuY29tL0FXU1NlcnZpY2VSb2xlRm9yRUMyU3BvdEZsZWV0YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxufVxuIl19