from impedance.validation import calc_mu, eval_linKK, residuals_linKK
from impedance.validation import get_tc_distribution, linKK, fit_linKK
from impedance import preprocessing
import numpy as np
import pytest


def test_eval_linKK():

    Rs = np.array([1, 1, 2, 0.5, 0.1])
    ts = np.array([.1, .2])
    f = np.array([.01, 1000])
    w = 2 * np.pi * f

    Z = Rs[0] + (Rs[1]/(1 + ts[0]*1j*w)) + (Rs[2]/(1 + ts[1]*1j*w)) + \
        1j*w*Rs[-1] + Rs[-2] / (1j*w)

    assert (eval_linKK(Rs, ts, f) == Z).all()

    Z_data = Z + np.array([1 + 1j, 1 + 1j])

    assert np.isclose(residuals_linKK(Rs, ts, Z_data, f),
                      (Z_data - Z).real/np.abs(Z_data)).all()

    assert np.isclose(residuals_linKK(Rs, ts, Z_data, f, residuals='imag'),
                      (Z_data - Z).imag/np.abs(Z_data)).all()

    diff_real = (Z_data - Z).real/np.abs(Z_data)
    diff_imag = (Z_data - Z).imag/np.abs(Z_data)
    assert np.isclose(residuals_linKK(Rs, ts, Z_data, f, residuals='both'),
                      [diff_real[0], diff_imag[0], diff_real[1],
                      diff_imag[1]]).all()


def test_calc_mu():
    Rs = [1, 2, 3, -3, -2, -1]
    assert calc_mu(Rs) == 0

    Rs = [-1, 2, 4, -3, 4, -1]
    assert calc_mu(Rs) == 0.5


def test_get_tc_distribution():
    M = 3
    f = np.array([1000, 100, 10, 1, .1])/(2 * np.pi)

    ts_true = np.array([0.001, 0.1, 10])

    assert (ts_true == get_tc_distribution(f, M)).all()


def test_linKK():
    # "True" values listed below are from Schonleber's Lin_KK software using
    # equivalent analysis options.
    fit_true = \
        np.array([1504.77272562, 1495.0921936, 1490.00925513, 1491.10559299,
                  1498.58811243, 1510.99839587, 1525.38888928, 1538.20266473,
                  1546.8146252, 1551.30214876, 1556.06735902, 1571.02993844,
                  1611.71804328, 1697.41876182, 1847.27253729, 2074.90120624,
                  2382.53062569, 2757.22269183, 3173.69682122, 3604.841718,
                  4032.7702192, 4450.8168238, 4855.22368647, 5234.98555099,
                  5569.5736037, 5836.53900962, 6021.90189107, 6125.58944854,
                  6160.67958136, 6149.14874076, 6115.96687044, 6082.88575023,
                  6064.08068366, 6064.93277516, 6083.22455438, 6111.60127879,
                  6140.78792197, 6162.82998495, 6173.20167035, 6171.33427209,
                  6160.03228389, 6144.2248715, 6129.26892121, 6119.3121128,
                  6116.34899865, 6120.09064219, 6128.37712222, 6137.9720589,
                  6145.61179356, 6148.94276639, 6146.99955138, 6140.22155443,
                  6132.74662155]) + \
        np.array([2.21601785e+01, -8.08838610e+00, -4.34187137e+01,
                  -8.26481327e+01, -1.24065234e+02, -1.66329887e+02,
                  -2.09588207e+02, -2.56542204e+02, -3.12958965e+02,
                  -3.87301923e+02, -4.89494687e+02, -6.28743075e+02,
                  -8.10378235e+02, -1.03243330e+03, -1.28325249e+03,
                  -1.54124807e+03, -1.77798369e+03, -1.96571203e+03,
                  -2.08731780e+03, -2.14154401e+03, -2.13776738e+03,
                  -2.08431712e+03, -1.98106122e+03, -1.82274574e+03,
                  -1.60947048e+03, -1.35439985e+03, -1.08242516e+03,
                  -8.22442371e+02, -5.99359860e+02, -4.28680765e+02,
                  -3.14010144e+02, -2.47955686e+02, -2.16112198e+02,
                  -2.02047736e+02, -1.91387718e+02, -1.74535420e+02,
                  -1.47914754e+02, -1.13376514e+02, -7.61122378e+01,
                  -4.22212394e+01, -1.67327105e+01, -2.24573095e+00,
                  1.62351816e+00, -2.23057042e+00, -9.59473214e+00,
                  -1.62154318e+01, -1.88331124e+01, -1.58905695e+01,
                  -7.73327665e+00, 3.80557278e+00, 1.61797537e+01,
                  2.69105999e+01, 3.28441808e+01]) * 1j

    fit_true_cap = \
        np.array([1504.77272562, 1495.0921936, 1490.00925513, 1491.10559299,
                  1498.58811243, 1510.99839587, 1525.38888928, 1538.20266473,
                  1546.8146252, 1551.30214876, 1556.06735902, 1571.02993844,
                  1611.71804328, 1697.41876182, 1847.27253729, 2074.90120624,
                  2382.53062569, 2757.22269183, 3173.69682122, 3604.841718,
                  4032.7702192, 4450.8168238, 4855.22368647, 5234.98555099,
                  5569.5736037, 5836.53900962, 6021.90189107, 6125.58944854,
                  6160.67958136, 6149.14874076, 6115.96687044, 6082.88575023,
                  6064.08068366, 6064.93277516, 6083.22455438, 6111.60127879,
                  6140.78792197, 6162.82998495, 6173.20167035, 6171.33427209,
                  6160.03228389, 6144.2248715, 6129.26892121, 6119.3121128,
                  6116.34899865, 6120.09064219, 6128.37712222, 6137.9720589,
                  6145.61179356, 6148.94276639, 6146.99955138, 6140.22155443,
                  6132.74662155]) + \
        np.array([2.21609593e+01, -8.08783856e+00, -4.34183703e+01,
                  -8.26479751e+01, -1.24065253e+02, -1.66330085e+02,
                  -2.09588594e+02, -2.56542801e+02, -3.12959803e+02,
                  -3.87303047e+02, -4.89496157e+02, -6.28744970e+02,
                  -8.10380655e+02, -1.03243637e+03, -1.28325639e+03,
                  -1.54125299e+03, -1.77798990e+03, -1.96571986e+03,
                  -2.08732766e+03, -2.14155642e+03, -2.13778302e+03,
                  -2.08433681e+03, -1.98108601e+03, -1.82277695e+03,
                  -1.60950978e+03, -1.35444932e+03, -1.08248743e+03,
                  -8.22520773e+02, -5.99458563e+02, -4.28805025e+02,
                  -3.14166579e+02, -2.48152626e+02, -2.16360131e+02,
                  -2.02359865e+02, -1.91780666e+02, -1.75030113e+02,
                  -1.48537535e+02, -1.14160550e+02, -7.70992802e+01,
                  -4.34638524e+01, -1.82970675e+01, -4.21514039e+00,
                  -8.55820774e-01, -5.35187256e+00, -1.35242179e+01,
                  -2.11623597e+01, -2.50609297e+01, -2.37309367e+01,
                  -1.76037019e+01, -8.62054511e+00,  5.36182534e-01,
                  7.21647174e+00, 9.37882405e+00]) * 1j

    f, Z = preprocessing.readZPlot('../impedance.py/data/Circuit3_EIS_1.z')

    resids_true_re = (Z - fit_true).real / np.abs(Z)
    resids_true_im = (Z - fit_true).imag / np.abs(Z)

    M, mu, Z_fit, resids_re, resids_im = linKK(f, Z, c=.69)

    assert np.isclose(fit_true, Z_fit).all()
    assert np.isclose(resids_true_re, resids_re).all()
    assert np.isclose(resids_true_im, resids_im).all()
    assert M == 10
    assert np.isclose(mu, 0.6843821117885378)

    # Test linKK with added capacitance
    resids_true_re = (Z - fit_true_cap).real / np.abs(Z)
    resids_true_im = (Z - fit_true_cap).imag / np.abs(Z)

    M, mu, Z_fit, resids_re, resids_im = linKK(f, Z, c=.69, add_cap=True)

    assert np.isclose(fit_true_cap, Z_fit).all()
    assert np.isclose(resids_true_re, resids_re).all()
    assert np.isclose(resids_true_im, resids_im).all()
    assert M == 10
    assert np.isclose(mu, 0.6843821117885381)

    # Specifing M should give same results as above
    M, mu, Z_fit, resids_re, resids_im = \
        linKK(f, Z, c=None, max_M=10, add_cap=True)

    assert np.isclose(fit_true_cap, Z_fit).all()
    assert np.isclose(resids_true_re, resids_re).all()
    assert np.isclose(resids_true_im, resids_im).all()
    assert M == 10
    assert np.isclose(mu, 0.6843821117885381)


def test_fit_linKK():
    # "True" values listed below are from Schonleber's Lin_KK software using
    # equivalent analysis options.
    fit_true_im = \
        np.array([1892.16670857, 1735.17866073, 1595.34869919, 1483.62092725,
                  1404.08871138, 1356.09108761, 1337.54493602, 1347.5270314,
                  1387.51185068, 1461.26044599, 1573.22397804, 1725.2980715,
                  1912.70463125, 2121.67049406, 2332.09328647, 2524.93169072,
                  2689.34607069, 2825.0091077, 2939.93788724, 3047.13537702,
                  3162.32135172, 3302.86711448, 3486.69461487, 3729.45394636,
                  4038.84161501, 4407.36385786, 4808.72682252, 5203.80832846,
                  5555.08167692, 5839.63432187, 6052.37916535, 6200.66510126,
                  6296.6956546, 6352.15110256, 6375.78018599, 6373.11446064,
                  6347.45708565, 6301.65201971, 6240.05239619, 6169.58830428,
                  6098.85243941, 6035.55826473, 5984.31870647, 5946.26167969,
                  5920.17874695, 5903.99645739, 5895.7929968, 5894.24236205,
                  5898.63972971, 5908.64317923, 5923.81098604, 5943.08676257,
                  5959.35300613]) + \
        np.array([-70.4776521, 23.83024186, 50.84494114, 25.88677919,
                  -35.74962294, -122.45215854, -227.18178042, -346.26102872,
                  -477.20636325, -616.29028254, -756.32307243, -885.6167092,
                  -989.70684567, -1056.65273354, -1083.59255493,
                  -1079.56960769, -1062.07484708, -1050.35335609,
                  -1060.22332082, -1102.06593845, -1180.63979118,
                  -1294.82054257, -1436.27259811, -1587.4498159,
                  -1721.03366501, -1804.20226876, -1809.32501067,
                  -1726.38772704, -1567.6426133, -1360.28285481,
                  -1133.62730069, -909.94086692, -701.95825097, -514.79535919,
                  -349.19837452, -204.47489479, -80.59143375, 20.81530157,
                  96.87089028, 145.37359197, 167.07538881, 166.46139247,
                  150.31388332, 125.3565128, 96.65765081, 67.28992165,
                  38.78343211, 11.81232837, -13.17831776, -35.51127909,
                  -54.0704255, -67.48447146, -73.57222491]) * 1j

    fit_true_comp = \
        np.array([1636.99559739, 1554.18810328, 1481.23911417, 1424.3900838,
                  1386.42808216, 1367.90765932, 1369.06145295, 1391.315147,
                  1438.01800827, 1514.22980492, 1625.24547239, 1773.58370231,
                  1955.13366155, 2157.00804753, 2360.1665209, 2546.56349679,
                  2706.05172614, 2838.65071263, 2952.5228364, 3060.81514474,
                  3179.54588511, 3326.61091254, 3520.64327344, 3777.99027267,
                  4106.63312453, 4498.45313082, 4925.36485549, 5345.65471926,
                  5719.32098597, 6021.92053781, 6247.97683411, 6405.23218601,
                  6506.56600035, 6564.25933403, 6587.42707198, 6581.72881809,
                  6550.47463369, 6496.61170472, 6424.9457516, 6343.33583694,
                  6261.61536605, 6188.64620036, 6129.75023804, 6086.26557292,
                  6056.87606879, 6039.31722373, 6031.55856349, 6032.31830118,
                  6041.07049939, 6057.67650742, 6081.71810383, 6111.74726757,
                  6136.91206761]) + \
        np.array([17.18695412, 35.89845324, 19.93303201, -24.04618822,
                  -89.31690932, -171.05398322, -266.73687047, -375.31778525,
                  -495.7085041, -624.93230106, -756.32867662, -878.75805143,
                  -978.36531405, -1043.72669304, -1072.17384645,
                  -1072.52206761, -1061.75528827, -1058.59331398,
                  -1078.4942003, -1131.66997952, -1222.80054094,
                  -1350.59588738, -1506.2444795, -1671.1947291,
                  -1816.50610085, -1907.3683077, -1914.47475074,
                  -1827.24555106, -1658.77755869, -1438.04893513,
                  -1196.28141241, -957.17525874, -734.28460084, -533.0719019,
                  -354.39918838, -197.69465823, -63.23427094, 46.76632237,
                  128.68462199, 179.70585913, 200.45381383, 195.88528088,
                  173.62890483, 141.26465518, 104.48271295, 66.69820353,
                  29.59329382, -6.05425717, -39.59351444, -69.9122427,
                  -95.18725792, -113.19938882, -120.94168923]) * 1j

    f, Z = preprocessing.readZPlot('../impedance.py/data/Circuit3_EIS_1.z')
    taus = get_tc_distribution(f, 5)

    # improper data types in fitting raise a TypeError
    with pytest.raises(ValueError):
        fit_linKK(f, taus, 5, Z, fit_type='abcd')

    p_values, _ = fit_linKK(f, taus, 5, Z, fit_type='imag')
    assert np.isclose(fit_true_im, eval_linKK(p_values, taus, f)).all()

    p_values, _ = fit_linKK(f, taus, 5, Z, fit_type='complex')
    assert np.isclose(fit_true_comp, eval_linKK(p_values, taus, f)).all()
