"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.convert = exports.generate = exports.X509CertificateConverter = exports.X509CertificateGenerator = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const crypto_1 = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
const kms_1 = require("../lib/kms");
const secrets_manager_1 = require("../lib/secrets-manager");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class X509Common extends custom_resource_1.DynamoBackedCustomResource {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.secretsManagerClient = secretsManagerClient;
    }
    async doDelete(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const resources = await resourceTable.query(physicalId);
        for (const [key, resource] of Object.entries(resources)) {
            console.log(`Deleting resource for '${key}'`);
            const arn = resource.ARN;
            try {
                const secret = secrets_manager_1.Secret.fromArn(arn, this.secretsManagerClient);
                await secret.delete();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We legit do not have the required permission to delete the secret (very unlikely)
                //  b) The Secret has already been deleted (much more likely; so we continue)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Secret ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async secretsPermissionsCheck(tags) {
        if (!this.debugMode) {
            return;
        }
        const secretName = crypto_1.randomBytes(16).toString('hex');
        const secret = await secrets_manager_1.Secret.create({
            name: secretName,
            client: this.secretsManagerClient,
            description: 'Permissions check',
            data: 'Test data',
            tags,
        });
        if (!secret) {
            throw new Error('Failed to create secret during permission test.');
        }
        await secret.putValue('Test data 2');
        // await secret.getValue(); // We don't give this permissions to the Lambda
        await secret.delete(true);
    }
    /**
     * Helper for creating a Secret and storing its ARN in the database.
     * CustomResources must be idempotent, and it is theoretically possible that Cfn
     * may invoke our lambda more than once for the same operation. If this happens,
     * then we may already have a Secret ARN stored in the database; if we do,
     * then we must reuse that ARN to avoid resource leakage.
     *
     * It's theoretically possible that this function may race with another invocation
     * of itself, but that is so unlikely. Handling it would complicate this function
     * substantially, so we do not implement for that case.
     * @param args
     */
    async createAndStoreSecret(args) {
        let secretArn;
        const existingItem = await args.database.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: args.purpose,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            secretArn = existingItem.ARN;
            const secret = secrets_manager_1.Secret.fromArn(secretArn, this.secretsManagerClient);
            await secret.putValue(args.data);
        }
        else {
            const secret = await secrets_manager_1.Secret.create({
                name: args.name,
                client: this.secretsManagerClient,
                description: args.description,
                data: args.data,
                tags: args.tags,
                encryptionKey: args.encryptionKey,
            });
            if (!secret || !secret.arn) {
                throw Error('Could not create Secret');
            }
            secretArn = secret.arn;
            await args.database.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: args.purpose,
                attributes: {
                    ARN: secretArn,
                },
                allow_overwrite: false,
            });
        }
        return secretArn;
    }
}
class X509CertificateGenerator extends X509Common {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient, secretsManagerClient);
    }
    validateInput(data) {
        return types_1.implementsIX509CertificateGenerate(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const subject = new x509_certs_1.DistinguishedName(resourceProperties.DistinguishedName);
        const passphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Passphrase, this.secretsManagerClient).getValue();
        let certExpiry = resourceProperties.CertificateValidFor ? Number(resourceProperties.CertificateValidFor) : 1095;
        let signingCert;
        if (resourceProperties.SigningCertificate) {
            const signCert = resourceProperties.SigningCertificate;
            const cert = await secrets_manager_1.Secret.fromArn(signCert.Cert, this.secretsManagerClient).getValue();
            const key = await secrets_manager_1.Secret.fromArn(signCert.Key, this.secretsManagerClient).getValue();
            const pass = await secrets_manager_1.Secret.fromArn(signCert.Passphrase, this.secretsManagerClient).getValue();
            const certChain = signCert.CertChain.length > 0
                ? await secrets_manager_1.Secret.fromArn(signCert.CertChain, this.secretsManagerClient).getValue()
                : '';
            signingCert = new x509_certs_1.Certificate(cert, key, pass, certChain);
        }
        const newCert = await x509_certs_1.Certificate.fromGenerated(subject, passphrase, certExpiry, signingCert);
        const now = new Date(Date.now());
        // timeSuffix = "<year>-<month>-<day>-<time since epoch>" -- to disambiguate secrets
        // in case we do an update on the same day as a create (both old & new exist at the same time)
        const timeSuffix = `${now.getFullYear()}-${now.getMonth()}-${now.getDate()}-${now.getTime()}`;
        const kmsKey = resourceProperties.Secret.EncryptionKey ? kms_1.Key.fromArn(resourceProperties.Secret.EncryptionKey) : undefined;
        const returnArns = {};
        const certComponents = [
            {
                key: 'Cert',
                purpose: 'Certificate',
                data: newCert.cert,
            },
            {
                key: 'Key',
                purpose: 'Private Key',
                data: newCert.key,
            },
            {
                key: 'CertChain',
                purpose: 'Certificate Chain',
                data: newCert.certChain,
            },
        ];
        for (const component of certComponents) {
            if (component.data) {
                const data = component.data;
                const purpose = component.purpose;
                const name = secrets_manager_1.sanitizeSecretName(`${resourceProperties.Secret.NamePrefix}-X.509-${purpose}-${timeSuffix}`);
                const arn = await this.createAndStoreSecret({
                    database: resourceTable,
                    name,
                    physicalId,
                    purpose,
                    data,
                    description: `X.509 ${component.purpose} for ${resourceProperties.Secret.Description}`,
                    tags: resourceProperties.Secret.Tags,
                    encryptionKey: kmsKey,
                });
                returnArns[component.key] = arn;
            }
            else {
                // Case for CertChain being empty. We cannot just skip it due to constraints put on us by CDK's CustomResource
                returnArns[component.key] = '';
            }
        }
        return returnArns;
    }
}
exports.X509CertificateGenerator = X509CertificateGenerator;
class X509CertificateConverter extends X509Common {
    constructor(dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient, secretsManagerClient);
    }
    validateInput(data) {
        return types_1.implementsIX509CertificateEncodePkcs12(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
            this.secretsPermissionsCheck(resourceProperties.Secret.Tags),
        ]);
        const cert = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Cert, this.secretsManagerClient).getValue();
        const certChain = resourceProperties.Certificate.CertChain.length > 0
            ? await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.CertChain, this.secretsManagerClient).getValue()
            : '';
        const key = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Key, this.secretsManagerClient).getValue();
        const sourcePassphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Certificate.Passphrase, this.secretsManagerClient).getValue();
        const sourceCert = new x509_certs_1.Certificate(cert, key, sourcePassphrase, certChain);
        const passphrase = await secrets_manager_1.Secret.fromArn(resourceProperties.Passphrase, this.secretsManagerClient).getValue();
        const newPkcs12 = await sourceCert.toPkcs12(passphrase);
        const now = new Date(Date.now());
        // timeSuffix = "<year>-<month>-<day>-<time since epoch>" -- to disambiguate secrets
        // in case we do an update on the same day as a create (both old & new exist at the same time)
        const timeSuffix = `${now.getFullYear()}-${now.getMonth()}-${now.getDate()}-${now.getTime()}`;
        const secretProps = resourceProperties.Secret;
        const kmsKey = secretProps.EncryptionKey
            ? kms_1.Key.fromArn(secretProps.EncryptionKey)
            : undefined;
        const returnArns = {};
        const pkcs12Props = {
            data: newPkcs12,
            database: resourceTable,
            description: `X.509 PKCS #12 Certificate for ${secretProps.Description}`,
            encryptionKey: kmsKey,
            name: secrets_manager_1.sanitizeSecretName(`${secretProps.NamePrefix}-X.509-CertificatePKCS12-${timeSuffix}`),
            physicalId,
            purpose: 'CertificatePKCS12',
            tags: secretProps.Tags,
        };
        returnArns.Cert = await this.createAndStoreSecret(pkcs12Props);
        return returnArns;
    }
}
exports.X509CertificateConverter = X509CertificateConverter;
/**
 * The handler used to generate an X.509 certificate and then store it in SecretsManager
 */
async function generate(event, context) {
    const handler = new X509CertificateGenerator(new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.generate = generate;
/**
 * The handler used to convert an X.509 certificate to PKCS #12 and store that in SecretsManager
 */
async function convert(event, context) {
    const handler = new X509CertificateConverter(new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.convert = convert;
//# sourceMappingURL=data:application/json;base64,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