"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.importCert = exports.AcmCertificateImporter = void 0;
/* eslint-disable no-console */
const crypto = require("crypto");
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const backoff_generator_1 = require("../lib/backoff-generator");
const custom_resource_1 = require("../lib/custom-resource");
const x509_certs_1 = require("../lib/x509-certs");
const types_1 = require("./types");
const ACM_VERSION = '2015-12-08';
const DYNAMODB_VERSION = '2012-08-10';
const SECRETS_MANAGER_VERSION = '2017-10-17';
class AcmCertificateImporter extends custom_resource_1.DynamoBackedCustomResource {
    constructor(acmClient, dynamoDbClient, secretsManagerClient) {
        super(dynamoDbClient);
        this.acmClient = acmClient;
        this.secretsManagerClient = secretsManagerClient;
    }
    /* istanbul ignore next */
    validateInput(data) {
        return types_1.implementsIAcmImportCertProps(data);
    }
    async doCreate(physicalId, resourceProperties) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const cert = await this.getSecretString(resourceProperties.X509CertificatePem.Cert);
        const certChainArn = resourceProperties.X509CertificatePem.CertChain;
        const certChain = certChainArn.length > 0 ? await this.getSecretString(certChainArn) : undefined;
        const key = await this.getSecretString(resourceProperties.X509CertificatePem.Key);
        const passphrase = await this.getSecretString(resourceProperties.X509CertificatePem.Passphrase);
        const decryptedKey = await x509_certs_1.Certificate.decryptKey(key, passphrase);
        const tags = resourceProperties.Tags;
        const certificateArn = await this.importAndStoreCertificate({
            resourceTable,
            key: decryptedKey,
            cert,
            certChain,
            physicalId,
            tags,
        });
        return { CertificateArn: certificateArn };
    }
    async doDelete(physicalId) {
        const resourceTable = await this.getResourceTable();
        await Promise.all([
            this.databasePermissionsCheck(resourceTable),
        ]);
        const resources = await resourceTable.query(physicalId);
        const maxAttempts = 10;
        for (const [key, resource] of Object.entries(resources)) {
            const arn = resource.ARN;
            let inUseByResources = [];
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base: 1000,
                jitterDivisor: 4,
                maxAttempts,
                maxIntervalMs: 30000,
            });
            do {
                const { Certificate: cert } = await this.acmClient.describeCertificate({
                    CertificateArn: arn,
                }).promise();
                inUseByResources = cert.InUseBy || [];
                if (inUseByResources.length) {
                    console.log(`Sleeping -- Resource ${arn} in use by ${inUseByResources.join(', ')}`);
                    await backoffGenerator.backoff();
                }
                else {
                    break;
                }
            } while (backoffGenerator.shouldContinue());
            if (inUseByResources.length) {
                throw new Error(`Response from describeCertificate did not contain an empty InUseBy list after ${maxAttempts} attempts.`);
            }
            console.log(`Deleting resource for '${key}'`);
            try {
                await this.acmClient.deleteCertificate({ CertificateArn: arn }).promise();
            }
            catch (e) {
                // AccessDeniedException can happen if either:
                //  a) We do not have the required permission to delete the Certificate (unlikely)
                //  b) The Certificate has already been deleted (more likely)
                if (e.message.indexOf('AccessDeniedException')) {
                    console.warn(`Could not delete Certificate ${arn}. Please ensure it has been deleted.`);
                }
                throw e; // Rethrow so the custom resource handler will error-out.
            }
            await resourceTable.deleteItem({
                primaryKeyValue: physicalId,
                sortKeyValue: key,
            });
        }
    }
    async importAndStoreCertificate(args) {
        let certificateArn;
        const sortKey = crypto.createHash('md5').update(args.cert).digest('hex');
        const existingItem = await args.resourceTable.getItem({
            primaryKeyValue: args.physicalId,
            sortKeyValue: sortKey,
        });
        if (existingItem) {
            if (!existingItem.ARN) {
                throw Error("Database Item missing 'ARN' attribute");
            }
            // Verify that the cert is in ACM
            certificateArn = existingItem.ARN;
            try {
                await this.acmClient.getCertificate({ CertificateArn: certificateArn }).promise();
            }
            catch (e) {
                throw Error(`Database entry ${existingItem.ARN} could not be found in ACM: ${JSON.stringify(e)}`);
            }
            // Update the cert by performing an import again, with the new values.
            const importCertRequest = {
                CertificateArn: certificateArn,
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            await this.importCertificate(importCertRequest);
        }
        else {
            const importCertRequest = {
                Certificate: args.cert,
                CertificateChain: args.certChain,
                PrivateKey: args.key,
                Tags: args.tags,
            };
            const resp = await this.importCertificate(importCertRequest);
            if (!resp.CertificateArn) {
                throw new Error(`CertificateArn was not properly populated after attempt to import ${args.cert}`);
            }
            certificateArn = resp.CertificateArn;
            await args.resourceTable.putItem({
                primaryKeyValue: args.physicalId,
                sortKeyValue: sortKey,
                attributes: {
                    ARN: certificateArn,
                },
                allow_overwrite: false,
            });
        }
        return certificateArn;
    }
    async importCertificate(importCertRequest) {
        var _a;
        // ACM cert imports are limited to 1 per second (see https://docs.aws.amazon.com/acm/latest/userguide/acm-limits.html#api-rate-limits)
        // We need to backoff & retry in the event that two imports happen in the same second
        const maxAttempts = 10;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base: 200,
            jitterDivisor: 4,
            maxAttempts,
        });
        let retry = false;
        do {
            try {
                return await this.acmClient.importCertificate(importCertRequest).promise();
            }
            catch (e) {
                console.warn(`Could not import certificate: ${e}`);
                retry = await backoffGenerator.backoff();
                if (retry) {
                    console.log('Retrying...');
                }
            }
        } while (retry);
        throw new Error(`Failed to import certificate ${(_a = importCertRequest.CertificateArn) !== null && _a !== void 0 ? _a : ''} after ${maxAttempts} attempts.`);
    }
    async getSecretString(SecretId) {
        console.debug(`Retrieving secret: ${SecretId}`);
        const resp = await this.secretsManagerClient.getSecretValue({ SecretId }).promise();
        if (!resp.SecretString) {
            throw new Error(`Secret ${SecretId} did not contain a SecretString as expected`);
        }
        return resp.SecretString;
    }
}
exports.AcmCertificateImporter = AcmCertificateImporter;
/**
 * The handler used to import an X.509 certificate to ACM from a Secret
 */
/* istanbul ignore next */
async function importCert(event, context) {
    const handler = new AcmCertificateImporter(new aws_sdk_1.ACM({ apiVersion: ACM_VERSION }), new aws_sdk_1.DynamoDB({ apiVersion: DYNAMODB_VERSION }), new aws_sdk_1.SecretsManager({ apiVersion: SECRETS_MANAGER_VERSION }));
    return await handler.handler(event, context);
}
exports.importCert = importCert;
//# sourceMappingURL=data:application/json;base64,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