"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.wait = exports.WaitForStableServiceResource = void 0;
/* eslint-disable no-console */
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const custom_resource_1 = require("../lib/custom-resource");
;
/**
 * A custom resource used to save Spot Event Plugin server data and configurations.
 */
class WaitForStableServiceResource extends custom_resource_1.SimpleCustomResource {
    constructor(ecsClient) {
        super();
        this.ecsClient = ecsClient;
    }
    /**
     * @inheritdoc
     */
    validateInput(data) {
        return this.implementsWaitForStableServiceResourceProps(data);
    }
    /**
     * @inheritdoc
     */
    async doCreate(_physicalId, resourceProperties) {
        const options = {
            services: resourceProperties.services,
            cluster: resourceProperties.cluster,
        };
        try {
            console.log(`Waiting for ECS services to stabilize. Cluster: ${resourceProperties.cluster}. Services: ${resourceProperties.services}`);
            await this.ecsClient.waitFor('servicesStable', options).promise();
            console.log('Finished waiting. ECS services are stable.');
        }
        catch (e) {
            throw new Error(`ECS services failed to stabilize in expected time: ${e.code} -- ${e.message}`);
        }
        return undefined;
    }
    /**
     * @inheritdoc
     */
    async doDelete(_physicalId, _resourceProperties) {
        // Nothing to do -- we don't modify anything.
        return;
    }
    implementsWaitForStableServiceResourceProps(value) {
        if (!value || typeof (value) !== 'object' || Array.isArray(value)) {
            return false;
        }
        if (!value.cluster || typeof (value.cluster) !== 'string') {
            return false;
        }
        if (!value.services || !Array.isArray(value.services)) {
            return false;
        }
        for (let service of value.services) {
            if (typeof (service) !== 'string') {
                return false;
            }
        }
        if (value.forceRun && typeof (value.forceRun) !== 'string') {
            return false;
        }
        return true;
    }
}
exports.WaitForStableServiceResource = WaitForStableServiceResource;
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
/* istanbul ignore next */
async function wait(event, context) {
    const handler = new WaitForStableServiceResource(new aws_sdk_1.ECS());
    return await handler.handler(event, context);
}
exports.wait = wait;
//# sourceMappingURL=data:application/json;base64,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