"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.padFilesystem = exports.getDiskUsage = exports.shrinkFilesystem = exports.growFilesystem = exports.setDefaultFilesize = void 0;
/* eslint-disable no-console */
const fs_1 = require("fs");
const path_1 = require("path");
const filesystem_ops_1 = require("./filesystem-ops");
/**
 * Default filesize for lambda operation, in MiB.
 * External code that calls this assumes that this is exactly 1GiB=1024MiB
 */
var FILESIZE = 1024;
/**
 * Provided solely for the purpose of testing to shrink the default file size from 1GiB
 * to something smaller.
 * @param filesize Desired filesize in MiB. Must be a multiple of 10
 */
function setDefaultFilesize(filesize) {
    FILESIZE = filesize;
}
exports.setDefaultFilesize = setDefaultFilesize;
/**
 * Local helper to extract the desiredPadding field from the input event, and validate the input.
 * @param event Event object passed to the lambda handler.
 * @returns Value of desiredPadding or throws if invalid.
 */
function getDesiredPadding(event) {
    const desiredPadding = parseInt(event.desiredPadding);
    if (isNaN(desiredPadding)) {
        throw new Error(`Could not parse 'desiredPadding' field of the given event: ${event.desiredPadding}`);
    }
    return desiredPadding;
}
/**
 * Local helper to extract the mountPoint field from the input event, and validate the input.
 * @param event Event object passed to the lambda handler.
 * @returns Value of mountPoint or throws if invalid.
 */
function getMountPoint(event) {
    const mountPoint = event.mountPoint;
    if (!mountPoint) {
        throw new Error(`Invalid mount point given in event: ${mountPoint}`);
    }
    return mountPoint;
}
/**
 * Add numbered files (e.g. 00000, 00001) of a given size to a filesystem.
 * Note: exported so that we can test it.
 * @param numFilesToAdd How many numbered files to add.
 * @param filesize Size, in MiB, of the files to create.
 * @param mountPoint Directory in which to create the directory.
 */
async function growFilesystem(numFilesToAdd, filesize, mountPoint) {
    // We find the highest numbered file created thus far, and start adding numbered files after it.
    let filename = await filesystem_ops_1.determineNextSequentialFilename(mountPoint);
    for (let i = 0; i < numFilesToAdd; i++) {
        const outfilename = path_1.join(mountPoint, filename);
        await filesystem_ops_1.writePaddingFile(outfilename, filesize);
        filename = filesystem_ops_1.nextSequentialFile(filename);
    }
}
exports.growFilesystem = growFilesystem;
/**
 * Delete a given number of numbered files from the given filesystem.
 * Note: exported so that we can test it.
 * @param numFilesToRemove How many files to remove from the directory.
 * @param mountPoint Directory from which to delete files.
 */
async function shrinkFilesystem(numFilesToRemove, mountPoint) {
    // Find all of the numbered "files" in the directory, and then delete the highest numbered ones until
    // we've deleted as many as we need to.
    const numberedFiles = await filesystem_ops_1.listNumberedFiles(mountPoint);
    let numFilesDeleted = 0;
    let index = numberedFiles.length - 1;
    while (numFilesDeleted < numFilesToRemove && index >= 0) {
        const filename = path_1.join(mountPoint, numberedFiles[index]);
        try {
            const stat = await fs_1.promises.stat(filename);
            if (stat.isFile()) {
                console.log(`Deleting: ${filename}`);
                try {
                    await fs_1.promises.unlink(filename);
                    numFilesDeleted += 1;
                }
                catch (err) {
                    console.error(`Unable to delete: ${filename} -- Error: ${err.message}`);
                }
            }
        }
        catch (err) {
            console.warn(`Warning: Unable to stat file '${filename}'`);
        }
        index -= 1;
    }
    console.log(`Deleted ${numFilesDeleted} numbered files`);
    if (numFilesToRemove !== numFilesDeleted) {
        throw Error('Could not delete the desired number of files.');
    }
}
exports.shrinkFilesystem = shrinkFilesystem;
/**
 * Lambda handler. Expected to be invoked from a step function.
 * Calculates the disk size under the given directory. This is equivalent to calling:
 * du -sh -BG <directory>
 * @param event { "mountPoint": <string> }
 * @param context
 * @returns Disk usage in GiB
 */
async function getDiskUsage(event, context) {
    console.log(`Executing event: ${JSON.stringify(event)}`);
    console.log(`Context: ${JSON.stringify(context)}`);
    const mountPoint = getMountPoint(event);
    try {
        // Make sure that the given directory has been mounted before continuing.
        await filesystem_ops_1.ensureIsDirectory(mountPoint);
    }
    catch (err) {
        throw new Error(`Mount point '${mountPoint}' is not a directory. Please ensure that the EFS is mounted to this directory.`);
    }
    const scaledUsage = Math.floor(await filesystem_ops_1.diskUsage(mountPoint) / FILESIZE);
    return scaledUsage;
}
exports.getDiskUsage = getDiskUsage;
/**
 * Lambda handler. Expected to be invoked from a step function.
 * Adds or removes files from a given EFS filesystem to pad the filesystem with data.
 *
 * If the filesystem is padded to less than the number of desired GiB then files are added as numbered
 * files 1GiB in size to the given 'mountPoint'; adding at most 20 on each invocation.
 *
 * If the filesystem is padded with more than the desired GiB then numbered files are removed from the
 * given filesystem. Each numbered file is assumed to be exactly 1GiB in size.
 * @param event {
 *    "desiredPadding": "<integer number of GiB>",
 *    "mountPoint": "<string>"
 * }
 * @param context
 * @returns Nothing
 */
async function padFilesystem(event, context) {
    console.log(`Executing event: ${JSON.stringify(event)}`);
    console.log(`Context: ${JSON.stringify(context)}`);
    const desiredPadding = getDesiredPadding(event);
    const mountPoint = getMountPoint(event);
    try {
        // Make sure that the given directory has been mounted before continuing.
        await filesystem_ops_1.ensureIsDirectory(mountPoint);
    }
    catch (err) {
        throw new Error(`Mount point '${mountPoint}' is not a directory. Please ensure that the EFS is mounted to this directory.`);
    }
    const scaledUsage = Math.floor(await filesystem_ops_1.diskUsage(mountPoint) / FILESIZE);
    console.log(`Access point contains ${scaledUsage}GiB (rounded down) of data. Desired size is ${desiredPadding}GiB.`);
    if (scaledUsage < desiredPadding) {
        // Create files.
        // We'll be running this in a loop driven by a step function. Limit to 20GiB written per invocation,
        // just to avoid any chance of hitting a lambda timeout.
        const delta = Math.min(desiredPadding - scaledUsage, 20);
        console.log(`Adding ${delta}GiB of files to the filesystem to grow size.`);
        await growFilesystem(delta, FILESIZE, mountPoint);
    }
    else if (scaledUsage > desiredPadding) {
        // Remove files
        const delta = scaledUsage - desiredPadding;
        console.log(`Removing ${delta}GiB of files from the filesystem`);
        await shrinkFilesystem(delta, mountPoint);
    }
    else {
        console.log('No change to filesystem required.');
    }
}
exports.padFilesystem = padFilesystem;
//# sourceMappingURL=data:application/json;base64,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