"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const AWS = require("aws-sdk");
const aws_sdk_mock_1 = require("aws-sdk-mock");
const handler_1 = require("../handler");
jest.mock('../../lib/secrets-manager/read-certificate');
const secretArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('readCertificateData', () => {
    test('success', async () => {
        // GIVEN
        const certData = 'BEGIN CERTIFICATE';
        jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve(certData));
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readCertificateData'](secretArn);
        // THEN
        expect(data).toStrictEqual(certData);
    });
    test('failure', async () => {
        // GIVEN
        jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockImplementation(() => {
            throw new Error('must contain a Certificate in PEM format');
        });
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readCertificateData'](secretArn)).rejects.toThrowError(/must contain a Certificate in PEM format/);
    });
});
describe('readLoginCredentials', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const loginData = {
            username: 'testuser',
            password: 'testpassword',
        };
        const secretContents = {
            SecretString: JSON.stringify(loginData),
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readLoginCredentials'](secretArn);
        // THEN
        expect(data).toStrictEqual(loginData);
    });
    test('binary data', async () => {
        // GIVEN
        const loginData = Buffer.from('some binary data', 'utf-8');
        const secretContents = {
            SecretBinary: loginData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readLoginCredentials'](secretArn)).rejects.toThrowError(/must be a JSON encoded string/);
    });
    test.each([
        ['}{', /Failed to parse JSON in MongoDB login credentials/],
        [
            JSON.stringify({
                password: 'testpassword',
            }),
            /is missing: username/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
            }),
            /is missing: password/,
        ],
    ])('failed: %p to get %p', async (data, expected) => {
        // GIVEN
        const secretContents = {
            SecretString: data,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readLoginCredentials'](secretArn)).rejects.toThrowError(expected);
    });
});
describe('mongoLogin', () => {
    beforeEach(() => {
        jest.spyOn(console, 'log').mockReturnValue(undefined);
    });
    afterEach(() => {
        jest.clearAllMocks();
    });
    test('mongoLogin', async () => {
        // GIVEN
        async function stringSuccessRequestMock(value) {
            return value;
        }
        const mockReadCert = jest.fn((request) => stringSuccessRequestMock(request));
        const mockReadLogin = jest.fn((request) => successRequestMock(request, { username: 'test', password: 'pass' }));
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // tslint:disable-next-line: no-string-literal
        handler['readCertificateData'] = mockReadCert;
        // tslint:disable-next-line: no-string-literal
        handler['readLoginCredentials'] = mockReadLogin;
        const mockDriver = {
            MongoClient: {
                connect: jest.fn(),
            },
        };
        const loginOptions = {
            Hostname: 'testhostname',
            Port: '27017',
            Credentials: 'some credentials',
            CaCertificate: 'cert arn',
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['mongoLogin'](mockDriver, loginOptions);
        // THEN
        expect(mockReadCert.mock.calls.length).toBe(1);
        expect(mockReadCert.mock.calls[0][0]).toStrictEqual(loginOptions.CaCertificate);
        expect(mockReadLogin.mock.calls.length).toBe(1);
        expect(mockReadLogin.mock.calls[0][0]).toStrictEqual(loginOptions.Credentials);
        expect(mockDriver.MongoClient.connect.mock.calls.length).toBe(1);
        expect(mockDriver.MongoClient.connect.mock.calls[0][0]).toStrictEqual('mongodb://testhostname:27017');
        expect(mockDriver.MongoClient.connect.mock.calls[0][1]).toStrictEqual({
            tls: true,
            tlsInsecure: false,
            tlsCAFile: '/tmp/ca.crt',
            auth: {
                user: 'test',
                password: 'pass',
            },
            useUnifiedTopology: true,
        });
    });
});
describe('readPasswordAuthUserInfo', () => {
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    test('success', async () => {
        // GIVEN
        const userData = {
            username: 'testuser',
            password: 'testpassword',
            roles: [{ role: 'readWrite', db: 'somedb' }],
        };
        const secretContents = {
            SecretString: JSON.stringify(userData),
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const data = await handler['readPasswordAuthUserInfo'](secretArn);
        // THEN
        expect(data).toStrictEqual(userData);
    });
    test('binary data', async () => {
        // GIVEN
        const loginData = Buffer.from('Some binary data', 'utf-8');
        const secretContents = {
            SecretBinary: loginData,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readPasswordAuthUserInfo'](secretArn)).rejects.toThrowError(/must be a JSON encoded string/);
    });
    test.each([
        ['}{', /Failed to parse JSON for password-auth user Secret/],
        [
            JSON.stringify({
                password: 'testpassword',
                roles: [{ role: 'readWrite', db: 'somedb' }],
            }),
            /is missing: username/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
                roles: [{ role: 'readWrite', db: 'somedb' }],
            }),
            /is missing: password/,
        ],
        [
            JSON.stringify({
                username: 'testuser',
                password: 'testpassword',
            }),
            /is missing: roles/,
        ],
    ])('failed: %p to get %p', async (data, expected) => {
        // GIVEN
        const secretContents = {
            SecretString: data,
        };
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['readPasswordAuthUserInfo'](secretArn)).rejects.toThrowError(expected);
    });
});
describe('userExists', () => {
    test('user exists', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [
                {
                    _id: 'admin.test',
                    user: 'test',
                    db: 'admin',
                },
            ],
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const exists = await handler['userExists'](mockDb, 'test');
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: 'test',
        });
        expect(exists).toStrictEqual(true);
    });
    test('user does not exists', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [],
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const exists = await handler['userExists'](mockDb, 'test');
        // THEN
        expect(exists).toStrictEqual(false);
    });
    test('request failed', async () => {
        // GIVEN
        const mongoQueryResult = {
            users: [],
            ok: 0,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['userExists'](mockDb, 'test')).rejects.toThrowError(/MongoDB error checking whether user exists 'test'/);
    });
});
describe('createUser', () => {
    let consoleLogMock;
    beforeEach(() => {
        consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
    });
    afterEach(() => {
        jest.clearAllMocks();
    });
    test('create success with password', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            password: 'password',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['createUser'](mockDb, credentials);
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            createUser: credentials.username,
            pwd: credentials.password,
            roles: credentials.roles,
        });
        expect(consoleLogMock.mock.calls.length).toBe(1);
        expect(consoleLogMock.mock.calls[0][0]).toStrictEqual('Creating user: test');
    });
    test('create success no password', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 1,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // WHEN
        // tslint:disable-next-line: no-string-literal
        await handler['createUser'](mockDb, credentials);
        // THEN
        expect(mockDb.command.mock.calls.length).toBe(1);
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            createUser: credentials.username,
            roles: credentials.roles,
        });
        expect(consoleLogMock.mock.calls.length).toBe(1);
        expect(consoleLogMock.mock.calls[0][0]).toStrictEqual('Creating user: test');
    });
    test('request failed', async () => {
        // GIVEN
        const mongoQueryResult = {
            ok: 0,
        };
        const mockDb = {
            command: jest.fn((request) => successRequestMock(request, mongoQueryResult)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        const credentials = {
            username: 'test',
            password: 'password',
            roles: [{ role: 'readWrite', db: 'testdb' }],
        };
        // THEN
        // tslint:disable-next-line: no-string-literal
        await expect(handler['createUser'](mockDb, credentials)).rejects.toThrowError(/MongoDB error when adding user 'test'/);
    });
});
describe('createPasswordAuthUser', () => {
    let consoleLogMock;
    // GIVEN
    const username = 'testuser';
    const password = 'testpassword';
    const roles = [{ role: 'readwrite', db: 'somedb' }];
    const userData = {
        username,
        password,
        roles,
    };
    const secretContents = {
        SecretString: JSON.stringify(userData),
    };
    beforeEach(() => {
        // GIVEN
        aws_sdk_mock_1.setSDKInstance(AWS);
        consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
        const mockGetSecret = jest.fn((request) => successRequestMock(request, secretContents));
        aws_sdk_mock_1.mock('SecretsManager', 'getSecretValue', mockGetSecret);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
        jest.clearAllMocks();
    });
    test('existing user', async () => {
        // GIVEN
        const userExistsResponse = {
            users: [],
            ok: 1,
        };
        const addUserResponse = {
            ok: 1,
        };
        async function commandMock(request) {
            if ('createUser' in request) {
                return addUserResponse;
            }
            return userExistsResponse;
        }
        const mockDb = {
            command: jest.fn((request) => commandMock(request)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const result = await handler['createPasswordAuthUser'](mockDb, secretArn);
        expect(result).toStrictEqual(true);
        expect(mockDb.command.mock.calls.length).toBe(2);
        // Check args of userExits DB query.
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: username,
        });
        // Check args of createUser DB query.
        expect(mockDb.command.mock.calls[1][0]).toStrictEqual({
            createUser: username,
            pwd: password,
            roles,
        });
        expect(consoleLogMock.mock.calls.length).toBe(1);
        expect(consoleLogMock.mock.calls[0][0]).toStrictEqual(`Creating user: ${username}`);
    });
    test('non-existing user', async () => {
        // GIVEN
        const userExistsResponse = {
            users: [
                {
                    _id: 'admin.test',
                    user: 'test',
                    db: 'admin',
                },
            ],
            ok: 1,
        };
        const addUserResponse = {
            ok: 1,
        };
        async function commandMock(request) {
            if ('createUser' in request) {
                return addUserResponse;
            }
            return userExistsResponse;
        }
        const mockDb = {
            command: jest.fn((request) => commandMock(request)),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // WHEN
        // tslint:disable-next-line: no-string-literal
        const result = await handler['createPasswordAuthUser'](mockDb, secretArn);
        expect(result).toStrictEqual(false);
        expect(mockDb.command.mock.calls.length).toBe(1);
        // Check args of userExits DB query.
        expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
            usersInfo: username,
        });
    });
});
describe('createX509AuthUser', () => {
    let consoleLogMock;
    const username = 'CN=TestUser,O=TestOrg,OU=TestOrgUnit';
    beforeEach(() => {
        aws_sdk_mock_1.setSDKInstance(AWS);
        consoleLogMock = jest.spyOn(console, 'log')
            .mockReset()
            .mockReturnValue(undefined);
    });
    afterEach(() => {
        aws_sdk_mock_1.restore('SecretsManager');
    });
    describe.each([
        [
            [], true,
        ],
        [
            [
                {
                    _id: '$external.CN=myName,OU=myOrgUnit,O=myOrg',
                    user: 'CN=myName,OU=myOrgUnit,O=myOrg',
                    db: '$external',
                },
            ],
            false,
        ],
    ])('userExists %p gives %p', (userExists, expected) => {
        let mockDb;
        let result;
        // GIVEN
        const dbCommandExpectedCallCount = expected ? 2 : 1;
        const userExistsResponse = {
            users: userExists,
            ok: 1,
        };
        const addUserResponse = {
            ok: 1,
        };
        const roles = [{ role: 'readWrite', db: 'somedb' }];
        beforeEach(async () => {
            // GIVEN
            async function commandMock(request) {
                if ('createUser' in request) {
                    return addUserResponse;
                }
                return userExistsResponse;
            }
            mockDb = {
                command: jest.fn((request) => commandMock(request)),
            };
            async function stringSuccessRequestMock(value) {
                return value;
            }
            async function rfc2253(_arg) {
                return username;
            }
            const mockReadCert = jest.fn((request) => stringSuccessRequestMock(request));
            const mockRfc2253 = jest.fn((arg) => rfc2253(arg));
            const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
            // tslint:disable-next-line: no-string-literal
            handler['readCertificateData'] = mockReadCert;
            // tslint:disable-next-line: no-string-literal
            handler['retrieveRfc2253Subject'] = mockRfc2253;
            const userData = {
                certificate: secretArn,
                roles,
            };
            const userToCreate = {
                Certificate: userData.certificate,
                Roles: JSON.stringify(userData.roles),
            };
            // WHEN
            // tslint:disable-next-line: no-string-literal
            result = await handler['createX509AuthUser'](mockDb, userToCreate);
        });
        // THEN
        test('returns expected result', () => {
            expect(result).toStrictEqual(expected);
        });
        test(`db.command called ${dbCommandExpectedCallCount} times`, () => {
            expect(mockDb.command.mock.calls.length).toBe(dbCommandExpectedCallCount);
        });
        test('correct arguments passed to userExits DB query', () => {
            expect(mockDb.command.mock.calls[0][0]).toStrictEqual({
                usersInfo: username,
            });
        });
        if (expected) {
            test('correct arguments passed to createUser DB query', () => {
                expect(mockDb.command.mock.calls[1][0]).toStrictEqual({
                    createUser: username,
                    roles,
                });
            });
            test('user creation logged to output', () => {
                expect(consoleLogMock.mock.calls.length).toBe(1);
                expect(consoleLogMock.mock.calls[0][0]).toStrictEqual(`Creating user: ${username}`);
            });
        }
    });
});
describe('doCreate', () => {
    let consoleLogMock;
    let mockedHandler;
    let mockMongoClient;
    beforeEach(() => {
        consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
        mockMongoClient = {
            db: jest.fn(),
            close: jest.fn(),
        };
        const handler = new handler_1.MongoDbConfigure(new AWS.SecretsManager());
        // tslint:disable-next-line: no-string-literal
        handler['installMongoDbDriver'] = jest.fn();
        async function returnMockMongoClient(_v1, _v2) {
            return mockMongoClient;
        }
        // tslint:disable-next-line: no-string-literal
        handler['mongoLogin'] = jest.fn((a, b) => returnMockMongoClient(a, b));
        mockedHandler = handler;
    });
    afterEach(() => {
        jest.clearAllMocks();
    });
    test('create password auth user', async () => {
        // GIVEN
        async function returnTrue(_v1, _v2) {
            return true;
        }
        const mockCreatePwAuthUser = jest.fn((a, b) => returnTrue(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createPasswordAuthUser'] = mockCreatePwAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            PasswordAuthUsers: ['arn1', 'arn2'],
        };
        // WHEN
        const result = await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(result).toBeUndefined();
        expect(mockCreatePwAuthUser.mock.calls.length).toBe(2);
        expect(mockCreatePwAuthUser.mock.calls[0][1]).toBe('arn1');
        expect(mockCreatePwAuthUser.mock.calls[1][1]).toBe('arn2');
    });
    test('log when cannot create password auth user', async () => {
        // GIVEN
        async function returnFalse(_v1, _v2) {
            return false;
        }
        const mockCreatePwAuthUser = jest.fn((a, b) => returnFalse(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createPasswordAuthUser'] = mockCreatePwAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            PasswordAuthUsers: ['arn1'],
        };
        // WHEN
        await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/No action performed for this user./);
    });
    test('create x509 auth user', async () => {
        // GIVEN
        async function returnTrue(_v1, _v2) {
            return true;
        }
        const mockCreateAuthUser = jest.fn((a, b) => returnTrue(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createX509AuthUser'] = mockCreateAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            X509AuthUsers: [
                {
                    Certificate: 'some arn1',
                    Roles: 'json encoded role',
                },
                {
                    Certificate: 'some arn2',
                    Roles: 'json encoded role',
                },
            ],
        };
        // WHEN
        const result = await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(result).toBeUndefined();
        expect(mockCreateAuthUser.mock.calls.length).toBe(2);
        expect(mockCreateAuthUser.mock.calls[0][1]).toStrictEqual(properties.X509AuthUsers[0]);
        expect(mockCreateAuthUser.mock.calls[1][1]).toStrictEqual(properties.X509AuthUsers[1]);
    });
    test('log when cannot create x509 auth user', async () => {
        // GIVEN
        async function returnFalse(_v1, _v2) {
            return false;
        }
        const mockCreateAuthUser = jest.fn((a, b) => returnFalse(a, b));
        // tslint:disable-next-line: no-string-literal
        mockedHandler['createX509AuthUser'] = mockCreateAuthUser;
        const properties = {
            Connection: {
                Hostname: 'testhost',
                Port: '27017',
                Credentials: 'credentialArn',
                CaCertificate: 'certArn',
            },
            X509AuthUsers: [
                {
                    Certificate: 'some arn',
                    Roles: 'json encoded role',
                },
            ],
        };
        // WHEN
        await mockedHandler.doCreate('physicalId', properties);
        // THEN
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/No action performed for this user./);
    });
});
//# sourceMappingURL=data:application/json;base64,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