/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { SecretsManager } from 'aws-sdk';
import { LambdaContext } from '../lib/aws-lambda';
import { CfnRequestEvent, SimpleCustomResource } from '../lib/custom-resource';
import { IConnectionOptions, IMongoDbConfigureResource, IX509AuthenticatedUser } from './types';
export declare class MongoDbConfigure extends SimpleCustomResource {
    protected readonly secretsManagerClient: SecretsManager;
    constructor(secretsManagerClient: SecretsManager);
    /**
     * @inheritdoc
     */
    validateInput(data: object): boolean;
    /**
     * @inheritdoc
     */
    doCreate(physicalId: string, resourceProperties: IMongoDbConfigureResource): Promise<object | undefined>;
    /**
     * @inheritdoc
     */
    doDelete(physicalId: string, resourceProperties: IMongoDbConfigureResource): Promise<void>;
    /**
     * Installs the official NodeJS MongoDB driver into /tmp, and returns the module object for it.
     */
    protected installMongoDbDriver(): any;
    /**
     * Login to MongoDB and return the MongoClient connection object.
     * @param mongoDbDriver
     * @param options
     */
    protected mongoLogin(mongoDbDriver: any, options: IConnectionOptions): Promise<any>;
    /**
     * Retrieve CA certificate data from the Secret with the given ARN.
     * @param certificateArn
     */
    protected readCertificateData(certificateArn: string): Promise<string>;
    /**
     * Use openssl to retrieve the subject, in RFC2253 format, of the given certificate.
     * @param certificateData
     */
    protected retrieveRfc2253Subject(certificateData: string): Promise<string>;
    /**
     * Retrieve the credentials of the user that we're to login to the DB with.
     * @param credentialsArn
     */
    protected readLoginCredentials(credentialsArn: string): Promise<{
        [key: string]: string;
    }>;
    /**
     * Read, from the given Secret, the information for a password-authenticated user to be created
     * in the DB.
     * @param userArn
     */
    protected readPasswordAuthUserInfo(userArn: string): Promise<{
        [key: string]: string;
    }>;
    /**
     * Query the given DB to determine whether or not there is a user with the given username.
     * @param db
     * @param username
     */
    protected userExists(db: any, username: string): Promise<boolean>;
    /**
     * Add a user to the database. This must only be called if you know that the user does not
     * already exist.
     * @param db
     * @param credentials
     */
    protected createUser(db: any, credentials: {
        [key: string]: any;
    }): Promise<void>;
    /**
     * Create a user in the admin DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param userArn
     */
    protected createPasswordAuthUser(db: any, userArn: string): Promise<boolean>;
    /**
     * Create a user in the $external DB if it does not already exist. If it does exist, then do nothing.
     * @param db
     * @param user
     */
    protected createX509AuthUser(db: any, user: IX509AuthenticatedUser): Promise<boolean>;
}
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
export declare function configureMongo(event: CfnRequestEvent, context: LambdaContext): Promise<string>;
