"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const AWS = require("aws-sdk");
const deadline_1 = require("../../../../deadline");
const handler_1 = require("../handler");
jest.mock('../../lib/secrets-manager/read-certificate');
const secretArn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/Cert';
// @ts-ignore
async function successRequestMock(request, returnValue) {
    return returnValue;
}
describe('SEPConfiguratorResource', () => {
    const validConnection = {
        hostname: 'internal-hostname.com',
        protocol: 'HTTPS',
        port: '4433',
        caCertificateArn: secretArn,
    };
    const validLaunchSpecification = {
        IamInstanceProfile: {
            Arn: 'iamInstanceProfileArn',
        },
        ImageId: 'any-ami',
        InstanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL).toString(),
        SecurityGroups: [{
                GroupId: 'sg-id',
            }],
        TagSpecifications: [{
                ResourceType: deadline_1.SpotFleetResourceType.INSTANCE,
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        UserData: 'userdata',
        KeyName: 'keyname',
        SubnetId: 'subnet-id',
        BlockDeviceMappings: [{
                DeviceName: 'device',
                NoDevice: '',
                VirtualName: 'virtualname',
                Ebs: {
                    DeleteOnTermination: true,
                    Encrypted: true,
                    Iops: 10,
                    SnapshotId: 'snapshot-id',
                    VolumeSize: 10,
                    VolumeType: 'volume-type',
                },
            }],
    };
    const validSpotFleetRequestProps = {
        AllocationStrategy: deadline_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED,
        IamFleetRole: 'roleArn',
        LaunchSpecifications: [validLaunchSpecification],
        ReplaceUnhealthyInstances: true,
        TargetCapacity: 1,
        TerminateInstancesWithExpiration: true,
        Type: deadline_1.SpotFleetRequestType.MAINTAIN,
        TagSpecifications: [{
                ResourceType: deadline_1.SpotFleetResourceType.SPOT_FLEET_REQUEST,
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        ValidUntil: core_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
    };
    const validConvertedLaunchSpecifications = {
        BlockDeviceMappings: [{
                DeviceName: 'device',
                Ebs: {
                    DeleteOnTermination: true,
                    Encrypted: true,
                    Iops: 10,
                    SnapshotId: 'snapshot-id',
                    VolumeSize: 10,
                    VolumeType: 'volume-type',
                },
                NoDevice: '',
                VirtualName: 'virtualname',
            }],
        IamInstanceProfile: {
            Arn: 'iamInstanceProfileArn',
        },
        ImageId: 'any-ami',
        KeyName: 'keyname',
        SecurityGroups: [{
                GroupId: 'sg-id',
            }],
        SubnetId: 'subnet-id',
        TagSpecifications: [{
                ResourceType: 'instance',
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
        UserData: 'userdata',
        InstanceType: 't2.small',
    };
    const validConvertedSpotFleetRequestProps = {
        AllocationStrategy: 'capacityOptimized',
        IamFleetRole: 'roleArn',
        LaunchSpecifications: [validConvertedLaunchSpecifications],
        ReplaceUnhealthyInstances: true,
        TargetCapacity: 1,
        TerminateInstancesWithExpiration: true,
        Type: 'maintain',
        ValidUntil: core_1.Expiration.atDate(new Date(2022, 11, 17)).date.toISOString(),
        TagSpecifications: [{
                ResourceType: 'spot-fleet-request',
                Tags: [
                    {
                        Key: 'name',
                        Value: 'test',
                    },
                ],
            }],
    };
    const validSpotFleetRequestConfig = {
        group_name1: validSpotFleetRequestProps,
    };
    const validConvertedSpotFleetRequestConfig = {
        group_name1: validConvertedSpotFleetRequestProps,
    };
    const validSpotEventPluginConfig = {
        AWSInstanceStatus: deadline_1.SpotEventPluginDisplayInstanceStatus.DISABLED,
        DeleteInterruptedSlaves: true,
        DeleteTerminatedSlaves: true,
        IdleShutdown: 20,
        Logging: deadline_1.SpotEventPluginLoggingLevel.STANDARD,
        PreJobTaskMode: deadline_1.SpotEventPluginPreJobTaskMode.CONSERVATIVE,
        Region: 'us-west-2',
        ResourceTracker: true,
        StaggerInstances: 50,
        State: deadline_1.SpotEventPluginState.GLOBAL_ENABLED,
        StrictHardCap: true,
    };
    const validConvertedPluginConfig = {
        AWSInstanceStatus: 'Disabled',
        DeleteInterruptedSlaves: true,
        DeleteTerminatedSlaves: true,
        IdleShutdown: 20,
        Logging: 'Standard',
        PreJobTaskMode: 'Conservative',
        Region: 'us-west-2',
        ResourceTracker: true,
        StaggerInstances: 50,
        State: 'Global Enabled',
        StrictHardCap: true,
    };
    // Valid configurations
    const noPluginConfigs = {
        connection: validConnection,
        spotFleetRequestConfigurations: validSpotFleetRequestConfig,
    };
    const noFleetRequestConfigs = {
        spotPluginConfigurations: validSpotEventPluginConfig,
        connection: validConnection,
    };
    const deadlineGroups = ['group_name'];
    const deadlinePools = ['pool_name'];
    const allConfigs = {
        spotPluginConfigurations: validSpotEventPluginConfig,
        connection: validConnection,
        spotFleetRequestConfigurations: validSpotFleetRequestConfig,
        deadlineGroups,
        deadlinePools,
    };
    const noConfigs = {
        connection: validConnection,
    };
    async function returnTrue(_v1) {
        return true;
    }
    async function returnFalse(_v1) {
        return false;
    }
    describe('doCreate', () => {
        let handler;
        let mockSpotEventPluginClient;
        beforeEach(() => {
            mockSpotEventPluginClient = {
                saveServerData: jest.fn((a) => returnTrue(a)),
                configureSpotEventPlugin: jest.fn((a) => returnTrue(a)),
                addGroups: jest.fn((a) => returnTrue(a)),
                addPools: jest.fn((a) => returnTrue(a)),
            };
            handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            async function returnSpotEventPluginClient(_v1) {
                return mockSpotEventPluginClient;
            }
            // eslint-disable-next-line dot-notation
            handler['spotEventPluginClient'] = jest.fn((a) => returnSpotEventPluginClient(a));
        });
        afterEach(() => {
            jest.clearAllMocks();
        });
        test('with no configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const result = await handler.doCreate('physicalId', noConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(0);
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(0);
        });
        test('save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', noPluginConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            expect(calledWithObject).toEqual(validConvertedSpotFleetRequestConfig);
        });
        test('save spot fleet request configs without BlockDeviceMappings', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const noEbs = {
                ...noPluginConfigs,
                spotFleetRequestConfigurations: {
                    ...validSpotFleetRequestConfig,
                    group_name1: {
                        ...validSpotFleetRequestProps,
                        LaunchSpecifications: [
                            {
                                ...validLaunchSpecification,
                                BlockDeviceMappings: undefined,
                            },
                        ],
                    },
                },
            };
            const convertedNoEbs = {
                ...validConvertedSpotFleetRequestConfig,
                group_name1: {
                    ...validConvertedSpotFleetRequestProps,
                    LaunchSpecifications: [
                        {
                            ...validConvertedLaunchSpecifications,
                            BlockDeviceMappings: undefined,
                        },
                    ],
                },
            };
            // WHEN
            await handler.doCreate('physicalId', noEbs);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            // THEN
            expect(calledWithObject).toEqual(convertedNoEbs);
        });
        test('save spot fleet request configs without Ebs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            const blockDevicesNoEbs = [{
                    DeviceName: 'device',
                }];
            const noEbs = {
                ...noPluginConfigs,
                spotFleetRequestConfigurations: {
                    ...validSpotFleetRequestConfig,
                    group_name1: {
                        ...validSpotFleetRequestProps,
                        LaunchSpecifications: [
                            {
                                ...validLaunchSpecification,
                                BlockDeviceMappings: blockDevicesNoEbs,
                            },
                        ],
                    },
                },
            };
            const convertedNoEbs = {
                ...validConvertedSpotFleetRequestConfig,
                group_name1: {
                    ...validConvertedSpotFleetRequestProps,
                    LaunchSpecifications: [
                        {
                            ...validConvertedLaunchSpecifications,
                            BlockDeviceMappings: blockDevicesNoEbs,
                        },
                    ],
                },
            };
            // WHEN
            await handler.doCreate('physicalId', noEbs);
            const calledWithString = mockSaveServerData.mock.calls[0][0];
            const calledWithObject = JSON.parse(calledWithString);
            // THEN
            expect(calledWithObject).toEqual(convertedNoEbs);
        });
        test('save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validConvertedPluginConfig)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            const result = await handler.doCreate('physicalId', noFleetRequestConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('save server data', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const result = await handler.doCreate('physicalId', allConfigs);
            // THEN
            expect(result).toBeUndefined();
            expect(mockSaveServerData.mock.calls.length).toBe(1);
            expect(mockSaveServerData.mock.calls[0][0]).toEqual(JSON.stringify(validConvertedSpotFleetRequestConfig));
        });
        test('configure spot event plugin', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            const configs = [];
            for (const [key, value] of Object.entries(validConvertedPluginConfig)) {
                configs.push({
                    Key: key,
                    Value: value,
                });
            }
            const securitySettings = [{
                    Key: 'UseLocalCredentials',
                    Value: true,
                },
                {
                    Key: 'NamedProfile',
                    Value: '',
                }];
            // WHEN
            await handler.doCreate('physicalId', allConfigs);
            // THEN
            expect(mockConfigureSpotEventPlugin.mock.calls.length).toBe(1);
            expect(mockConfigureSpotEventPlugin.mock.calls[0][0]).toEqual([...configs, ...securitySettings]);
        });
        test('create groups', async () => {
            // GIVEN
            const mockAddGroups = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.addGroups = mockAddGroups;
            // WHEN
            await handler.doCreate('physicalId', allConfigs);
            // THEN
            expect(mockAddGroups.mock.calls.length).toBe(1);
            expect(mockAddGroups).toHaveBeenCalledWith(deadlineGroups);
        });
        test('create pools', async () => {
            // GIVEN
            const mockAddPools = jest.fn((a) => returnTrue(a));
            mockSpotEventPluginClient.addPools = mockAddPools;
            // WHEN
            await handler.doCreate('physicalId', allConfigs);
            // THEN
            expect(mockAddPools.mock.calls.length).toBe(1);
            expect(mockAddPools).toHaveBeenCalledWith(deadlinePools);
        });
        test('throw when cannot add groups', async () => {
            // GIVEN
            mockSpotEventPluginClient.addGroups = jest.fn((a) => returnFalse(a));
            // WHEN
            const promise = handler.doCreate('physicalId', allConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline group(s) ${allConfigs.deadlineGroups}`);
        });
        test('throw when cannot add pools', async () => {
            // GIVEN
            mockSpotEventPluginClient.addPools = jest.fn((a) => returnFalse(a));
            // WHEN
            const promise = handler.doCreate('physicalId', allConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(`Failed to add Deadline pool(s) ${allConfigs.deadlinePools}`);
        });
        test('throw when cannot save spot fleet request configs', async () => {
            // GIVEN
            const mockSaveServerData = jest.fn((a) => returnFalse(a));
            mockSpotEventPluginClient.saveServerData = mockSaveServerData;
            // WHEN
            const promise = handler.doCreate('physicalId', noPluginConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save spot fleet request with configuration/);
        });
        test('throw when cannot save spot event plugin configs', async () => {
            // GIVEN
            const mockConfigureSpotEventPlugin = jest.fn((a) => returnFalse(a));
            mockSpotEventPluginClient.configureSpotEventPlugin = mockConfigureSpotEventPlugin;
            // WHEN
            const promise = handler.doCreate('physicalId', noFleetRequestConfigs);
            // THEN
            await expect(promise)
                .rejects
                .toThrowError(/Failed to save Spot Event Plugin Configurations/);
        });
    });
    test('doDelete does not do anything', async () => {
        // GIVEN
        const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
        // WHEN
        const promise = await handler.doDelete('physicalId', noConfigs);
        // THEN
        await expect(promise).toBeUndefined();
    });
    describe('.validateInput()', () => {
        describe('should return true', () => {
            test.each([
                allConfigs,
                noPluginConfigs,
                noFleetRequestConfigs,
                noConfigs,
            ])('with valid input', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeTruthy();
            });
        });
        // Invalid connection
        const noProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            port: '4433',
        };
        const noHostnameConnection = {
            protocol: 'HTTPS',
            port: '4433',
        };
        const noPortConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            caCertificateArn: secretArn,
        };
        const invalidHostnameConnection = {
            hostname: 10,
            protocol: 'HTTPS',
            port: '4433',
        };
        const invalidProtocolConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'TCP',
            port: '4433',
        };
        const invalidProtocolTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: ['HTTPS'],
            port: '4433',
        };
        const invalidPortTypeConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: 4433,
        };
        const invalidPortRange1Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '-1',
        };
        const invalidPortRange2Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '65536',
        };
        const invalidPortRange3Connection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: Number.NaN.toString(),
        };
        const invalidCaCertConnection = {
            hostname: 'internal-hostname.us-east-1.elb.amazonaws.com',
            protocol: 'HTTPS',
            port: '4433',
            caCertificateArn: 'notArn',
        };
        describe('should return false if', () => {
            test.each([
                noProtocolConnection,
                noHostnameConnection,
                noPortConnection,
                invalidCaCertConnection,
                invalidHostnameConnection,
                invalidProtocolConnection,
                invalidProtocolTypeConnection,
                invalidPortTypeConnection,
                invalidPortRange1Connection,
                invalidPortRange2Connection,
                invalidPortRange3Connection,
                undefined,
                [],
            ])('invalid connection', (invalidConnection) => {
                // GIVEN
                const input = {
                    spotPluginConfigurations: validSpotEventPluginConfig,
                    connection: invalidConnection,
                    spotFleetRequestConfigurations: validSpotFleetRequestConfig,
                };
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
            test.each([
                undefined,
                [],
                '',
            ])('{input=%s}', (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                const returnValue = handler.validateInput(input);
                // THEN
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.isSecretArnOrUndefined()', () => {
        describe('should return true if', () => {
            test.each([
                secretArn,
                undefined,
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeTruthy();
            });
        });
        describe('should return false if', () => {
            test.each([
                'any string',
                10,
                [],
            ])('{input=%s}', async (input) => {
                // WHEN
                const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
                // eslint-disable-next-line dot-notation
                const returnValue = handler['isSecretArnOrUndefined'](input);
                expect(returnValue).toBeFalsy();
            });
        });
    });
    describe('.spotEventPluginClient()', () => {
        test('creates a valid object with http', async () => {
            // GIVEN
            const validHTTPConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPConnection);
            expect(result).toBeDefined();
        });
        test('creates a valid object with https', async () => {
            // GIVEN
            const validHTTPSConnection = {
                hostname: 'internal-hostname.com',
                protocol: 'HTTP',
                port: '8080',
                caCertificateArn: secretArn,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            jest.requireMock('../../lib/secrets-manager/read-certificate').readCertificateData.mockReturnValue(Promise.resolve('BEGIN CERTIFICATE'));
            // eslint-disable-next-line dot-notation
            const result = await handler['spotEventPluginClient'](validHTTPSConnection);
            expect(result).toBeDefined();
        });
    });
    describe('.toKeyValueArray()', () => {
        test('converts to array of key value pairs', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: 'Disabled',
            };
            const expectedResult = {
                Key: 'AWSInstanceStatus',
                Value: 'Disabled',
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            // eslint-disable-next-line dot-notation
            const returnValue = handler['toKeyValueArray'](pluginConfig);
            // THEN
            expect(returnValue).toContainEqual(expectedResult);
        });
        test('throws with undefined values', () => {
            // GIVEN
            const pluginConfig = {
                AWSInstanceStatus: undefined,
            };
            // WHEN
            const handler = new handler_1.SEPConfiguratorResource(new AWS.SecretsManager());
            function toKeyValueArray() {
                // eslint-disable-next-line dot-notation
                handler['toKeyValueArray'](pluginConfig);
            }
            // THEN
            expect(toKeyValueArray).toThrowError();
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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