"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let vpc;
let workerFleet;
const DEFAULT_CONSTRUCT_ID = 'UBL';
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        jest.spyOn(renderQueue, 'configureSecretsManagementAutoRegistration');
        stack = new core_1.Stack(app, 'Stack', { env });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(stack, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        licenses = [lib_1.UsageBasedLicense.forMaya()];
    });
    function createUbl(props) {
        return new lib_1.UsageBasedLicensing(stack, DEFAULT_CONSTRUCT_ID, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            ...props,
        });
    }
    test('vpcSubnets specified => does not emit warnings', () => {
        // GIVEN
        const vpcSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        };
        // WHEN
        const ubl = createUbl({
            vpcSubnets,
        });
        // THEN
        expect(ubl.node.metadataEntry).not.toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: expect.stringMatching(/dedicated subnet/i),
        }));
    });
    test('vpcSubnets not specified => emits warning about dedicated subnets', () => {
        // WHEN
        const ubl = createUbl();
        // THEN
        expect(ubl.node.metadataEntry).toContainEqual(expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
        }));
    });
    describe('configures auto registration', () => {
        test('default to private subnets', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
        test.each([
            [{
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                }],
        ])('%s', (vpcSubnets) => {
            // WHEN
            const ubl = createUbl({
                vpcSubnets,
            });
            // THEN
            const expectedCall = {
                dependent: ubl.service.node.defaultChild,
                registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                role: lib_1.SecretsManagementRole.CLIENT,
                vpc,
                vpcSubnets,
            };
            // THEN
            expect(renderQueue.configureSecretsManagementAutoRegistration).toHaveBeenCalledWith(expectedCall);
        });
    });
    test('creates an ECS cluster', () => {
        // WHEN
        createUbl();
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            createUbl({
                desiredCount: 2,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const logGroup = ubl.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
        test('uses the supplied security group', () => {
            // GIVEN
            const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'UblSecurityGroup', {
                vpc,
            });
            // WHEN
            createUbl({ securityGroup });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // WHEN
                createUbl();
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                // WHEN
                createUbl({ desiredCount });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // WHEN
            const ubl = createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const ubl = createUbl();
            // WHEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: certificateSecret.secretArn,
                    }),
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        // WHEN
        createUbl({
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = DEFAULT_CONSTRUCT_ID;
        // WHEN
        createUbl({
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // WHEN
            createUbl({
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const ubl = createUbl();
            const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
            workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(workerStack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                }),
            });
            // WHEN
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                assert_1.expect(workerStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(ubl).stackName}:ExportsOutputFnGetAttUBLClusterASGInstanceSecurityGroup*`),
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                createUbl({ licenses: [] });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule from UsageBasedLicensing ASG to RenderQueue ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.connections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 4433,
                ToPort: 4433,
                GroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(ublSecurityGroup.securityGroupId),
            }));
        });
        test('adds ingress rule from RenderQueue ASG to UsageBasedLicensing ASG', () => {
            // GIVEN
            const renderQueueSecurityGroup = renderQueue.backendConnections.securityGroups[0];
            // WHEN
            const ubl = createUbl();
            const ublSecurityGroup = ubl.connections.securityGroups[0];
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 17004,
                ToPort: 17004,
                GroupId: stack.resolve(ublSecurityGroup.securityGroupId),
                SourceSecurityGroupId: stack.resolve(renderQueueSecurityGroup.securityGroupId),
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // WHEN
            createUbl();
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: stack.resolve(`${renderQueue.endpoint.applicationProtocol.toLowerCase()}://${renderQueue.endpoint.socketAddress}`),
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                createUbl();
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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