"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const lib_1 = require("../lib");
const secrets_management_1 = require("../lib/secrets-management");
class MockUserData extends aws_ec2_1.UserData {
    constructor() {
        super();
        this.addCommands = jest.fn();
        this.addCommands = jest.fn();
        this.addOnExitCommands = jest.fn();
        this.render = jest.fn();
        this.addExecuteFileCommand = jest.fn();
        this.addS3DownloadCommand = jest.fn();
        this.addSignalOnExitCommand = jest.fn();
    }
}
class MockDeploymentInstance extends deployment_instance_1.DeploymentInstance {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mockUserData = new MockUserData();
    }
    get userData() {
        return this.mockUserData;
    }
}
function writeSynthedTemplate(stack, outputFile) {
    const template = assert_1.SynthUtils.synthesize(stack).template;
    fs.writeFileSync(outputFile, JSON.stringify(template, null, 2), { encoding: 'utf8' });
}
const DEADLINE_CLIENT_SUBNET_NAME = 'DeadlineClient';
const RENDER_QUEUE_ALB_SUBNET_NAME = 'RenderQueueALB';
describe('SecretsManagementIdentityRegistration', () => {
    let app;
    let dependencyStack;
    let deploymentInstanceStack;
    let stack;
    let vpc;
    let version;
    let repository;
    let deploymentInstance;
    let deploymentInstanceRole;
    let renderQueueSubnets;
    let target;
    // @ts-ignore
    function writeSynthedTemplates() {
        writeSynthedTemplate(deploymentInstanceStack, 'deployment-instance-stack.json');
        writeSynthedTemplate(stack, 'secrets-management-stack.json');
    }
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack');
        deploymentInstanceStack = new core_1.Stack(app, 'DeploymentInstanceStack');
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc', {
            subnetConfiguration: [
                {
                    name: RENDER_QUEUE_ALB_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    cidrMask: 28,
                },
                {
                    name: 'Public',
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
                {
                    name: DEADLINE_CLIENT_SUBNET_NAME,
                    subnetType: aws_ec2_1.SubnetType.PUBLIC,
                    cidrMask: 28,
                },
            ],
        });
        version = new lib_1.VersionQuery(dependencyStack, 'Version');
        deploymentInstance = new MockDeploymentInstance(deploymentInstanceStack, 'DeploymentInstance', {
            vpc,
        });
        renderQueueSubnets = vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME });
    });
    describe('when Repository uses secrets management', () => {
        beforeEach(() => {
            // GIVEN
            repository = new lib_1.Repository(dependencyStack, 'Repository', {
                version,
                vpc,
                secretsManagementSettings: {
                    enabled: true,
                },
            });
            jest.spyOn(repository, 'configureClientInstance');
            // Get a reference to the DeploymentInstance's IAM role L1 resource
            deploymentInstanceRole = (deploymentInstance
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
        });
        function createTarget() {
            target = new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistration', {
                deploymentInstance,
                renderQueueSubnets,
                repository,
                version,
            });
        }
        describe('Deadline Client installer', () => {
            test('grant S3 read to client installer', () => {
                // WHEN
                createTarget();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: assert_1.arrayWith(...deploymentInstanceStack.resolve([
                                version.linuxInstallers.client.s3Bucket.bucketArn,
                                version.linuxInstallers.client.s3Bucket.arnForObjects(version.linuxInstallers.client.objectKey),
                            ])),
                        }),
                    },
                    Roles: [
                        deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                    ],
                }));
            });
            test('downloads and executes Client installer', () => {
                // GIVEN
                const clientInstallerLocalFilename = 'clientInstallerLocalFilename';
                const userData = deploymentInstance.userData;
                userData.addS3DownloadCommand.mockReturnValueOnce(clientInstallerLocalFilename);
                // WHEN
                createTarget();
                // THEN
                expect(userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: version.linuxInstallers.client.s3Bucket,
                    bucketKey: version.linuxInstallers.client.objectKey,
                });
                expect(userData.addCommands).toHaveBeenCalledWith(`chmod +x "${clientInstallerLocalFilename}"`);
                expect(userData.addCommands).toHaveBeenCalledWith([
                    // This is required b/c USER and HOME environment variables are not defined when running
                    // user-data
                    'sudo', '--login',
                    // Run the Deadline Client installer
                    `"${clientInstallerLocalFilename}"`,
                    '--mode', 'unattended',
                    '--connectiontype', 'Remote',
                    '--proxyrootdir', '127.0.0.1:8080',
                    '--noguimode', 'true',
                    '--slavestartup', 'false',
                    '--launcherdaemon', 'false',
                    '--restartstalled', 'true',
                    '--autoupdateoverride', 'False',
                ].join(' '));
            });
        });
        test('grants DeploymentInstance role permissions to describe subnets', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 'ec2:DescribeSubnets',
                        Effect: 'Allow',
                        Resource: '*',
                    }),
                },
                Roles: [stack.resolve(deploymentInstanceRole.ref)],
            }));
        });
        test('configures direct repository connection', () => {
            // WHEN
            createTarget();
            // THEN
            expect(repository.configureClientInstance).toHaveBeenCalledWith({
                host: deploymentInstance,
                mountPoint: expect.any(String),
            });
        });
        test('grants DeploymentInstance read access to the Deadline Secrets Management admin credentials secret', () => {
            // WHEN
            createTarget();
            // THEN
            assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: deploymentInstanceStack.resolve(repository.secretsManagementSettings.credentials.secretArn),
                    }),
                },
                Roles: [
                    deploymentInstanceStack.resolve(deploymentInstanceRole.ref),
                ],
            }));
        });
        describe('Identity registration settings script', () => {
            function getIdentityRegistrationSettingsScript() {
                return target.node.findChild('ConfigureIdentityRegistrationSettingScript');
            }
            test('DeploymentInstance granted S3 read access', () => {
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                assert_1.expect(deploymentInstanceStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: deploymentInstanceStack.resolve([
                                identityRegistrationSettingsScript.bucket.bucketArn,
                                identityRegistrationSettingsScript.bucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [deploymentInstanceStack.resolve(deploymentInstanceRole.ref)],
                }));
            });
            test('DeploymentInstance downloads script', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce('deadlineClientLocalPath');
                deploymentInstance.userData.addS3DownloadCommand.mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath);
                // WHEN
                createTarget();
                const identityRegistrationSettingsScript = getIdentityRegistrationSettingsScript();
                // THEN
                expect(deploymentInstance.userData.addS3DownloadCommand).toHaveBeenCalledWith({
                    bucket: identityRegistrationSettingsScript.bucket,
                    bucketKey: identityRegistrationSettingsScript.s3ObjectKey,
                    localFile: expect.stringMatching(/^\/home\/ec2-user\//),
                });
            });
            test('DeploymentInstance sets ownership and executable permissions for ec2-user', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                // WHEN
                createTarget();
                // THEN
                expect(deploymentInstance.userData.addCommands).toHaveBeenCalledWith(`chmod +x ${identityRegistrationSettingsScriptLocalPath}`, `chown ec2-user:ec2-user ${identityRegistrationSettingsScriptLocalPath}`);
            });
        });
        describe('.addSubnetIdentityRegistrationSetting(...)', () => {
            describe.each([
                [lib_1.SecretsManagementRole.SERVER],
                [lib_1.SecretsManagementRole.CLIENT],
            ])('identityRole=%s', (identityRole) => {
                describe.each([
                    [lib_1.SecretsManagementRegistrationStatus.PENDING],
                    [lib_1.SecretsManagementRegistrationStatus.REGISTERED],
                    [lib_1.SecretsManagementRegistrationStatus.REVOKED],
                ])('registrationStatus=%s', (registrationStatus) => {
                    test('executes identity registration settings configuration script with proper arguments', () => {
                        // GIVEN
                        const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                        (deploymentInstance.userData.addS3DownloadCommand
                            .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                            .mockReturnValueOnce('efsMountScriptLocalPath')
                            .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                            .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                        const clientStack = new core_1.Stack(app, 'ClientStack');
                        createTarget();
                        // WHEN
                        target.addSubnetIdentityRegistrationSetting({
                            dependent: new core_1.Construct(clientStack, 'DeadlineClient'),
                            registrationStatus,
                            role: identityRole,
                            vpc,
                            vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                        });
                        // THEN
                        const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                            return deploymentInstanceStack.resolve(call);
                        });
                        const expectedCall = [{
                                'Fn::Join': [
                                    '',
                                    [
                                        // Command is run as "ec2-user" which has the database credentials stored
                                        `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                        stack.resolve(core_1.Fn.join(' ', [
                                            '--region',
                                            stack.region,
                                            // The Deadline Secrets Management admin credentials secret ARN is passed
                                            '--credentials',
                                            `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                            // The Render Queue's ALB subnets are passed as --connection-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                                .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                            // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                            ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                                .subnetIds.map(subnetID => {
                                                return `--source-subnet "${subnetID},${identityRole},${registrationStatus}"`;
                                            })),
                                        ])),
                                    ],
                                ],
                            }];
                        expect(resolvedCalls).toContainEqual(expectedCall);
                    });
                });
            });
            test('throws execption when using Administrator role', () => {
                // GIVEN
                createTarget();
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: new core_1.Construct(stack, 'Dependent'),
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: lib_1.SecretsManagementRole.ADMINISTRATOR,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError('The Administrator role cannot be set using a Deadline identity registration setting');
            });
            test('throws when two rules for same source subnet with different roles', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const existingRole = lib_1.SecretsManagementRole.SERVER;
                const newRole = lib_1.SecretsManagementRole.CLIENT;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: existingRole,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                        role: newRole,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with role "${existingRole}" but another caller requested "${newRole}"`);
            });
            test('throws when two rules for same source subnet with different registration statuses', () => {
                // GIVEN
                const client1 = new core_1.Construct(stack, 'client1');
                const client2 = new core_1.Construct(stack, 'client2');
                const role = lib_1.SecretsManagementRole.CLIENT;
                const existingStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const newStatus = lib_1.SecretsManagementRegistrationStatus.PENDING;
                createTarget();
                target.addSubnetIdentityRegistrationSetting({
                    dependent: client1,
                    registrationStatus: existingStatus,
                    role,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                });
                // WHEN
                function when() {
                    target.addSubnetIdentityRegistrationSetting({
                        dependent: client2,
                        registrationStatus: newStatus,
                        role,
                        vpc,
                        vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                    });
                }
                // THEN
                expect(when)
                    .toThrowError(`Subnet is already registered with registrationStatus "${existingStatus}" but another caller requested "${newStatus}"`);
            });
            test('de-duplicates subnets', () => {
                // GIVEN
                const identityRegistrationSettingsScriptLocalPath = 'identityRegistrationSettingsScriptLocalPath';
                (deploymentInstance.userData.addS3DownloadCommand
                    .mockReturnValueOnce('deadlineClientInstallerLocalPath')
                    .mockReturnValueOnce('efsMountScriptLocalPath')
                    .mockReturnValueOnce('directRepoConnectionConfigScriptLocalPath')
                    .mockReturnValueOnce(identityRegistrationSettingsScriptLocalPath));
                const clientStack = new core_1.Stack(app, 'ClientStack');
                const client1 = new core_1.Construct(clientStack, 'client1');
                const client2 = new core_1.Construct(clientStack, 'client2');
                createTarget();
                const baseProps = {
                    registrationStatus: lib_1.SecretsManagementRegistrationStatus.REGISTERED,
                    role: lib_1.SecretsManagementRole.CLIENT,
                    vpc,
                    vpcSubnets: { subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME },
                };
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client1,
                });
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    ...baseProps,
                    dependent: client2,
                });
                // THEN
                const resolvedCalls = deploymentInstance.userData.addCommands.mock.calls.map(call => {
                    return deploymentInstanceStack.resolve(call);
                });
                const expectedCall = [{
                        'Fn::Join': [
                            '',
                            [
                                // Command is run as "ec2-user" which has the database credentials stored
                                `sudo --login -u ec2-user ${identityRegistrationSettingsScriptLocalPath} `,
                                stack.resolve(core_1.Fn.join(' ', [
                                    '--region',
                                    stack.region,
                                    // The Deadline Secrets Management admin credentials secret ARN is passed
                                    '--credentials',
                                    `"${repository.secretsManagementSettings.credentials.secretArn}"`,
                                    // The Render Queue's ALB subnets are passed as --connection-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME })
                                        .subnetIds.map(subnetID => `--connection-subnet "${subnetID}"`)),
                                    // The Deadline Client's subnets, desired role, and registration status are passed as --source-subnet args
                                    ...(vpc.selectSubnets({ subnetGroupName: DEADLINE_CLIENT_SUBNET_NAME })
                                        .subnetIds.map(subnetID => {
                                        return `--source-subnet "${subnetID},${baseProps.role},${baseProps.registrationStatus}"`;
                                    })),
                                ])),
                            ],
                        ],
                    }];
                expect(resolvedCalls).toContainEqual(expectedCall);
            });
            test('warns about dedicated subnets when render queue ALB and source subnets match', () => {
                // GIVEN
                createTarget();
                const dependent = new core_1.Construct(stack, 'Dependent');
                const registrationStatus = lib_1.SecretsManagementRegistrationStatus.REGISTERED;
                const role = lib_1.SecretsManagementRole.CLIENT;
                const vpcSubnets = {
                    subnetGroupName: RENDER_QUEUE_ALB_SUBNET_NAME,
                };
                // WHEN
                target.addSubnetIdentityRegistrationSetting({
                    dependent,
                    registrationStatus,
                    role,
                    vpc,
                    vpcSubnets,
                });
                expect(dependent.node.metadataEntry).toContainEqual(expect.objectContaining({
                    type: 'aws:cdk:warning',
                    data: `Deadline Secrets Management is enabled on the Repository and VPC subnets of the Render Queue match the subnets of ${dependent.node.path}. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components`,
                }));
            });
        });
        test('Repository with no admin credentials throws an error', () => {
            // GIVEN
            class RepositoryNoCredentials extends lib_1.Repository {
                constructor(scope, id, props) {
                    super(scope, id, props);
                    this.secretsManagementSettings = {
                        enabled: true,
                        credentials: undefined,
                    };
                }
            }
            repository = new RepositoryNoCredentials(dependencyStack, 'RepoNoCreds', {
                version,
                vpc,
            });
            // WHEN
            const when = createTarget;
            // THEN
            expect(when).toThrowError('Repository does not contain secrets management credentials');
        });
    });
    test('when Repository disables secrets management throws an exception', () => {
        // GIVEN
        repository = new lib_1.Repository(stack, 'Repository', {
            version,
            vpc,
            secretsManagementSettings: {
                enabled: false,
            },
        });
        // WHEN
        function when() {
            new secrets_management_1.SecretsManagementIdentityRegistration(stack, 'IdentityRegistrationSettings', {
                deploymentInstance,
                renderQueueSubnets: vpc.selectSubnets({
                    subnetGroupName: 'RenderQueueALB',
                }),
                repository,
                version,
            });
        }
        // THEN
        expect(when).toThrow();
    });
});
//# sourceMappingURL=data:application/json;base64,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