"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const secrets_management_ref_1 = require("./secrets-management-ref");
const version_1 = require("./version");
const worker_configuration_1 = require("./worker-configuration");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * ![architecture diagram](/diagrams/deadline/WorkerInstanceFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.validateProps(props);
        const minCapacity = (_b = props.minCapacity) !== null && _b !== void 0 ? _b : 1;
        const signals = minCapacity > 0 ? aws_autoscaling_1.Signals.waitForMinCapacity({
            timeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
        }) : undefined;
        if (signals === undefined) {
            core_1.Annotations.of(this).addWarning('Deploying with 0 minimum capacity. If there is an error in the EC2 UserData for this fleet, then your stack deployment will not fail. Watch for errors in your CloudWatch logs.');
        }
        const vpcSubnets = props.vpcSubnets ? props.vpcSubnets : {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        };
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets,
            securityGroup: props.securityGroup,
            minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            signals,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: (_c = props.spotPrice) === null || _c === void 0 ? void 0 : _c.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        // Configure the health monitoring if provided.
        // Note: This must be done *BEFORE* configuring the worker. We rely on the worker configuration
        // script restarting the launcher.
        this.configureHealthMonitor(props);
        const workerConfig = new worker_configuration_1.WorkerInstanceConfiguration(this, id, {
            worker: this.fleet,
            cloudWatchLogSettings: {
                logGroupPrefix: WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX,
                ...props.logGroupProps,
            },
            renderQueue: props.renderQueue,
            workerSettings: props,
            userDataProvider: props.userDataProvider,
        });
        this.listeningPorts = aws_ec2_1.Port.tcpRange(workerConfig.listenerPort, workerConfig.listenerPort + WorkerInstanceFleet.MAX_WORKERS_PER_HOST);
        if (props.renderQueue.repository.secretsManagementSettings.enabled) {
            if (!props.vpcSubnets) {
                core_1.Annotations.of(this).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
            }
            props.renderQueue.configureSecretsManagementAutoRegistration({
                vpc: props.vpc,
                vpcSubnets,
                role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                dependent: this.fleet,
            });
        }
        // Updating the user data with successful cfn-signal commands.
        if (signals) {
            this.fleet.userData.addSignalOnExitCommand(this.fleet);
        }
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowListenerPortTo().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortFrom(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortFrom(other) {
        this.connections.allowFrom(other.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowListenerPortFrom().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortTo(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortTo(other) {
        other.connections.allowTo(this.connections, this.listeningPorts, 'Worker remote command listening port');
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    core_1.Annotations.of(this).addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props) {
        var _b, _c;
        if (props.healthMonitor) {
            const healthCheckPort = (_c = (_b = props.healthCheckConfig) === null || _b === void 0 ? void 0 : _b.port) !== null && _c !== void 0 ? _c : core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
            const configureHealthMonitorScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
                osType: this.fleet.osType,
                baseName: 'configureWorkerHealthCheck',
                rootDir: path.join(__dirname, '..', 'scripts/'),
            });
            configureHealthMonitorScriptAsset.executeOn({
                host: this.fleet,
                args: [
                    `'${healthCheckPort}'`,
                    `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                ],
            });
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            core_1.Annotations.of(this).addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
_a = JSII_RTTI_SYMBOL_1;
WorkerInstanceFleet[_a] = { fqn: "aws-rfdk.deadline.WorkerInstanceFleet", version: "0.39.0" };
/**
 * The min limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 *
 * @stability stable
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * This is the current maximum for number of workers that can be started on a single host. Currently the
 * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
 * but only the first 8 will have their ports opened in the workers' security group.
 */
WorkerInstanceFleet.MAX_WORKERS_PER_HOST = 8;
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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