"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_1 = require("./version");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *    access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *    Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *    Repository construct instead.
 *
 * @stability stable
 */
class Repository extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        if (props.database && ((_b = props.backupOptions) === null || _b === void 0 ? void 0 : _b.databaseRetention)) {
            core_1.Annotations.of(this).addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_d = props.removalPolicy) === null || _d === void 0 ? void 0 : _d.database)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem instanceof core_2.MountableEfs && !props.fileSystem.accessPoint) {
            throw new Error('When using EFS with the Repository, you must provide an EFS Access Point');
        }
        if (((_f = (_e = props.secretsManagementSettings) === null || _e === void 0 ? void 0 : _e.enabled) !== null && _f !== void 0 ? _f : true) && props.database && !props.database.databaseConstruct) {
            throw new Error('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
        }
        this.version = props.version;
        const meetsMinSecretsVersion = !this.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION);
        const secretsManagementIsEnabled = (_h = (_g = props.secretsManagementSettings) === null || _g === void 0 ? void 0 : _g.enabled) !== null && _h !== void 0 ? _h : meetsMinSecretsVersion;
        if (secretsManagementIsEnabled && !meetsMinSecretsVersion) {
            throw new Error(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        }
        this.secretsManagementSettings = {
            enabled: secretsManagementIsEnabled,
            credentials: (_k = (_j = props.secretsManagementSettings) === null || _j === void 0 ? void 0 : _j.credentials) !== null && _k !== void 0 ? _k : (secretsManagementIsEnabled ? new aws_secretsmanager_1.Secret(((_l = props.database) === null || _l === void 0 ? void 0 : _l.databaseConstruct) ? core_1.Stack.of((_m = props.database) === null || _m === void 0 ? void 0 : _m.databaseConstruct) : this, 'SMAdminUser', {
                description: 'Admin credentials for Deadline Secrets Management',
                generateSecretString: {
                    excludeCharacters: '\"$&\'()/<>[\\]\`{|}',
                    includeSpace: false,
                    passwordLength: 24,
                    requireEachIncludedType: true,
                    generateStringKey: 'password',
                    secretStringTemplate: JSON.stringify({ username: Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME }),
                },
                removalPolicy: (_p = (_o = props.secretsManagementSettings) === null || _o === void 0 ? void 0 : _o.credentialsRemovalPolicy) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            }) : undefined),
        };
        this.fileSystem = (_q = props.fileSystem) !== null && _q !== void 0 ? _q : (() => {
            var _b, _c, _d, _e, _f;
            const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_d = (_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.RETAIN,
                securityGroup: (_e = props.securityGroupsOptions) === null || _e === void 0 ? void 0 : _e.fileSystem,
            });
            const paddingAccess = fs.addAccessPoint('PaddingAccessPoint', {
                createAcl: {
                    ownerGid: '0',
                    ownerUid: '0',
                    permissions: '744',
                },
                path: '/RFDK_PaddingFiles',
            });
            new core_2.PadEfsStorage(this, 'PadEfsStorage', {
                vpc: props.vpc,
                vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                accessPoint: paddingAccess,
                desiredPadding: core_1.Size.gibibytes(40),
            });
            const accessPoint = fs.addAccessPoint('AccessPoint', {
                posixUser: {
                    uid: '0',
                    gid: '0',
                },
            });
            return new core_2.MountableEfs(this, {
                filesystem: fs,
                accessPoint,
            });
        })();
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                core_1.Annotations.of(this).addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_r = props.databaseAuditLogging) !== null && _r !== void 0 ? _r : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_s = props.documentDbInstanceCount) !== null && _s !== void 0 ? _s : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                vpc: props.vpc,
                vpcSubnets: (_t = props.vpcSubnets) !== null && _t !== void 0 ? _t : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                securityGroup: (_u = props.securityGroupsOptions) === null || _u === void 0 ? void 0 : _u.database,
                instances,
                backup: {
                    retention: (_w = (_v = props.backupOptions) === null || _v === void 0 ? void 0 : _v.databaseRetention) !== null && _w !== void 0 ? _w : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_y = (_x = props.removalPolicy) === null || _x === void 0 ? void 0 : _x.database) !== null && _y !== void 0 ? _y : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_z = props.vpcSubnets) !== null && _z !== void 0 ? _z : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            }),
            securityGroup: (_0 = props.securityGroupsOptions) === null || _0 === void 0 ? void 0 : _0.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version, props.repositorySettings, 
        // Change ownership of the Deadline repository files if-and-only-if the mounted file-system
        // uses the POSIX permissions based on the process' user UID/GID
        this.fileSystem.usesUserPosixPermissions() ? Repository.REPOSITORY_OWNER : undefined);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof version_query_1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _b, _c;
        const hostMountPoint = (_b = props.containerInstances.filesystemMountPoint) !== null && _b !== void 0 ? _b : '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = (_c = props.containers.filesystemMountPoint) !== null && _c !== void 0 ? _c : `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this.installerGroup);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _b;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this.installerGroup);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _b;
        const prefix = (_b = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _b !== void 0 ? _b : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.installerGroup).add(tagKey, core_1.Names.uniqueId(this));
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)');
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version, settings, owner) {
        var _b, _c;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        const installerArgs = [
            '-i', `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
            '-p', `"${installPath}"`,
            '-v', version.linuxFullVersionString(),
        ];
        if (this.secretsManagementSettings.enabled) {
            installerArgs.push('-r', core_1.Stack.of((_b = this.secretsManagementSettings.credentials) !== null && _b !== void 0 ? _b : this).region);
            this.secretsManagementSettings.credentials.grantRead(installerGroup);
            installerArgs.push('-c', (_c = this.secretsManagementSettings.credentials.secretArn) !== null && _c !== void 0 ? _c : '');
        }
        if (settings) {
            const repositorySettingsFilePath = installerGroup.userData.addS3DownloadCommand({
                bucket: settings.bucket,
                bucketKey: settings.s3ObjectKey,
            });
            installerArgs.push('-s', repositorySettingsFilePath);
        }
        // We can ignore this in test coverage because we always use Repository.REPOSITORY_OWNER
        /* istanbul ignore next */
        if (owner) {
            installerArgs.push('-o', `${owner.uid}:${owner.gid}`);
        }
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: installerArgs,
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "0.39.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
/**
 * The Repository owner is 1000:1000 (ec2-user on AL2).
 */
Repository.REPOSITORY_OWNER = { uid: 1000, gid: 1000 };
/**
 * Default username for auto generated admin credentials in Secret Manager.
 */
Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME = 'admin';
//# sourceMappingURL=data:application/json;base64,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