"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 *
 * @stability stable
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Note: Deadline officially supports only databases that are compatible with MongoDB 3.6.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @stability stable
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
_a = JSII_RTTI_SYMBOL_1;
DatabaseConnection[_a] = { fqn: "aws-rfdk.deadline.DatabaseConnection", version: "0.39.0" };
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        if (!this.isCompatibleDocDBVersion()) {
            core_1.Annotations.of(props.database).addError('engineVersion must be 3.6.0 to be compatible with Deadline');
        }
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
        if (props.database instanceof aws_docdb_1.DatabaseCluster) {
            this.databaseConstruct = props.database;
        }
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        let added = false;
        const errorReasons = [];
        if (this.props.database instanceof aws_docdb_1.DatabaseCluster) {
            const resource = this.props.database.node.tryFindChild('Resource');
            // TODO: Replace this code with the addSecurityGroup method of DatabaseCluster once this PR is merged:
            // https://github.com/aws/aws-cdk/pull/13290
            if (resource instanceof aws_docdb_1.CfnDBCluster) {
                const cfnCluster = resource;
                const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
                if (cfnCluster.vpcSecurityGroupIds === undefined) {
                    cfnCluster.vpcSecurityGroupIds = securityGroupIds;
                }
                else {
                    cfnCluster.vpcSecurityGroupIds.push(...securityGroupIds);
                }
                added = true;
            }
            else {
                errorReasons.push('The internal implementation of AWS CDK\'s DocumentDB cluster construct has changed.');
            }
        }
        else {
            errorReasons.push('The "database" property passed to this class is not an instance of AWS CDK\'s DocumentDB cluster construct.');
        }
        if (!added) {
            core_1.Annotations.of(this.props.database).addWarning(`Failed to add the following security groups to ${this.props.database.node.id}: ${securityGroups.map(sg => sg.node.id).join(', ')}. ` +
                errorReasons.join(' '));
        }
    }
    /**
     * Deadline is only compatible with MongoDB 3.6. This function attempts to determine whether
     * the given DocDB version is compatible.
     */
    isCompatibleDocDBVersion() {
        var _b, _c;
        // The defaultChild of a DocDB DatabaseCluster is a CfnDBCluster, but we only have this
        // child if the customer didn't import from attributes. We can check the DB version by
        // checking the value of the engineVersion property of that object.
        if (this.props.database.node.defaultChild) {
            const cluster = this.props.database.node.defaultChild;
            return (_c = (_b = cluster.engineVersion) === null || _b === void 0 ? void 0 : _b.startsWith('3.6')) !== null && _c !== void 0 ? _c : false;
        }
        return true; // No information, assume it's compatible.
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
        if (props.database instanceof core_2.MongoDbInstance) {
            if (props.database.mongoDataVolume instanceof aws_ec2_1.Volume) {
                this.databaseConstruct = props.database.mongoDataVolume;
            }
        }
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * @inheritdoc
     */
    addSecurityGroup(...securityGroups) {
        this.props.database.addSecurityGroup(...securityGroups);
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        var _b;
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: host.osType,
            baseName: 'getSecretToFile',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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