"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const secrets_management_ref_1 = require("./secrets-management-ref");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 *
 * @stability stable
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 *
 * @stability stable
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 *
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 *
 * @stability stable
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 *
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 *
 * @stability stable
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    vpc,
 *    images: images.forRenderQueue(),
 *    repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *    vpc,
 *    renderQueue,
 *    deadlineGroups: ['group_name'],
 *    instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *    workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *    naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *    vpc,
 *    renderQueue: renderQueue,
 *    spotFleets: [fleet],
 *    configuration: {
 *      enableResourceTracker: true,
 *    },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *   - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *   - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *   - A policy to pass a fleet and instance role
 *   - A policy to create tags for spot fleet requests
 *
 * ![architecture diagram](/diagrams/deadline/ConfigureSpotEventPlugin.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *    and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *    execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4;
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: ['arn:aws:ec2:*:*:spot-fleet-request/*'],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        (_c = props.renderQueue.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: (_e = (_d = props.configuration) === null || _d === void 0 ? void 0 : _d.awsInstanceStatus) !== null && _e !== void 0 ? _e : SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: (_g = (_f = props.configuration) === null || _f === void 0 ? void 0 : _f.deleteEC2SpotInterruptedWorkers) !== null && _g !== void 0 ? _g : false,
            DeleteTerminatedSlaves: (_j = (_h = props.configuration) === null || _h === void 0 ? void 0 : _h.deleteSEPTerminatedWorkers) !== null && _j !== void 0 ? _j : false,
            IdleShutdown: (_m = (_l = (_k = props.configuration) === null || _k === void 0 ? void 0 : _k.idleShutdown) === null || _l === void 0 ? void 0 : _l.toMinutes({ integral: true })) !== null && _m !== void 0 ? _m : 10,
            Logging: (_p = (_o = props.configuration) === null || _o === void 0 ? void 0 : _o.loggingLevel) !== null && _p !== void 0 ? _p : SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: (_r = (_q = props.configuration) === null || _q === void 0 ? void 0 : _q.preJobTaskMode) !== null && _r !== void 0 ? _r : SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: (_t = (_s = props.configuration) === null || _s === void 0 ? void 0 : _s.region) !== null && _t !== void 0 ? _t : region,
            ResourceTracker: (_v = (_u = props.configuration) === null || _u === void 0 ? void 0 : _u.enableResourceTracker) !== null && _v !== void 0 ? _v : true,
            StaggerInstances: (_x = (_w = props.configuration) === null || _w === void 0 ? void 0 : _w.maximumInstancesStartedPerCycle) !== null && _x !== void 0 ? _x : 50,
            State: (_z = (_y = props.configuration) === null || _y === void 0 ? void 0 : _y.state) !== null && _z !== void 0 ? _z : SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: (_1 = (_0 = props.configuration) === null || _0 === void 0 ? void 0 : _0.strictHardCap) !== null && _1 !== void 0 ? _1 : false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const deadlineGroups = Array.from(new Set((_2 = props.spotFleets) === null || _2 === void 0 ? void 0 : _2.map(fleet => fleet.deadlineGroups).reduce((p, c) => p.concat(c), [])));
        const deadlinePools = Array.from(new Set((_3 = props.spotFleets) === null || _3 === void 0 ? void 0 : _3.map(fleet => fleet.deadlinePools).reduce((p, c) => p === null || p === void 0 ? void 0 : p.concat(c !== null && c !== void 0 ? c : []), [])));
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: (_4 = props.renderQueue.certChain) === null || _4 === void 0 ? void 0 : _4.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
            deadlineGroups,
            deadlinePools,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        if (props.spotFleets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.spotFleets.forEach(spotFleet => {
                if (spotFleet.defaultSubnets) {
                    core_1.Annotations.of(spotFleet).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
                }
                props.renderQueue.configureSecretsManagementAutoRegistration({
                    dependent: resource,
                    role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                    registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                    vpc: props.vpc,
                    vpcSubnets: spotFleet.subnets,
                });
            });
        }
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each congiguration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        var _b;
        const securityGroupsToken = core_1.Lazy.any({ produce: () => {
                return fleet.securityGroups.map(sg => {
                    const securityGroupId = {
                        GroupId: sg.securityGroupId,
                    };
                    return securityGroupId;
                });
            } });
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(fleet.userData.render()) });
        const blockDeviceMappings = (fleet.blockDevices !== undefined ?
            this.synthesizeBlockDeviceMappings(fleet.blockDevices) : undefined);
        const { subnetIds } = fleet.subnets;
        const subnetId = subnetIds.join(',');
        const instanceTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.INSTANCE);
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const launchSpecifications = [];
        fleet.instanceTypes.map(instanceType => {
            const launchSpecification = {
                BlockDeviceMappings: blockDeviceMappings,
                IamInstanceProfile: {
                    Arn: fleet.instanceProfile.attrArn,
                },
                ImageId: fleet.imageId,
                KeyName: fleet.keyName,
                // Need to convert from IResolvable to bypass TypeScript
                SecurityGroups: securityGroupsToken,
                SubnetId: subnetId,
                // Need to convert from IResolvable to bypass TypeScript
                TagSpecifications: instanceTagsToken,
                UserData: userDataToken,
                InstanceType: instanceType.toString(),
            };
            launchSpecifications.push(launchSpecification);
        });
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchSpecifications: launchSpecifications,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: (_b = fleet.validUntil) === null || _b === void 0 ? void 0 : _b.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group.toLowerCase()]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    /**
     * Synthesize an array of block device mappings from a list of block devices
     *
     * @param blockDevices list of block devices
     */
    synthesizeBlockDeviceMappings(blockDevices) {
        return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
            const { virtualName, ebsDevice: ebs } = volume;
            if (volume === aws_autoscaling_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
                return {
                    DeviceName: deviceName,
                    // To omit the device from the block device mapping, specify an empty string.
                    // See https://docs.aws.amazon.com/cli/latest/reference/ec2/request-spot-fleet.html
                    NoDevice: '',
                };
            }
            let Ebs;
            if (ebs) {
                const { iops, volumeType, volumeSize, snapshotId, deleteOnTermination } = ebs;
                Ebs = {
                    DeleteOnTermination: deleteOnTermination,
                    Iops: iops,
                    SnapshotId: snapshotId,
                    VolumeSize: volumeSize,
                    VolumeType: volumeType,
                    // encrypted is not exposed as part of ebsDeviceProps so we need to access it via [].
                    // eslint-disable-next-line dot-notation
                    Encrypted: 'encrypted' in ebs ? ebs['encrypted'] : undefined,
                };
            }
            return {
                DeviceName: deviceName,
                Ebs,
                VirtualName: virtualName,
            };
        });
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.39.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7OztHQUdHO0FBRUgsNkJBQTZCO0FBRTdCLDhEQUdrQztBQUNsQyw4Q0FJMEI7QUFDMUIsOENBSTBCO0FBQzFCLG9EQUk2QjtBQUM3QixnREFBa0Q7QUFDbEQsd0NBU3VCO0FBYXZCLGlEQUd3QjtBQUN4QixxRUFHa0M7QUFFbEMsK0VBR3VDO0FBQ3ZDLHVDQUFvQzs7Ozs7O0FBR3BDLElBQVksb0JBTVg7QUFORCxXQUFZLG9CQUFvQjtJQUU5Qix5REFBaUMsQ0FBQTtJQUdqQyw2Q0FBcUIsQ0FBQTtBQUN2QixDQUFDLEVBTlcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUFNL0I7Ozs7OztBQUdELElBQVksMkJBWVg7QUFaRCxXQUFZLDJCQUEyQjtJQUVyQyxvREFBcUIsQ0FBQTtJQUdyQixrREFBbUIsQ0FBQTtJQUduQiw4Q0FBZSxDQUFBO0lBR2YsMENBQVcsQ0FBQTtBQUNiLENBQUMsRUFaVywyQkFBMkIsR0FBM0IsbUNBQTJCLEtBQTNCLG1DQUEyQixRQVl0Qzs7Ozs7Ozs7QUFHRCxJQUFZLDZCQVNYO0FBVEQsV0FBWSw2QkFBNkI7SUFFdkMsOERBQTZCLENBQUE7SUFHN0Isa0RBQWlCLENBQUE7SUFHakIsa0RBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQVRXLDZCQUE2QixHQUE3QixxQ0FBNkIsS0FBN0IscUNBQTZCLFFBU3hDOzs7Ozs7Ozs7QUFHRCxJQUFZLG9DQVlYO0FBWkQsV0FBWSxvQ0FBb0M7SUFDOUMsNkRBQXFCLENBQUE7SUFDckIsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7SUFDM0IsbUVBQTJCLENBQUE7QUFDN0IsQ0FBQyxFQVpXLG9DQUFvQyxHQUFwQyw0Q0FBb0MsS0FBcEMsNENBQW9DLFFBWS9DOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQXlERCxNQUFhLHdCQUF5QixTQUFRLGdCQUFTOzs7O0lBT3JELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBb0M7O1FBQzVFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSx3QkFBd0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMsMEVBQTBFLENBQUMsQ0FBQztTQUM3RjthQUNJO1lBQ0gsd0JBQXdCLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNwRTtRQUVELElBQUksS0FBSyxDQUFDLFdBQVcsWUFBWSwwQkFBVyxFQUFFO1lBQzVDLHVEQUF1RDtZQUN2RCxNQUFNLGNBQWMsR0FBWSxJQUFJLGlCQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRTVELElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksTUFBTTtvQkFDckYsR0FBRyxjQUFjLENBQUMsYUFBYSxJQUFJO29CQUNuQyxhQUFhLEtBQUssQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUM7YUFDMUQ7WUFFRCxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUNyRCx5R0FBeUc7Z0JBQ3pHLHdGQUF3RjtnQkFDeEYsb0hBQW9IO2dCQUNwSCxLQUFLLENBQUMsV0FBVyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztnQkFFdkMsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNwRSxNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNwRixJQUFJLGdCQUFNLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO29CQUN4QyxVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixPQUFPLEVBQUU7Z0NBQ1AsY0FBYzs2QkFDZjs0QkFDRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLFVBQVUsRUFBRSxHQUFHLGtCQUFrQixDQUFDOzRCQUNqRCxVQUFVLEVBQUU7Z0NBQ1YsVUFBVSxFQUFFO29DQUNWLHFCQUFxQixFQUFFLG1CQUFtQjtpQ0FDM0M7NkJBQ0Y7eUJBQ0YsQ0FBQzt3QkFDRixJQUFJLHlCQUFlLENBQUM7NEJBQ2xCLE9BQU8sRUFBRTtnQ0FDUCxnQkFBZ0I7NkJBQ2pCOzRCQUNELFNBQVMsRUFBRSxDQUFDLHNDQUFzQyxDQUFDO3lCQUNwRCxDQUFDO3FCQUNIO29CQUNELEtBQUssRUFBRTt3QkFDTCxLQUFLLENBQUMsV0FBVyxDQUFDLGNBQXNCO3FCQUN6QztpQkFDRixDQUFDLENBQUM7YUFDSjtTQUNGO2FBQ0k7WUFDSCxNQUFNLElBQUksS0FBSyxDQUFDLHlHQUF5RyxDQUFDLENBQUM7U0FDNUg7UUFFRCxNQUFNLE1BQU0sR0FBRyxnQkFBUyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUM7UUFFckgsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3ZCLE1BQU0sWUFBWSxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzVELEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLFVBQVUsUUFBRSxLQUFLLENBQUMsVUFBVSxtQ0FBSSxFQUFFLFVBQVUsRUFBRSxvQkFBVSxDQUFDLE9BQU8sRUFBRTtZQUNsRSxXQUFXLEVBQUUsc0NBQXNDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSx5REFBeUQ7WUFDMUgsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLFFBQVEsQ0FBQyxFQUFFLEVBQzNFLENBQUM7WUFDRixXQUFXLEVBQUU7Z0JBQ1gsS0FBSyxFQUFFLE9BQU87Z0JBQ2QsbUJBQW1CLEVBQUUsV0FBVyxDQUFDLFFBQVEsRUFBRTthQUM1QztZQUNELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7WUFDNUIsT0FBTyxFQUFFLDBDQUEwQztZQUNuRCxPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7WUFDdEMsWUFBWSxFQUFFLHdCQUFhLENBQUMsUUFBUTtTQUNyQyxDQUFDLENBQUM7UUFFSCxZQUFZLENBQUMsV0FBVyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUMvRCxNQUFBLEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUywwQ0FBRSxTQUFTLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRTtRQUVwRSxNQUFNLFlBQVksR0FBbUI7WUFDbkMsaUJBQWlCLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsaUJBQWlCLG1DQUFJLG9DQUFvQyxDQUFDLFFBQVE7WUFDMUcsdUJBQXVCLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsK0JBQStCLG1DQUFJLEtBQUs7WUFDdEYsc0JBQXNCLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsMEJBQTBCLG1DQUFJLEtBQUs7WUFDaEYsWUFBWSxvQkFBRSxLQUFLLENBQUMsYUFBYSwwQ0FBRSxZQUFZLDBDQUFFLFNBQVMsQ0FBQyxFQUFDLFFBQVEsRUFBRSxJQUFJLEVBQUMsb0NBQUssRUFBRTtZQUNsRixPQUFPLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsWUFBWSxtQ0FBSSwyQkFBMkIsQ0FBQyxRQUFRO1lBQ2xGLGNBQWMsY0FBRSxLQUFLLENBQUMsYUFBYSwwQ0FBRSxjQUFjLG1DQUFJLDZCQUE2QixDQUFDLFlBQVk7WUFDakcsTUFBTSxjQUFFLEtBQUssQ0FBQyxhQUFhLDBDQUFFLE1BQU0sbUNBQUksTUFBTTtZQUM3QyxlQUFlLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUscUJBQXFCLG1DQUFJLElBQUk7WUFDbkUsZ0JBQWdCLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsK0JBQStCLG1DQUFJLEVBQUU7WUFDNUUsS0FBSyxjQUFFLEtBQUssQ0FBQyxhQUFhLDBDQUFFLEtBQUssbUNBQUksb0JBQW9CLENBQUMsY0FBYztZQUN4RSxhQUFhLGNBQUUsS0FBSyxDQUFDLGFBQWEsMENBQUUsYUFBYSxtQ0FBSSxLQUFLO1NBQzNELENBQUM7UUFDRixNQUFNLHVCQUF1QixHQUFHLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFcEYsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsT0FBQyxLQUFLLENBQUMsVUFBVSwwQ0FBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNuSSxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxPQUFDLEtBQUssQ0FBQyxVQUFVLDBDQUFFLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxhQUFELENBQUMsdUJBQUQsQ0FBQyxDQUFFLE1BQU0sQ0FBQyxDQUFDLGFBQUQsQ0FBQyxjQUFELENBQUMsR0FBSSxFQUFFLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3hJLE1BQU0sVUFBVSxHQUFpQztZQUMvQyxVQUFVLEVBQUU7Z0JBQ1YsUUFBUSxFQUFFLEtBQUssQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLFFBQVE7Z0JBQzdDLElBQUksRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7Z0JBQy9DLFFBQVEsRUFBRSxLQUFLLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxtQkFBbUI7Z0JBQ3hELGdCQUFnQixRQUFFLEtBQUssQ0FBQyxXQUFXLENBQUMsU0FBUywwQ0FBRSxTQUFTO2FBQ3pEO1lBQ0QsOEJBQThCLEVBQUUsdUJBQXVCO1lBQ3ZELHdCQUF3QixFQUFFLFlBQVk7WUFDdEMsY0FBYztZQUNkLGFBQWE7U0FDZCxDQUFDO1FBRUYsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQkFBYyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkQsWUFBWSxFQUFFLFlBQVksQ0FBQyxXQUFXO1lBQ3RDLFlBQVksRUFBRSx1Q0FBdUM7WUFDckQsVUFBVTtTQUNYLENBQUMsQ0FBQztRQUVILHlDQUF5QztRQUN6QyxRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsSUFBSyxDQUFDLENBQUM7UUFFaEQscUZBQXFGO1FBQ3JGLDBHQUEwRztRQUMxRyw0R0FBNEc7UUFDNUcsK0ZBQStGO1FBQy9GLDBDQUEwQztRQUMxQyxxR0FBcUc7UUFDckcsc0ZBQXNGO1FBQ3RGLHlGQUF5RjtRQUN6RixnRUFBZ0U7UUFDaEUsUUFBUSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRXZDLDZCQUE2QjtRQUM3QixzREFBc0Q7UUFDdEQsc0RBQXNEO1FBQ3RELFFBQVEsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUUvQyxJQUFJLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMseUJBQXlCLENBQUMsT0FBTyxFQUFFO1lBQ3RGLEtBQUssQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFO2dCQUNuQyxJQUFJLFNBQVMsQ0FBQyxjQUFjLEVBQUU7b0JBQzVCLGtCQUFXLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFVBQVUsQ0FDbEMscVJBQXFSLENBQ3RSLENBQUM7aUJBQ0g7Z0JBQ0QsS0FBSyxDQUFDLFdBQVcsQ0FBQywwQ0FBMEMsQ0FBQztvQkFDM0QsU0FBUyxFQUFFLFFBQVE7b0JBQ25CLElBQUksRUFBRSw4Q0FBcUIsQ0FBQyxNQUFNO29CQUNsQyxrQkFBa0IsRUFBRSw0REFBbUMsQ0FBQyxVQUFVO29CQUNsRSxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7b0JBQ2QsVUFBVSxFQUFFLFNBQVMsQ0FBQyxPQUFPO2lCQUM5QixDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDO0lBQ3BDLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxLQUEyQixFQUFFLFlBQW1DO1FBQ3hGLE9BQU8sV0FBSSxDQUFDLEdBQUcsQ0FBQztZQUNkLE9BQU8sRUFBRSxHQUFHLEVBQUU7Z0JBQ1osSUFBSSxLQUFLLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFO29CQUN4QixNQUFNLGdCQUFnQixHQUE4Qjt3QkFDbEQsWUFBWSxFQUFFLFlBQVk7d0JBQzFCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtxQkFDOUIsQ0FBQztvQkFDRixPQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztpQkFDM0I7Z0JBQ0QsT0FBTyxTQUFTLENBQUM7WUFDbkIsQ0FBQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7O09BR0c7SUFDSyw4QkFBOEIsQ0FBQyxLQUEyQjs7UUFDaEUsTUFBTSxtQkFBbUIsR0FBRyxXQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRTtnQkFDbkQsT0FBTyxLQUFLLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRTtvQkFDbkMsTUFBTSxlQUFlLEdBQTZCO3dCQUNoRCxPQUFPLEVBQUUsRUFBRSxDQUFDLGVBQWU7cUJBQzVCLENBQUM7b0JBQ0YsT0FBTyxlQUFlLENBQUM7Z0JBQ3pCLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxFQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sYUFBYSxHQUFHLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsU0FBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBRXpGLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxLQUFLLFNBQVMsQ0FBQyxDQUFDO1lBQzdELElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXRFLE1BQU0sRUFBRSxTQUFTLEVBQUUsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQ3BDLE1BQU0sUUFBUSxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFckMsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLG1EQUFxQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3hGLE1BQU0seUJBQXlCLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxtREFBcUIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRTFHLE1BQU0sb0JBQW9CLEdBQTBCLEVBQUUsQ0FBQztRQUV2RCxLQUFLLENBQUMsYUFBYSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUNyQyxNQUFNLG1CQUFtQixHQUF3QjtnQkFDL0MsbUJBQW1CLEVBQUUsbUJBQW1CO2dCQUN4QyxrQkFBa0IsRUFBRTtvQkFDbEIsR0FBRyxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsT0FBTztpQkFDbkM7Z0JBQ0QsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO2dCQUN0QixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3RCLHdEQUF3RDtnQkFDeEQsY0FBYyxFQUFHLG1CQUE4RDtnQkFDL0UsUUFBUSxFQUFFLFFBQVE7Z0JBQ2xCLHdEQUF3RDtnQkFDeEQsaUJBQWlCLEVBQUcsaUJBQTREO2dCQUNoRixRQUFRLEVBQUUsYUFBYTtnQkFDdkIsWUFBWSxFQUFFLFlBQVksQ0FBQyxRQUFRLEVBQUU7YUFDdEMsQ0FBQztZQUNGLG9CQUFvQixDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQ2pELENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxxQkFBcUIsR0FBMEI7WUFDbkQsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM1QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPO1lBQ3JDLG9CQUFvQixFQUFFLG9CQUFvQjtZQUMxQyx5QkFBeUIsRUFBRSxJQUFJO1lBQy9CLHFGQUFxRjtZQUNyRiwwREFBMEQ7WUFDMUQsY0FBYyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQ2pDLGdDQUFnQyxFQUFFLElBQUk7WUFDdEMsZ0ZBQWdGO1lBQ2hGLElBQUksRUFBRSxrREFBb0IsQ0FBQyxRQUFRO1lBQ25DLFVBQVUsUUFBRSxLQUFLLENBQUMsVUFBVSwwQ0FBRSxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ2hELHdEQUF3RDtZQUN4RCxpQkFBaUIsRUFBRyx5QkFBb0U7U0FDekYsQ0FBQztRQUVGLE1BQU0sOEJBQThCLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdEUsTUFBTSw2QkFBNkIsR0FBa0M7Z0JBQ25FLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDLEVBQUUscUJBQXFCO2FBQzdDLENBQUM7WUFDRixPQUFPLDZCQUE2QixDQUFDO1FBQ3ZDLENBQUMsQ0FBQyxDQUFDO1FBRUgsT0FBTyw4QkFBOEIsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLDZCQUE2QixDQUFDLFlBQTJCO1FBQy9ELE9BQU8sWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUFFLE1BQU0sRUFBRSxjQUFjLEVBQUUsRUFBRSxFQUFFO1lBQ2pFLE1BQU0sRUFBRSxXQUFXLEVBQUUsU0FBUyxFQUFFLEdBQUcsRUFBRSxHQUFHLE1BQU0sQ0FBQztZQUUvQyxJQUFJLE1BQU0sS0FBSyxtQ0FBaUIsQ0FBQyxVQUFVLElBQUksY0FBYyxLQUFLLEtBQUssRUFBRTtnQkFDdkUsT0FBTztvQkFDTCxVQUFVLEVBQUUsVUFBVTtvQkFDdEIsNkVBQTZFO29CQUM3RSxtRkFBbUY7b0JBQ25GLFFBQVEsRUFBRSxFQUFFO2lCQUNiLENBQUM7YUFDSDtZQUVELElBQUksR0FBb0MsQ0FBQztZQUV6QyxJQUFJLEdBQUcsRUFBRTtnQkFDUCxNQUFNLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsVUFBVSxFQUFFLG1CQUFtQixFQUFFLEdBQUcsR0FBRyxDQUFDO2dCQUU5RSxHQUFHLEdBQUc7b0JBQ0osbUJBQW1CLEVBQUUsbUJBQW1CO29CQUN4QyxJQUFJLEVBQUUsSUFBSTtvQkFDVixVQUFVLEVBQUUsVUFBVTtvQkFDdEIsVUFBVSxFQUFFLFVBQVU7b0JBQ3RCLFVBQVUsRUFBRSxVQUFVO29CQUN0QixxRkFBcUY7b0JBQ3JGLHdDQUF3QztvQkFDeEMsU0FBUyxFQUFFLFdBQVcsSUFBSSxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztpQkFDN0QsQ0FBQzthQUNIO1lBRUQsT0FBTztnQkFDTCxVQUFVLEVBQUUsVUFBVTtnQkFDdEIsR0FBRztnQkFDSCxXQUFXLEVBQUUsV0FBVzthQUN6QixDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sNEJBQTRCLENBQUMsVUFBbUM7UUFDdEUsSUFBSSxDQUFDLFVBQVUsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMxQyxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUVELE1BQU0saUNBQWlDLEdBQWtDLEVBQUUsQ0FBQztRQUM1RSxVQUFVLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sOEJBQThCLEdBQUcsSUFBSSxDQUFDLDhCQUE4QixDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ2xGLDhCQUE4QixDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRTtnQkFDakQsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsYUFBYSxDQUFDLEVBQUU7b0JBQ3hELElBQUksR0FBRyxJQUFJLGlDQUFpQyxFQUFFO3dCQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLG1CQUFtQixHQUFHLHNFQUFzRSxDQUFDLENBQUM7cUJBQy9HO29CQUNELGlDQUFpQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztpQkFDaEQ7WUFDSCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxpQ0FBaUMsQ0FBQztJQUMzQyxDQUFDOztBQXZUSCw0REF3VEM7OztBQXRUQzs7R0FFRztBQUNZLDJDQUFrQixHQUFzQixJQUFJLEdBQUcsRUFBZ0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5cbmltcG9ydCB7XG4gIEJsb2NrRGV2aWNlLFxuICBCbG9ja0RldmljZVZvbHVtZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWF1dG9zY2FsaW5nJztcbmltcG9ydCB7XG4gIElWcGMsXG4gIFN1Ym5ldFNlbGVjdGlvbixcbiAgU3VibmV0VHlwZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQge1xuICBSb2xlLFxuICBQb2xpY3ksXG4gIFBvbGljeVN0YXRlbWVudCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQge1xuICBDb2RlLFxuICBGdW5jdGlvbiBhcyBMYW1iZGFGdW5jdGlvbixcbiAgUnVudGltZSxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBSZXRlbnRpb25EYXlzIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0IHtcbiAgQW5ub3RhdGlvbnMsXG4gIENvbnN0cnVjdCxcbiAgQ3VzdG9tUmVzb3VyY2UsXG4gIER1cmF0aW9uLFxuICBGbixcbiAgSVJlc29sdmFibGUsXG4gIExhenksXG4gIFN0YWNrLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgUGx1Z2luU2V0dGluZ3MsXG4gIFNFUENvbmZpZ3VyYXRvclJlc291cmNlUHJvcHMsXG4gIExhdW5jaFNwZWNpZmljYXRpb24sXG4gIFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uLFxuICBTcG90RmxlZXRSZXF1ZXN0UHJvcHMsXG4gIFNwb3RGbGVldFNlY3VyaXR5R3JvdXBJZCxcbiAgU3BvdEZsZWV0VGFnU3BlY2lmaWNhdGlvbixcbiAgQmxvY2tEZXZpY2VNYXBwaW5nUHJvcGVydHksXG4gIEJsb2NrRGV2aWNlUHJvcGVydHksXG59IGZyb20gJy4uLy4uL2xhbWJkYXMvbm9kZWpzL2NvbmZpZ3VyZS1zcG90LWV2ZW50LXBsdWdpbic7XG5pbXBvcnQge1xuICBJUmVuZGVyUXVldWUsXG4gIFJlbmRlclF1ZXVlLFxufSBmcm9tICcuL3JlbmRlci1xdWV1ZSc7XG5pbXBvcnQge1xuICBTZWNyZXRzTWFuYWdlbWVudFJlZ2lzdHJhdGlvblN0YXR1cyxcbiAgU2VjcmV0c01hbmFnZW1lbnRSb2xlLFxufSBmcm9tICcuL3NlY3JldHMtbWFuYWdlbWVudC1yZWYnO1xuaW1wb3J0IHsgU3BvdEV2ZW50UGx1Z2luRmxlZXQgfSBmcm9tICcuL3Nwb3QtZXZlbnQtcGx1Z2luLWZsZWV0JztcbmltcG9ydCB7XG4gIFNwb3RGbGVldFJlcXVlc3RUeXBlLFxuICBTcG90RmxlZXRSZXNvdXJjZVR5cGUsXG59IGZyb20gJy4vc3BvdC1ldmVudC1wbHVnaW4tZmxlZXQtcmVmJztcbmltcG9ydCB7IFZlcnNpb24gfSBmcm9tICcuL3ZlcnNpb24nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5TdGF0ZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgR0xPQkFMX0VOQUJMRUQgPSAnR2xvYmFsIEVuYWJsZWQnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBESVNBQkxFRCA9ICdEaXNhYmxlZCcsXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5Mb2dnaW5nTGV2ZWwge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBTVEFOREFSRCA9ICdTdGFuZGFyZCcsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBWRVJCT1NFID0gJ1ZlcmJvc2UnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgREVCVUcgPSAnRGVidWcnLFxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgT0ZGID0gJ09mZicsXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIFNwb3RFdmVudFBsdWdpblByZUpvYlRhc2tNb2RlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBDT05TRVJWQVRJVkUgPSAnQ29uc2VydmF0aXZlJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgSUdOT1JFID0gJ0lnbm9yZScsXG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIE5PUk1BTCA9ICdOb3JtYWwnLFxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgZW51bSBTcG90RXZlbnRQbHVnaW5EaXNwbGF5SW5zdGFuY2VTdGF0dXMge1xuICBESVNBQkxFRCA9ICdEaXNhYmxlZCcsXG4gIEVYVFJBX0lORk9fMCA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT18xID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzIgPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fMyA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT180ID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzUgPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fNiA9ICdFeHRyYUluZm8wJyxcbiAgRVhUUkFfSU5GT183ID0gJ0V4dHJhSW5mbzAnLFxuICBFWFRSQV9JTkZPXzggPSAnRXh0cmFJbmZvMCcsXG4gIEVYVFJBX0lORk9fOSA9ICdFeHRyYUluZm8wJyxcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNwb3RFdmVudFBsdWdpblNldHRpbmdzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN0YXRlPzogU3BvdEV2ZW50UGx1Z2luU3RhdGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbmFibGVSZXNvdXJjZVRyYWNrZXI/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGxvZ2dpbmdMZXZlbD86IFNwb3RFdmVudFBsdWdpbkxvZ2dpbmdMZXZlbDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGlkbGVTaHV0ZG93bj86IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlbGV0ZVNFUFRlcm1pbmF0ZWRXb3JrZXJzPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVsZXRlRUMyU3BvdEludGVycnVwdGVkV29ya2Vycz86IGJvb2xlYW47XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN0cmljdEhhcmRDYXA/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWF4aW11bUluc3RhbmNlc1N0YXJ0ZWRQZXJDeWNsZT86IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmVKb2JUYXNrTW9kZT86IFNwb3RFdmVudFBsdWdpblByZUpvYlRhc2tNb2RlO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGF3c0luc3RhbmNlU3RhdHVzPzogU3BvdEV2ZW50UGx1Z2luRGlzcGxheUluc3RhbmNlU3RhdHVzO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBjOiBJVnBjO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVuZGVyUXVldWU6IElSZW5kZXJRdWV1ZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGNTdWJuZXRzPzogU3VibmV0U2VsZWN0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHNwb3RGbGVldHM/OiBTcG90RXZlbnRQbHVnaW5GbGVldFtdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbmZpZ3VyYXRpb24/OiBTcG90RXZlbnRQbHVnaW5TZXR0aW5ncztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDb25maWd1cmVTcG90RXZlbnRQbHVnaW4gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8qKlxuICAgKiBPbmx5IG9uZSBTcG90IEV2ZW50IFBsdWdpbiBDb25maWd1cmF0aW9uIGlzIGFsbG93ZWQgcGVyIHJlbmRlciBxdWV1ZSAvIHJlcG9zaXRvcnkuXG4gICAqL1xuICBwcml2YXRlIHN0YXRpYyB1bmlxdWVSZW5kZXJRdWV1ZXM6IFNldDxJUmVuZGVyUXVldWU+ID0gbmV3IFNldDxJUmVuZGVyUXVldWU+KCk7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbmZpZ3VyZVNwb3RFdmVudFBsdWdpblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChDb25maWd1cmVTcG90RXZlbnRQbHVnaW4udW5pcXVlUmVuZGVyUXVldWVzLmhhcyhwcm9wcy5yZW5kZXJRdWV1ZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBvbmUgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luIGNvbnN0cnVjdCBpcyBhbGxvd2VkIHBlciByZW5kZXIgcXVldWUuJyk7XG4gICAgfVxuICAgIGVsc2Uge1xuICAgICAgQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luLnVuaXF1ZVJlbmRlclF1ZXVlcy5hZGQocHJvcHMucmVuZGVyUXVldWUpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5yZW5kZXJRdWV1ZSBpbnN0YW5jZW9mIFJlbmRlclF1ZXVlKSB7XG4gICAgICAvLyBXZSBkbyBub3QgY2hlY2sgdGhlIHBhdGNoIHZlcnNpb24sIHNvIGl0J3Mgc2V0IHRvIDAuXG4gICAgICBjb25zdCBtaW5pbXVtVmVyc2lvbjogVmVyc2lvbiA9IG5ldyBWZXJzaW9uKFsxMCwgMSwgMTIsIDBdKTtcblxuICAgICAgaWYgKHByb3BzLnJlbmRlclF1ZXVlLnZlcnNpb24uaXNMZXNzVGhhbihtaW5pbXVtVmVyc2lvbikpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBNaW5pbXVtIHN1cHBvcnRlZCBEZWFkbGluZSB2ZXJzaW9uIGZvciAke3RoaXMuY29uc3RydWN0b3IubmFtZX0gaXMgYCArXG4gICAgICAgIGAke21pbmltdW1WZXJzaW9uLnZlcnNpb25TdHJpbmd9LiBgICtcbiAgICAgICAgYFJlY2VpdmVkOiAke3Byb3BzLnJlbmRlclF1ZXVlLnZlcnNpb24udmVyc2lvblN0cmluZ30uYCk7XG4gICAgICB9XG5cbiAgICAgIGlmIChwcm9wcy5zcG90RmxlZXRzICYmIHByb3BzLnNwb3RGbGVldHMubGVuZ3RoICE9PSAwKSB7XG4gICAgICAgIC8vIEFsd2F5cyBhZGQgUmVzb3VyY2UgVHJhY2tlciBhZG1pbiBwb2xpY3ksIGV2ZW4gaWYgcHJvcHMuY29uZmlndXJhdGlvbj8uZW5hYmxlUmVzb3VyY2VUcmFja2VyIGlzIGZhbHNlLlxuICAgICAgICAvLyBUaGlzIGltcHJvdmVzIHVzYWJpbGl0eSwgYXMgY3VzdG9tZXJzIHdvbid0IG5lZWQgdG8gYWRkIHRoaXMgcG9saWN5IG1hbnVhbGx5LCBpZiB0aGV5XG4gICAgICAgIC8vIGVuYWJsZSBSZXNvdXJjZSBUcmFja2VyIGxhdGVyIGluIHRoZSBTcG90IEV2ZW50IFBsdWdpbiBjb25maWd1cmF0aW9uIChlLmcuLCB1c2luZyBEZWFkbGluZSBNb25pdG9yIGFuZCBub3QgUkZESykuXG4gICAgICAgIHByb3BzLnJlbmRlclF1ZXVlLmFkZFNFUFBvbGljaWVzKHRydWUpO1xuXG4gICAgICAgIGNvbnN0IGZsZWV0Um9sZXMgPSBwcm9wcy5zcG90RmxlZXRzLm1hcChzZiA9PiBzZi5mbGVldFJvbGUucm9sZUFybik7XG4gICAgICAgIGNvbnN0IGZsZWV0SW5zdGFuY2VSb2xlcyA9IHByb3BzLnNwb3RGbGVldHMubWFwKHNmID0+IHNmLmZsZWV0SW5zdGFuY2VSb2xlLnJvbGVBcm4pO1xuICAgICAgICBuZXcgUG9saWN5KHRoaXMsICdTcG90RXZlbnRQbHVnaW5Qb2xpY3knLCB7XG4gICAgICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAnaWFtOlBhc3NSb2xlJyxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbLi4uZmxlZXRSb2xlcywgLi4uZmxlZXRJbnN0YW5jZVJvbGVzXSxcbiAgICAgICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgICAgIFN0cmluZ0xpa2U6IHtcbiAgICAgICAgICAgICAgICAgICdpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ2VjMi5hbWF6b25hd3MuY29tJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgICAgICdlYzI6Q3JlYXRlVGFncycsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogWydhcm46YXdzOmVjMjoqOio6c3BvdC1mbGVldC1yZXF1ZXN0LyonXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgcm9sZXM6IFtcbiAgICAgICAgICAgIHByb3BzLnJlbmRlclF1ZXVlLmdyYW50UHJpbmNpcGFsIGFzIFJvbGUsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICAgIGVsc2Uge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgcHJvdmlkZWQgcmVuZGVyIHF1ZXVlIGlzIG5vdCBhbiBpbnN0YW5jZSBvZiBSZW5kZXJRdWV1ZSBjbGFzcy4gU29tZSBmdW5jdGlvbmFsaXR5IGlzIG5vdCBzdXBwb3J0ZWQuJyk7XG4gICAgfVxuXG4gICAgY29uc3QgcmVnaW9uID0gQ29uc3RydWN0LmlzQ29uc3RydWN0KHByb3BzLnJlbmRlclF1ZXVlKSA/IFN0YWNrLm9mKHByb3BzLnJlbmRlclF1ZXVlKS5yZWdpb24gOiBTdGFjay5vZih0aGlzKS5yZWdpb247XG5cbiAgICBjb25zdCB0aW1lb3V0TWlucyA9IDE1O1xuICAgIGNvbnN0IGNvbmZpZ3VyYXRvciA9IG5ldyBMYW1iZGFGdW5jdGlvbih0aGlzLCAnQ29uZmlndXJhdG9yJywge1xuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy52cGNTdWJuZXRzID8/IHsgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QUklWQVRFIH0sXG4gICAgICBkZXNjcmlwdGlvbjogYFVzZWQgYnkgYSBDb25maWd1cmVTcG90RXZlbnRQbHVnaW4gJHt0aGlzLm5vZGUuYWRkcn0gdG8gcGVyZm9ybSBjb25maWd1cmF0aW9uIG9mIERlYWRsaW5lIFNwb3QgRXZlbnQgUGx1Z2luYCxcbiAgICAgIGNvZGU6IENvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdsYW1iZGFzJywgJ25vZGVqcycpLCB7XG4gICAgICB9KSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIERFQlVHOiAnZmFsc2UnLFxuICAgICAgICBMQU1CREFfVElNRU9VVF9NSU5TOiB0aW1lb3V0TWlucy50b1N0cmluZygpLFxuICAgICAgfSxcbiAgICAgIHJ1bnRpbWU6IFJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICBoYW5kbGVyOiAnY29uZmlndXJlLXNwb3QtZXZlbnQtcGx1Z2luLmNvbmZpZ3VyZVNFUCcsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKHRpbWVvdXRNaW5zKSxcbiAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICB9KTtcblxuICAgIGNvbmZpZ3VyYXRvci5jb25uZWN0aW9ucy5hbGxvd1RvRGVmYXVsdFBvcnQocHJvcHMucmVuZGVyUXVldWUpO1xuICAgIHByb3BzLnJlbmRlclF1ZXVlLmNlcnRDaGFpbj8uZ3JhbnRSZWFkKGNvbmZpZ3VyYXRvci5ncmFudFByaW5jaXBhbCk7XG5cbiAgICBjb25zdCBwbHVnaW5Db25maWc6IFBsdWdpblNldHRpbmdzID0ge1xuICAgICAgQVdTSW5zdGFuY2VTdGF0dXM6IHByb3BzLmNvbmZpZ3VyYXRpb24/LmF3c0luc3RhbmNlU3RhdHVzID8/IFNwb3RFdmVudFBsdWdpbkRpc3BsYXlJbnN0YW5jZVN0YXR1cy5ESVNBQkxFRCxcbiAgICAgIERlbGV0ZUludGVycnVwdGVkU2xhdmVzOiBwcm9wcy5jb25maWd1cmF0aW9uPy5kZWxldGVFQzJTcG90SW50ZXJydXB0ZWRXb3JrZXJzID8/IGZhbHNlLFxuICAgICAgRGVsZXRlVGVybWluYXRlZFNsYXZlczogcHJvcHMuY29uZmlndXJhdGlvbj8uZGVsZXRlU0VQVGVybWluYXRlZFdvcmtlcnMgPz8gZmFsc2UsXG4gICAgICBJZGxlU2h1dGRvd246IHByb3BzLmNvbmZpZ3VyYXRpb24/LmlkbGVTaHV0ZG93bj8udG9NaW51dGVzKHtpbnRlZ3JhbDogdHJ1ZX0pID8/IDEwLFxuICAgICAgTG9nZ2luZzogcHJvcHMuY29uZmlndXJhdGlvbj8ubG9nZ2luZ0xldmVsID8/IFNwb3RFdmVudFBsdWdpbkxvZ2dpbmdMZXZlbC5TVEFOREFSRCxcbiAgICAgIFByZUpvYlRhc2tNb2RlOiBwcm9wcy5jb25maWd1cmF0aW9uPy5wcmVKb2JUYXNrTW9kZSA/PyBTcG90RXZlbnRQbHVnaW5QcmVKb2JUYXNrTW9kZS5DT05TRVJWQVRJVkUsXG4gICAgICBSZWdpb246IHByb3BzLmNvbmZpZ3VyYXRpb24/LnJlZ2lvbiA/PyByZWdpb24sXG4gICAgICBSZXNvdXJjZVRyYWNrZXI6IHByb3BzLmNvbmZpZ3VyYXRpb24/LmVuYWJsZVJlc291cmNlVHJhY2tlciA/PyB0cnVlLFxuICAgICAgU3RhZ2dlckluc3RhbmNlczogcHJvcHMuY29uZmlndXJhdGlvbj8ubWF4aW11bUluc3RhbmNlc1N0YXJ0ZWRQZXJDeWNsZSA/PyA1MCxcbiAgICAgIFN0YXRlOiBwcm9wcy5jb25maWd1cmF0aW9uPy5zdGF0ZSA/PyBTcG90RXZlbnRQbHVnaW5TdGF0ZS5HTE9CQUxfRU5BQkxFRCxcbiAgICAgIFN0cmljdEhhcmRDYXA6IHByb3BzLmNvbmZpZ3VyYXRpb24/LnN0cmljdEhhcmRDYXAgPz8gZmFsc2UsXG4gICAgfTtcbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0Q29uZmlncyA9IHRoaXMubWVyZ2VTcG90RmxlZXRSZXF1ZXN0Q29uZmlncyhwcm9wcy5zcG90RmxlZXRzKTtcblxuICAgIGNvbnN0IGRlYWRsaW5lR3JvdXBzID0gQXJyYXkuZnJvbShuZXcgU2V0KHByb3BzLnNwb3RGbGVldHM/Lm1hcChmbGVldCA9PiBmbGVldC5kZWFkbGluZUdyb3VwcykucmVkdWNlKChwLCBjKSA9PiBwLmNvbmNhdChjKSwgW10pKSk7XG4gICAgY29uc3QgZGVhZGxpbmVQb29scyA9IEFycmF5LmZyb20obmV3IFNldChwcm9wcy5zcG90RmxlZXRzPy5tYXAoZmxlZXQgPT4gZmxlZXQuZGVhZGxpbmVQb29scykucmVkdWNlKChwLCBjKSA9PiBwPy5jb25jYXQoYyA/PyBbXSksIFtdKSkpO1xuICAgIGNvbnN0IHByb3BlcnRpZXM6IFNFUENvbmZpZ3VyYXRvclJlc291cmNlUHJvcHMgPSB7XG4gICAgICBjb25uZWN0aW9uOiB7XG4gICAgICAgIGhvc3RuYW1lOiBwcm9wcy5yZW5kZXJRdWV1ZS5lbmRwb2ludC5ob3N0bmFtZSxcbiAgICAgICAgcG9ydDogcHJvcHMucmVuZGVyUXVldWUuZW5kcG9pbnQucG9ydEFzU3RyaW5nKCksXG4gICAgICAgIHByb3RvY29sOiBwcm9wcy5yZW5kZXJRdWV1ZS5lbmRwb2ludC5hcHBsaWNhdGlvblByb3RvY29sLFxuICAgICAgICBjYUNlcnRpZmljYXRlQXJuOiBwcm9wcy5yZW5kZXJRdWV1ZS5jZXJ0Q2hhaW4/LnNlY3JldEFybixcbiAgICAgIH0sXG4gICAgICBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnM6IHNwb3RGbGVldFJlcXVlc3RDb25maWdzLFxuICAgICAgc3BvdFBsdWdpbkNvbmZpZ3VyYXRpb25zOiBwbHVnaW5Db25maWcsXG4gICAgICBkZWFkbGluZUdyb3VwcyxcbiAgICAgIGRlYWRsaW5lUG9vbHMsXG4gICAgfTtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdEZWZhdWx0Jywge1xuICAgICAgc2VydmljZVRva2VuOiBjb25maWd1cmF0b3IuZnVuY3Rpb25Bcm4sXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OlJGREtfQ29uZmlndXJlU3BvdEV2ZW50UGx1Z2luJyxcbiAgICAgIHByb3BlcnRpZXMsXG4gICAgfSk7XG5cbiAgICAvLyBQcmV2ZW50cyBhIHJhY2UgZHVyaW5nIGEgc3RhY2stdXBkYXRlLlxuICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShjb25maWd1cmF0b3Iucm9sZSEpO1xuXG4gICAgLy8gV2UgbmVlZCB0byBhZGQgdGhpcyBkZXBlbmRlbmN5IHRvIGF2b2lkIGZhaWx1cmVzIHdoaWxlIGRlbGV0aW5nIGEgQ3VzdG9tIFJlc291cmNlOlxuICAgIC8vICdDdXN0b20gUmVzb3VyY2UgZmFpbGVkIHRvIHN0YWJpbGl6ZSBpbiBleHBlY3RlZCB0aW1lLiBJZiB5b3UgYXJlIHVzaW5nIHRoZSBQeXRob24gY2ZuLXJlc3BvbnNlIG1vZHVsZSxcbiAgICAvLyB5b3UgbWF5IG5lZWQgdG8gdXBkYXRlIHlvdXIgTGFtYmRhIGZ1bmN0aW9uIGNvZGUgc28gdGhhdCBDbG91ZEZvcm1hdGlvbiBjYW4gYXR0YWNoIHRoZSB1cGRhdGVkIHZlcnNpb24uJy5cbiAgICAvLyBUaGlzIGhhcHBlbnMsIGJlY2F1c2UgUm91dGUgVGFibGUgQXNzb2NpYXRpb25zIGFyZSBkZWxldGVkIGJlZm9yZSB0aGUgQ3VzdG9tIFJlc291cmNlIGFuZCB3ZVxuICAgIC8vIGRvbid0IGdldCBhIHJlc3BvbnNlIGZyb20gJ2RvRGVsZXRlKCknLlxuICAgIC8vIElkZWFsbHksIHdlIHdvdWxkIG9ubHkgd2FudCB0byBhZGQgZGVwZW5kZW5jeSBvbiAnaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCcgYXMgc2hvd24gYmVsb3csXG4gICAgLy8gYnV0IGl0IHNlZW1zIHRoYXQgQ0RLIG1pc3NlcyBkZXBlbmRlbmNpZXMgb24gUm91dGUgVGFibGUgQXNzb2NpYXRpb25zIGluIHRoYXQgY2FzZTpcbiAgICAvLyBjb25zdCB7IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQgfSA9IHByb3BzLnZwYy5zZWxlY3RTdWJuZXRzKHByb3BzLnZwY1N1Ym5ldHMpO1xuICAgIC8vIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeShpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkKTtcbiAgICByZXNvdXJjZS5ub2RlLmFkZERlcGVuZGVuY3kocHJvcHMudnBjKTtcblxuICAgIC8vIC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovXG4gICAgLy8gQWRkIGEgZGVwZW5kZW5jeSBvbiB0aGUgcmVuZGVyIHF1ZXVlIHRvIGVuc3VyZSB0aGF0XG4gICAgLy8gaXQgaXMgcnVubmluZyBiZWZvcmUgd2UgdHJ5IHRvIHNlbmQgcmVxdWVzdHMgdG8gaXQuXG4gICAgcmVzb3VyY2Uubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLnJlbmRlclF1ZXVlKTtcblxuICAgIGlmIChwcm9wcy5zcG90RmxlZXRzICYmIHByb3BzLnJlbmRlclF1ZXVlLnJlcG9zaXRvcnkuc2VjcmV0c01hbmFnZW1lbnRTZXR0aW5ncy5lbmFibGVkKSB7XG4gICAgICBwcm9wcy5zcG90RmxlZXRzLmZvckVhY2goc3BvdEZsZWV0ID0+IHtcbiAgICAgICAgaWYgKHNwb3RGbGVldC5kZWZhdWx0U3VibmV0cykge1xuICAgICAgICAgIEFubm90YXRpb25zLm9mKHNwb3RGbGVldCkuYWRkV2FybmluZyhcbiAgICAgICAgICAgICdEZWFkbGluZSBTZWNyZXRzIE1hbmFnZW1lbnQgaXMgZW5hYmxlZCBvbiB0aGUgUmVwb3NpdG9yeSBhbmQgVlBDIHN1Ym5ldHMgaGF2ZSBub3QgYmVlbiBzdXBwbGllZC4gVXNpbmcgZGVkaWNhdGVkIHN1Ym5ldHMgaXMgcmVjb21tZW5kZWQuIFNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1yZmRrL2Jsb2JzL3JlbGVhc2UvcGFja2FnZXMvYXdzLXJmZGsvbGliL2RlYWRsaW5lL1JFQURNRS5tZCN1c2luZy1kZWRpY2F0ZWQtc3VibmV0cy1mb3ItZGVhZGxpbmUtY29tcG9uZW50cycsXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgICBwcm9wcy5yZW5kZXJRdWV1ZS5jb25maWd1cmVTZWNyZXRzTWFuYWdlbWVudEF1dG9SZWdpc3RyYXRpb24oe1xuICAgICAgICAgIGRlcGVuZGVudDogcmVzb3VyY2UsXG4gICAgICAgICAgcm9sZTogU2VjcmV0c01hbmFnZW1lbnRSb2xlLkNMSUVOVCxcbiAgICAgICAgICByZWdpc3RyYXRpb25TdGF0dXM6IFNlY3JldHNNYW5hZ2VtZW50UmVnaXN0cmF0aW9uU3RhdHVzLlJFR0lTVEVSRUQsXG4gICAgICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICAgICAgdnBjU3VibmV0czogc3BvdEZsZWV0LnN1Ym5ldHMsXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfVxuXG4gICAgdGhpcy5ub2RlLmRlZmF1bHRDaGlsZCA9IHJlc291cmNlO1xuICB9XG5cbiAgcHJpdmF0ZSB0YWdTcGVjaWZpY2F0aW9ucyhmbGVldDogU3BvdEV2ZW50UGx1Z2luRmxlZXQsIHJlc291cmNlVHlwZTogU3BvdEZsZWV0UmVzb3VyY2VUeXBlKTogSVJlc29sdmFibGUge1xuICAgIHJldHVybiBMYXp5LmFueSh7XG4gICAgICBwcm9kdWNlOiAoKSA9PiB7XG4gICAgICAgIGlmIChmbGVldC50YWdzLmhhc1RhZ3MoKSkge1xuICAgICAgICAgIGNvbnN0IHRhZ1NwZWNpZmljYXRpb246IFNwb3RGbGVldFRhZ1NwZWNpZmljYXRpb24gPSB7XG4gICAgICAgICAgICBSZXNvdXJjZVR5cGU6IHJlc291cmNlVHlwZSxcbiAgICAgICAgICAgIFRhZ3M6IGZsZWV0LnRhZ3MucmVuZGVyVGFncygpLFxuICAgICAgICAgIH07XG4gICAgICAgICAgcmV0dXJuIFt0YWdTcGVjaWZpY2F0aW9uXTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgU3BvdCBGbGVldCBDb25maWd1cmF0aW9ucyBmcm9tIHRoZSBwcm92aWRlZCBmbGVldC5cbiAgICogRWFjaCBjb25naWd1cmF0aW9uIGlzIGEgbWFwcGluZyBiZXR3ZWVuIG9uZSBEZWFkbGluZSBHcm91cCBhbmQgb25lIFNwb3QgRmxlZXQgUmVxdWVzdCBDb25maWd1cmF0aW9uLlxuICAgKi9cbiAgcHJpdmF0ZSBnZW5lcmF0ZVNwb3RGbGVldFJlcXVlc3RDb25maWcoZmxlZXQ6IFNwb3RFdmVudFBsdWdpbkZsZWV0KTogU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25bXSB7XG4gICAgY29uc3Qgc2VjdXJpdHlHcm91cHNUb2tlbiA9IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4ge1xuICAgICAgcmV0dXJuIGZsZWV0LnNlY3VyaXR5R3JvdXBzLm1hcChzZyA9PiB7XG4gICAgICAgIGNvbnN0IHNlY3VyaXR5R3JvdXBJZDogU3BvdEZsZWV0U2VjdXJpdHlHcm91cElkID0ge1xuICAgICAgICAgIEdyb3VwSWQ6IHNnLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgICAgfTtcbiAgICAgICAgcmV0dXJuIHNlY3VyaXR5R3JvdXBJZDtcbiAgICAgIH0pO1xuICAgIH19KTtcblxuICAgIGNvbnN0IHVzZXJEYXRhVG9rZW4gPSBMYXp5LnN0cmluZyh7IHByb2R1Y2U6ICgpID0+IEZuLmJhc2U2NChmbGVldC51c2VyRGF0YS5yZW5kZXIoKSkgfSk7XG5cbiAgICBjb25zdCBibG9ja0RldmljZU1hcHBpbmdzID0gKGZsZWV0LmJsb2NrRGV2aWNlcyAhPT0gdW5kZWZpbmVkID9cbiAgICAgIHRoaXMuc3ludGhlc2l6ZUJsb2NrRGV2aWNlTWFwcGluZ3MoZmxlZXQuYmxvY2tEZXZpY2VzKSA6IHVuZGVmaW5lZCk7XG5cbiAgICBjb25zdCB7IHN1Ym5ldElkcyB9ID0gZmxlZXQuc3VibmV0cztcbiAgICBjb25zdCBzdWJuZXRJZCA9IHN1Ym5ldElkcy5qb2luKCcsJyk7XG5cbiAgICBjb25zdCBpbnN0YW5jZVRhZ3NUb2tlbiA9IHRoaXMudGFnU3BlY2lmaWNhdGlvbnMoZmxlZXQsIFNwb3RGbGVldFJlc291cmNlVHlwZS5JTlNUQU5DRSk7XG4gICAgY29uc3Qgc3BvdEZsZWV0UmVxdWVzdFRhZ3NUb2tlbiA9IHRoaXMudGFnU3BlY2lmaWNhdGlvbnMoZmxlZXQsIFNwb3RGbGVldFJlc291cmNlVHlwZS5TUE9UX0ZMRUVUX1JFUVVFU1QpO1xuXG4gICAgY29uc3QgbGF1bmNoU3BlY2lmaWNhdGlvbnM6IExhdW5jaFNwZWNpZmljYXRpb25bXSA9IFtdO1xuXG4gICAgZmxlZXQuaW5zdGFuY2VUeXBlcy5tYXAoaW5zdGFuY2VUeXBlID0+IHtcbiAgICAgIGNvbnN0IGxhdW5jaFNwZWNpZmljYXRpb246IExhdW5jaFNwZWNpZmljYXRpb24gPSB7XG4gICAgICAgIEJsb2NrRGV2aWNlTWFwcGluZ3M6IGJsb2NrRGV2aWNlTWFwcGluZ3MsXG4gICAgICAgIElhbUluc3RhbmNlUHJvZmlsZToge1xuICAgICAgICAgIEFybjogZmxlZXQuaW5zdGFuY2VQcm9maWxlLmF0dHJBcm4sXG4gICAgICAgIH0sXG4gICAgICAgIEltYWdlSWQ6IGZsZWV0LmltYWdlSWQsXG4gICAgICAgIEtleU5hbWU6IGZsZWV0LmtleU5hbWUsXG4gICAgICAgIC8vIE5lZWQgdG8gY29udmVydCBmcm9tIElSZXNvbHZhYmxlIHRvIGJ5cGFzcyBUeXBlU2NyaXB0XG4gICAgICAgIFNlY3VyaXR5R3JvdXBzOiAoc2VjdXJpdHlHcm91cHNUb2tlbiAgYXMgdW5rbm93bikgYXMgU3BvdEZsZWV0U2VjdXJpdHlHcm91cElkW10sXG4gICAgICAgIFN1Ym5ldElkOiBzdWJuZXRJZCxcbiAgICAgICAgLy8gTmVlZCB0byBjb252ZXJ0IGZyb20gSVJlc29sdmFibGUgdG8gYnlwYXNzIFR5cGVTY3JpcHRcbiAgICAgICAgVGFnU3BlY2lmaWNhdGlvbnM6IChpbnN0YW5jZVRhZ3NUb2tlbiBhcyB1bmtub3duKSBhcyBTcG90RmxlZXRUYWdTcGVjaWZpY2F0aW9uW10sXG4gICAgICAgIFVzZXJEYXRhOiB1c2VyRGF0YVRva2VuLFxuICAgICAgICBJbnN0YW5jZVR5cGU6IGluc3RhbmNlVHlwZS50b1N0cmluZygpLFxuICAgICAgfTtcbiAgICAgIGxhdW5jaFNwZWNpZmljYXRpb25zLnB1c2gobGF1bmNoU3BlY2lmaWNhdGlvbik7XG4gICAgfSk7XG5cbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0UHJvcHM6IFNwb3RGbGVldFJlcXVlc3RQcm9wcyA9IHtcbiAgICAgIEFsbG9jYXRpb25TdHJhdGVneTogZmxlZXQuYWxsb2NhdGlvblN0cmF0ZWd5LFxuICAgICAgSWFtRmxlZXRSb2xlOiBmbGVldC5mbGVldFJvbGUucm9sZUFybixcbiAgICAgIExhdW5jaFNwZWNpZmljYXRpb25zOiBsYXVuY2hTcGVjaWZpY2F0aW9ucyxcbiAgICAgIFJlcGxhY2VVbmhlYWx0aHlJbnN0YW5jZXM6IHRydWUsXG4gICAgICAvLyBJbiBvcmRlciB0byB3b3JrIHdpdGggRGVhZGxpbmUsIHRoZSAnVGFyZ2V0IENhcGFjaXR5JyBvZiB0aGUgU3BvdCBmbGVldCBSZXF1ZXN0IGlzXG4gICAgICAvLyB0aGUgbWF4aW11bSBudW1iZXIgb2YgV29ya2VycyB0aGF0IERlYWRsaW5lIHdpbGwgc3RhcnQuXG4gICAgICBUYXJnZXRDYXBhY2l0eTogZmxlZXQubWF4Q2FwYWNpdHksXG4gICAgICBUZXJtaW5hdGVJbnN0YW5jZXNXaXRoRXhwaXJhdGlvbjogdHJ1ZSxcbiAgICAgIC8vIEluIG9yZGVyIHRvIHdvcmsgd2l0aCBEZWFkbGluZSwgU3BvdCBGbGVldHMgUmVxdWVzdHMgbXVzdCBiZSBzZXQgdG8gbWFpbnRhaW4uXG4gICAgICBUeXBlOiBTcG90RmxlZXRSZXF1ZXN0VHlwZS5NQUlOVEFJTixcbiAgICAgIFZhbGlkVW50aWw6IGZsZWV0LnZhbGlkVW50aWw/LmRhdGUudG9JU09TdHJpbmcoKSxcbiAgICAgIC8vIE5lZWQgdG8gY29udmVydCBmcm9tIElSZXNvbHZhYmxlIHRvIGJ5cGFzcyBUeXBlU2NyaXB0XG4gICAgICBUYWdTcGVjaWZpY2F0aW9uczogKHNwb3RGbGVldFJlcXVlc3RUYWdzVG9rZW4gYXMgdW5rbm93bikgYXMgU3BvdEZsZWV0VGFnU3BlY2lmaWNhdGlvbltdLFxuICAgIH07XG5cbiAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnMgPSBmbGVldC5kZWFkbGluZUdyb3Vwcy5tYXAoZ3JvdXAgPT4ge1xuICAgICAgY29uc3Qgc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb246IFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uID0ge1xuICAgICAgICBbZ3JvdXAudG9Mb3dlckNhc2UoKV06IHNwb3RGbGVldFJlcXVlc3RQcm9wcyxcbiAgICAgIH07XG4gICAgICByZXR1cm4gc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb247XG4gICAgfSk7XG5cbiAgICByZXR1cm4gc3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25zO1xuICB9XG5cbiAgLyoqXG4gICAqIFN5bnRoZXNpemUgYW4gYXJyYXkgb2YgYmxvY2sgZGV2aWNlIG1hcHBpbmdzIGZyb20gYSBsaXN0IG9mIGJsb2NrIGRldmljZXNcbiAgICpcbiAgICogQHBhcmFtIGJsb2NrRGV2aWNlcyBsaXN0IG9mIGJsb2NrIGRldmljZXNcbiAgICovXG4gIHByaXZhdGUgc3ludGhlc2l6ZUJsb2NrRGV2aWNlTWFwcGluZ3MoYmxvY2tEZXZpY2VzOiBCbG9ja0RldmljZVtdKTogQmxvY2tEZXZpY2VNYXBwaW5nUHJvcGVydHlbXSB7XG4gICAgcmV0dXJuIGJsb2NrRGV2aWNlcy5tYXAoKHsgZGV2aWNlTmFtZSwgdm9sdW1lLCBtYXBwaW5nRW5hYmxlZCB9KSA9PiB7XG4gICAgICBjb25zdCB7IHZpcnR1YWxOYW1lLCBlYnNEZXZpY2U6IGVicyB9ID0gdm9sdW1lO1xuXG4gICAgICBpZiAodm9sdW1lID09PSBCbG9ja0RldmljZVZvbHVtZS5fTk9fREVWSUNFIHx8IG1hcHBpbmdFbmFibGVkID09PSBmYWxzZSkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgIERldmljZU5hbWU6IGRldmljZU5hbWUsXG4gICAgICAgICAgLy8gVG8gb21pdCB0aGUgZGV2aWNlIGZyb20gdGhlIGJsb2NrIGRldmljZSBtYXBwaW5nLCBzcGVjaWZ5IGFuIGVtcHR5IHN0cmluZy5cbiAgICAgICAgICAvLyBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NsaS9sYXRlc3QvcmVmZXJlbmNlL2VjMi9yZXF1ZXN0LXNwb3QtZmxlZXQuaHRtbFxuICAgICAgICAgIE5vRGV2aWNlOiAnJyxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgbGV0IEViczogQmxvY2tEZXZpY2VQcm9wZXJ0eSB8IHVuZGVmaW5lZDtcblxuICAgICAgaWYgKGVicykge1xuICAgICAgICBjb25zdCB7IGlvcHMsIHZvbHVtZVR5cGUsIHZvbHVtZVNpemUsIHNuYXBzaG90SWQsIGRlbGV0ZU9uVGVybWluYXRpb24gfSA9IGVicztcblxuICAgICAgICBFYnMgPSB7XG4gICAgICAgICAgRGVsZXRlT25UZXJtaW5hdGlvbjogZGVsZXRlT25UZXJtaW5hdGlvbixcbiAgICAgICAgICBJb3BzOiBpb3BzLFxuICAgICAgICAgIFNuYXBzaG90SWQ6IHNuYXBzaG90SWQsXG4gICAgICAgICAgVm9sdW1lU2l6ZTogdm9sdW1lU2l6ZSxcbiAgICAgICAgICBWb2x1bWVUeXBlOiB2b2x1bWVUeXBlLFxuICAgICAgICAgIC8vIGVuY3J5cHRlZCBpcyBub3QgZXhwb3NlZCBhcyBwYXJ0IG9mIGVic0RldmljZVByb3BzIHNvIHdlIG5lZWQgdG8gYWNjZXNzIGl0IHZpYSBbXS5cbiAgICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgZG90LW5vdGF0aW9uXG4gICAgICAgICAgRW5jcnlwdGVkOiAnZW5jcnlwdGVkJyBpbiBlYnMgPyBlYnNbJ2VuY3J5cHRlZCddIDogdW5kZWZpbmVkLFxuICAgICAgICB9O1xuICAgICAgfVxuXG4gICAgICByZXR1cm4ge1xuICAgICAgICBEZXZpY2VOYW1lOiBkZXZpY2VOYW1lLFxuICAgICAgICBFYnMsXG4gICAgICAgIFZpcnR1YWxOYW1lOiB2aXJ0dWFsTmFtZSxcbiAgICAgIH07XG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIG1lcmdlU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3Moc3BvdEZsZWV0cz86IFNwb3RFdmVudFBsdWdpbkZsZWV0W10pOiBTcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFzcG90RmxlZXRzIHx8IHNwb3RGbGVldHMubGVuZ3RoID09PSAwKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIGNvbnN0IGZ1bGxTcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbjogU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb24gPSB7fTtcbiAgICBzcG90RmxlZXRzLm1hcChmbGVldCA9PiB7XG4gICAgICBjb25zdCBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnMgPSB0aGlzLmdlbmVyYXRlU3BvdEZsZWV0UmVxdWVzdENvbmZpZyhmbGVldCk7XG4gICAgICBzcG90RmxlZXRSZXF1ZXN0Q29uZmlndXJhdGlvbnMubWFwKGNvbmZpZ3VyYXRpb24gPT4ge1xuICAgICAgICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhjb25maWd1cmF0aW9uKSkge1xuICAgICAgICAgIGlmIChrZXkgaW4gZnVsbFNwb3RGbGVldFJlcXVlc3RDb25maWd1cmF0aW9uKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEJhZCBHcm91cCBOYW1lOiAke2tleX0uIEdyb3VwIG5hbWVzIGluIFNwb3QgRmxlZXQgUmVxdWVzdCBDb25maWd1cmF0aW9ucyBzaG91bGQgYmUgdW5pcXVlLmApO1xuICAgICAgICAgIH1cbiAgICAgICAgICBmdWxsU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb25ba2V5XSA9IHZhbHVlO1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHJldHVybiBmdWxsU3BvdEZsZWV0UmVxdWVzdENvbmZpZ3VyYXRpb247XG4gIH1cbn1cbiJdfQ==