/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Stack } from '@aws-cdk/core';
/**
 * Returns a CDK matcher for an expected tag key/value pair for a given Cfn resource type.
 * This is known to support the following resource types:
 *
 * * `AWS::AutoScaling::AutoScalingGroup`
 * * `AWS::EC2::SecurityGroup`
 * * `AWS::IAM::Role`
 * * `AWS::SSM::Parameter`
 *
 * All other resources are assumed to allow passing the following tag properties:
 *
 * ```js
 * {
 *   Tags: [
 *     {
 *       Key: 'key',
 *       Value: 'value',
 *     },
 *     // ...
 *   ]
 * }
 * ```
 */
export declare function resourceTagMatcher(resourceType: string, tagName: string, tagValue: string): {
    Tags: {
        [x: string]: string;
    };
} | {
    Tags: import("@aws-cdk/assert").PropertyMatcher;
};
/**
 * Arguments for the {@link testConstructTags} function
 */
interface TestConstructTagsArgs {
    /**
     * The class name of the construct that should be tested for tagging RFDK meta-data.
     */
    readonly constructName: string;
    /**
     * A callback function called to create an instance of the construct being tested in its own isolated stack.
     * The isolated stack should be returned by the callback function.
     */
    readonly createConstruct: () => Stack;
    /**
     * A mapping of CloudFormation resource types to counts of the number of expected resources that should be tagged
     * with RFDK meta-data.
     *
     * E.g.
     *
     * ```ts
     * {
     *   'AWS::AutoScaling::AutoScalingGroup': 1,
     *   'AWS::EC2::SecurityGroup': 3,
     * }
     * ```
     */
    readonly resourceTypeCounts: {
        [resourceType: string]: number;
    };
}
/**
 * Creates tests for the specified resources created by a L2 (or higher construct) to ensure that the resources it
 * creates are created with the RFDK tagging convention.
 *
 * The convention is to create a tag named "aws-rfdk" whose value follows:
 *
 * <VERSION>:<CONSTRUCT_NAME>
 *
 * @param args Arguments to configure the creation of construct tagging tests
 */
export declare function testConstructTags(args: TestConstructTagsArgs): void;
export {};
