"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const mount_permissions_helper_1 = require("../lib/mount-permissions-helper");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableEFS', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let instance;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new efs.FileSystem(stack, 'EFS', { vpc });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the EFS's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 2049,
            ToPort: 2049,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'InstanceInstanceSecurityGroupF0E2D5BE',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'EFSEfsSecurityGroup56F189CE',
                    'GroupId',
                ],
            },
        }));
        // Make sure we download the mountEfs script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/efs/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false r')));
    });
    describe.each([
        [undefined],
        [lib_1.MountPermissions.READONLY],
        [lib_1.MountPermissions.READWRITE],
    ])('access point with %s access permissions', (mountPermission) => {
        describe.each([
            [
                'unspecified POSIX user',
                {
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved non-root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '1000' },
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '0' },
                    expectedClientRootAccess: true,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '0', gid: '1000' },
                    expectedClientRootAccess: true,
                },
            ],
        ])('%s', (_name, testCase) => {
            // GIVEN
            const { posixUser, expectedClientRootAccess } = testCase;
            const expectedActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mountPermission);
            if (expectedClientRootAccess) {
                expectedActions.push('elasticfilesystem:ClientRootAccess');
            }
            const mountPath = '/mnt/efs/fs1';
            let userData;
            let accessPoint;
            let expectedMountMode;
            beforeEach(() => {
                // GIVEN
                accessPoint = new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser,
                });
                const mount = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                });
                expectedMountMode = (mountPermission === lib_1.MountPermissions.READONLY) ? 'ro' : 'rw';
                // WHEN
                mount.mountToLinuxInstance(instance, {
                    location: mountPath,
                    permissions: mountPermission,
                });
                userData = stack.resolve(instance.userData.render());
            });
            test('userdata specifies access point when mounting', () => {
                // THEN
                expect(userData).toEqual({
                    'Fn::Join': [
                        '',
                        expect.arrayContaining([
                            expect.stringMatching(new RegExp('(\\n|^)bash \\./mountEfs.sh $')),
                            stack.resolve(efsFS.fileSystemId),
                            ` ${mountPath} false ${expectedMountMode},iam,accesspoint=`,
                            stack.resolve(accessPoint.accessPointId),
                            expect.stringMatching(/^\n/),
                        ]),
                    ],
                });
            });
            test('grants IAM access point permissions', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: expectedActions.length === 1 ? expectedActions[0] : expectedActions,
                            Condition: {
                                StringEquals: {
                                    'elasticfilesystem:AccessPointArn': stack.resolve(accessPoint.accessPointArn),
                                },
                            },
                            Effect: 'Allow',
                            Resource: stack.resolve(efsFS.node.defaultChild.attrArn),
                        }),
                        Version: '2012-10-17',
                    }),
                    Roles: assert_1.arrayWith(
                    // The Policy construct micro-optimizes the reference to a role in the same stack using its logical ID
                    stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw,option1,option2')));
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const mount1 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        const mount2 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    describe('resolves mount target using API', () => {
        describe.each([
            ['with access point', () => {
                    return new efs.AccessPoint(stack, 'AccessPoint', {
                        fileSystem: efsFS,
                        posixUser: {
                            gid: '1',
                            uid: '1',
                        },
                    });
                }],
            ['without access point', () => undefined],
        ])('%s', (_, getAccessPoint) => {
            let accessPoint;
            beforeEach(() => {
                // GIVEN
                accessPoint = getAccessPoint();
                const mountable = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                    resolveMountTargetDnsWithApi: true,
                });
                // WHEN
                mountable.mountToLinuxInstance(instance, {
                    location: '/mnt/efs',
                });
            });
            test('grants DescribeMountTargets permission', () => {
                const expectedResources = [
                    stack.resolve(efsFS.node.defaultChild.attrArn),
                ];
                if (accessPoint) {
                    expectedResources.push(stack.resolve(accessPoint === null || accessPoint === void 0 ? void 0 : accessPoint.accessPointArn));
                }
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: 'elasticfilesystem:DescribeMountTargets',
                            Effect: 'Allow',
                            Resource: expectedResources.length == 1 ? expectedResources[0] : expectedResources,
                        }),
                    }),
                    Roles: assert_1.arrayWith(stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    describe('.usesUserPosixPermissions()', () => {
        test('access point with POSIX user returns false', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser: {
                        uid: '1000',
                        gid: '1000',
                    },
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(false);
        });
        test('access point without POSIX user returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint: new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                }),
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
        test.each([
            [
                'AccessPoint.fromAccessPointId(...)',
                (inputStack) => efs.AccessPoint.fromAccessPointId(inputStack, 'AccessPoint', 'accessPointId'),
            ],
            [
                'AccessPoint.fromAccessPointAttributes(...)',
                (inputStack) => {
                    return efs.AccessPoint.fromAccessPointAttributes(inputStack, 'AccessPoint', {
                        accessPointArn: core_1.Arn.format({
                            resource: 'AccessPoint',
                            service: 'efs',
                            resourceName: 'accessPointName',
                        }, inputStack),
                        fileSystem: efsFS,
                    });
                },
            ],
        ])('%s throws error', (_label, accessPointProvider) => {
            // GIVEN
            const accessPoint = accessPointProvider(stack);
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
                accessPoint,
            });
            // WHEN
            function when() {
                mount.usesUserPosixPermissions();
            }
            // THEN
            expect(when).toThrow(/^MountableEfs.usesUserPosixPermissions\(\) only supports efs.AccessPoint instances, got ".*"$/);
        });
        test('no access point returns true', () => {
            // GIVEN
            const mount = new lib_1.MountableEfs(stack, {
                filesystem: efsFS,
            });
            // WHEN
            const usesUserPosixPermissions = mount.usesUserPosixPermissions();
            // THEN
            expect(usesUserPosixPermissions).toEqual(true);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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