"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('MongoDbPostInstall', () => {
    let stack;
    let vpc;
    let mongoDb;
    let pwUser1Arn;
    let pwUser2Arn;
    let pwUser1;
    let pwUser2;
    let x509User1Arn;
    let x509User2Arn;
    let x509User1;
    let x509User2;
    beforeEach(() => {
        const hostname = 'mongodb';
        const zoneName = 'testZone.internal';
        const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
        const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        mongoDb = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        pwUser1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1-abcdef';
        pwUser2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2-abcdef';
        pwUser1 = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'PwUser1', pwUser1Arn);
        pwUser2 = aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'PwUser2', pwUser2Arn);
        x509User1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User1-abcdef';
        x509User2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User2-abcdef';
        x509User1 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'x509User1', x509User1Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb1' }]),
        };
        x509User2 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretCompleteArn(stack, 'x509User2', x509User2Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb2' }]),
        };
    });
    test('created correctly: both user types', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs12.x',
            VpcConfig: {
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'MongoPostInstallLambdaSecurityGroup62729E3A',
                            'GroupId',
                        ],
                    },
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                    {
                        Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                    },
                ],
            },
        }));
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser1Arn,
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser2Arn,
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_MongoDbPostInstallSetup', {
            Properties: {
                Connection: {
                    Hostname: 'mongodb.testZone.internal',
                    Port: '27017',
                    CaCertificate: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                    Credentials: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                },
                PasswordAuthUsers: [
                    pwUser1Arn,
                    pwUser2Arn,
                ],
                X509AuthUsers: [
                    {
                        Certificate: x509User1Arn,
                        Roles: x509User1.roles,
                    },
                    {
                        Certificate: x509User2Arn,
                        Roles: x509User2.roles,
                    },
                ],
            },
            DependsOn: [
                'MongoDbInstanceServerAsgASG2643AD1D',
                'MongoPostInstallLambdaServiceRoleDefaultPolicy8B1C1CE8',
                'MongoPostInstallLambdaServiceRoleCD03B9B9',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('created correctly: only password users', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser1Arn,
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: pwUser2Arn,
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            PasswordAuthUsers: [
                pwUser1Arn,
                pwUser2Arn,
            ],
        }));
    });
    test('created correctly: only x509 users', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            X509AuthUsers: [
                {
                    Certificate: x509User1Arn,
                    Roles: x509User1.roles,
                },
                {
                    Certificate: x509User2Arn,
                    Roles: x509User2.roles,
                },
            ],
        }));
    });
    test('use selected subnets', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            VpcConfig: {
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                ],
            },
        }));
    });
    test('assert bad x509 role', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [
                {
                    certificate: x509User1.certificate,
                    roles: '}{',
                },
            ],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
                vpc,
                mongoDb,
                users,
            });
        }).toThrowError(/MongoDbPostInstallSetup: Could not parse JSON role for x509 user:/);
    });
});
//# sourceMappingURL=data:application/json;base64,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