"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
let stack;
let secretCert;
let certPassphraseID;
let importedAcmCertificate;
describe('ImportedAcmCertificate', () => {
    beforeEach(() => {
        // GIVEN
        stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
        secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
            subject: { cn: 'Server' },
        });
        certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
        // WHEN
        importedAcmCertificate = new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
            cert: secretCert.cert,
            certChain: secretCert.certChain,
            key: secretCert.key,
            passphrase: secretCert.passphrase,
        });
    });
    test('creates Custom::RFDK_AcmImportedCertificate', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
            X509CertificatePem: {
                Cert: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Cert',
                    ],
                },
                Key: {
                    'Fn::GetAtt': [
                        'Pem',
                        'Key',
                    ],
                },
                Passphrase: {
                    Ref: certPassphraseID,
                },
                CertChain: '',
            },
            Tags: [
                {
                    Key: 'AcmCertImport-F4E2ABF9',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
                {
                    Key: 'Name',
                    Value: 'f4e2abf974443234fdb095fafcfa9ee2',
                },
            ],
        }));
    });
    describe('creates AWS::DynamoDB::Table for database', () => {
        test('with PhysicalID partition key', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DynamoDB::Table', {
                AttributeDefinitions: assert_1.arrayWith({
                    AttributeName: 'PhysicalId',
                    AttributeType: 'S',
                }),
                KeySchema: assert_1.arrayWith({
                    AttributeName: 'PhysicalId',
                    KeyType: 'HASH',
                }),
            }));
        });
        test('with CustomResource sort key', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DynamoDB::Table', {
                AttributeDefinitions: assert_1.arrayWith({
                    AttributeName: 'CustomResource',
                    AttributeType: 'S',
                }),
                KeySchema: assert_1.arrayWith({
                    AttributeName: 'CustomResource',
                    KeyType: 'RANGE',
                }),
            }));
        });
    });
    test('creates AWS::IAM::Policy', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:ConditionCheckItem',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                    },
                    {
                        Action: 'dynamodb:DescribeTable',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'Pem',
                                'Key',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: certPassphraseID,
                        },
                    },
                    {
                        Action: [
                            'acm:AddTagsToCertificate',
                            'acm:ImportCertificate',
                        ],
                        Condition: {
                            StringEquals: {
                                'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'acm:DeleteCertificate',
                            'acm:DescribeCertificate',
                            'acm:GetCertificate',
                        ],
                        Resource: '*',
                    },
                ],
            },
        }));
    });
    describe('custom resource lambda function', () => {
        test('uses correct handler', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Handler: 'x509-certificate.importCert',
            }));
        });
        test('uses RFDK lambda layer', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Layers: assert_1.arrayWith(assert_1.stringLike('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:*')),
            }));
        });
        test('sets DATABASE environment variable', () => {
            // GIVEN
            const table = importedAcmCertificate.node.findChild('Table');
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Environment: {
                    Variables: assert_1.objectLike({
                        DATABASE: stack.resolve(table.tableName),
                    }),
                },
            }));
        });
    });
    describe('metricDaysToExpiry', () => {
        let metricExpiry;
        beforeEach(() => {
            // GIVEN
            metricExpiry = importedAcmCertificate.metricDaysToExpiry();
        });
        test('uses certificate ARN', () => {
            var _a;
            // THEN
            expect((_a = metricExpiry.dimensions) === null || _a === void 0 ? void 0 : _a.CertificateArn).toEqual(importedAcmCertificate.certificateArn);
        });
        test('uses correct metric', () => {
            // THEN
            expect(metricExpiry.metricName).toEqual('DaysToExpiry');
            expect(metricExpiry.namespace).toEqual('AWS/CertificateManager');
            expect(metricExpiry.statistic).toEqual(aws_cloudwatch_1.Statistic.MINIMUM);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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