"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('CloudWatchAgent', () => {
    let stack;
    let vpc;
    let cloudWatchConfig;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        // Generate CloudWatch Agent configuration JSON
        const configBuilder = new lib_1.CloudWatchConfigBuilder();
        cloudWatchConfig = configBuilder.generateCloudWatchConfiguration();
    });
    test('creates an SSM parameter containing the configuration', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Type: 'String',
            Value: cloudWatchConfig,
        }));
    });
    test('creates an asset', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
        });
        // THEN
        // Find an asset created by the CloudWatchAgentConfigResource
        const asset = stack.node.metadata.find(m => m.type === 'aws:cdk:asset');
        expect(asset).toBeDefined();
    });
    test('creates an IAM policy to access the SSM parameter, CDK asset bucket, and CloudWatch agent bucket', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            'ssm:DescribeParameters',
                            'ssm:GetParameters',
                            'ssm:GetParameter',
                            'ssm:GetParameterHistory',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':ssm:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':parameter/',
                                    { Ref: 'StringParameter472EED0E' },
                                ],
                            ],
                        },
                    },
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::amazoncloudwatch-agent-',
                                        { Ref: 'AWS::Region' },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::amazoncloudwatch-agent-',
                                        { Ref: 'AWS::Region' },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::rfdk-external-dependencies-',
                                        { Ref: 'AWS::Region' },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:aws:s3:::rfdk-external-dependencies-',
                                        { Ref: 'AWS::Region' },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'InstanceInstanceRoleDefaultPolicy4ACE9290',
            Roles: [{ Ref: 'InstanceInstanceRoleE9785DE5' }],
        }));
    });
    test.each([
        ["' -i ", undefined],
        ["' -i ", true],
        ["' ", false],
    ])('adds user data commands to fetch and execute the script (linux). installFlag: %s shouldInstallAgent: %p', (installFlag, shouldInstallAgent) => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
            shouldInstallAgent,
        });
        // THEN
        const userData = stack.resolve(host.userData.render());
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    "')\naws s3 cp 's3://",
                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                    '/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    "' '/tmp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    "'\nset -e\nchmod +x '/tmp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    "'\n'/tmp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                ],
                            },
                        ],
                    },
                    installFlag,
                    { Ref: 'AWS::Region' },
                    ' ',
                    { Ref: 'StringParameter472EED0E' },
                ],
            ],
        });
    });
    test.each([
        ["' -i ", undefined],
        ["' -i ", true],
        ["' ", false],
    ])('adds user data commands to fetch and execute the script (windows). installFlag: %s shouldInstallAgent: %p', (installFlag, shouldInstallAgent) => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Instance', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.WindowsImage(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
            vpc,
        });
        // WHEN
        new lib_1.CloudWatchAgent(stack, 'testResource', {
            cloudWatchConfig,
            host,
            shouldInstallAgent,
        });
        // THEN
        const userData = stack.resolve(host.userData.render());
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    "<powershell>mkdir (Split-Path -Path 'C:/temp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    "' ) -ea 0\nRead-S3Object -BucketName '",
                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Bucket },
                    "' -key '",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    "' -file 'C:/temp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    "' -ErrorAction Stop\n&'C:/temp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    installFlag,
                    { Ref: 'AWS::Region' },
                    ' ',
                    { Ref: 'StringParameter472EED0E' },
                    "\nif (!$?) { Write-Error 'Failed to execute the file \"C:/temp/",
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    "\"' -ErrorAction Stop }</powershell>",
                ],
            ],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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