/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { ISecurityGroup, IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import { IAccessPoint } from '@aws-cdk/aws-efs';
import { Construct, Size } from '@aws-cdk/core';
/**
 * Input properties for PadEfsStorage.
 *
 * @stability stable
 */
export interface PadEfsStorageProps {
    /**
     * VPC in which the given access point is deployed.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * PadEfsStorage deploys AWS Lambda Functions that need to contact your Amazon EFS mount target(s).
     *
     * To do this, AWS Lambda creates network interfaces in these given subnets in your VPC.
     * These can be any subnet(s) in your VPC that can route traffic to the EFS mount target(s).
     *
     * @default All private subnets
     * @stability stable
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Amazon EFS Access Point into which the filesystem padding files will be added.
     *
     * Files will
     * be added/removed from the root directory of the given access point.
     * We strongly recommend that you provide an access point that is for a dedicated padding-files
     * directory in your EFS filesystem, rather than the root directory or some other in-use directory
     * of the filesystem.
     *
     * @stability stable
     */
    readonly accessPoint: IAccessPoint;
    /**
     * Security group for the AWS Lambdas created by this construct.
     *
     * @default Security group with no egress or ingress will be automatically created for each Lambda.
     * @stability stable
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * The desired total size, in GiB, of files stored in the access point directory.
     *
     * @stability stable
     */
    readonly desiredPadding: Size;
}
/**
 * This construct provides a mechanism that adds 1GB-sized files containing only zero-bytes to an Amazon EFS filesystem through a given Access Point to that filesystem.
 *
 * This is being
 * provided to give you a way to increase the baseline throughput of an Amazon EFS filesystem
 * that has been deployed in bursting throughput mode (see: https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes).
 * This is most useful for your Amazon EFS filesystems that contain a very small amount of data and
 * have a baseline throughput that exceeds the throughput provided by the size of the filesystem.
 *
 * When deployed in bursting throughput mode, an Amazon EFS filesystem provides you with a baseline
 * throughput that is proportional to the amount of data stored in that filesystem. However, usage
 * of that filesystem is allowed to burst above that throughput; doing so consumes burst credits that
 * are associated with the filesystem. When all burst credits have been expended, then your filesystem
 * is no longer allowed to burst throughput and you will be limited in throughput to the greater of 1MiB/s
 * or the throughput dictated by the amount of data stored in your filesystem; the filesystem will be able
 * to burst again if it is able to accrue burst credits by staying below its baseline throughput for a time.
 *
 * Customers that deploy the Deadline Repository Filesystem on an Amazon EFS filesystem may find that
 * the filesystem does not contain sufficient data to meet the throughput needs of Deadline; evidenced by
 * a downward trend in EFS bursting credits over time. When bursting credits are expended, then the render
 * farm may begin to exhibit failure mode behaviors such as the RenderQueue dropping or refusing connections,
 * or becoming unresponsive.
 *
 * Warning: The implementation of this construct creates and starts an AWS Step Function to add the files
 * to the filesystem. The execution of this Step Function occurs asynchronously from your deployment. We recommend
 * verifying that the step function completed successfully via your Step Functions console.
 *
 * Resources Deployed
 * --------------------------
 * - Two AWS Lambda Functions, with roles, with full access to the given EFS Access Point.
 * - An Elastic Network Interface (ENI) for each Lambda Function in each of the selected VPC Subnets, so
 *    that the Lambda Functions can connect to the given EFS Access Point.
 * - An AWS Step Function to coordinate execution of the two Lambda Functions.
 * - Security Groups for each AWS Lambda Function.
 * - A CloudFormation custom resource that executes StepFunctions.startExecution on the Step Function
 *    whenever the stack containing this construct is created or updated.
 *
 * Security Considerations
 * ---------------------------
 * - The AWS Lambdas that are deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by these Lambdas.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - By default, the network interfaces created by this construct's AWS Lambda Functions have Security Groups
 *    that restrict egress access from the Lambda Function into your VPC such that the Lambda Functions can
 *    access only the given EFS Access Point.
 *
 * @stability stable
 */
export declare class PadEfsStorage extends Construct {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: PadEfsStorageProps);
}
