/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { IVolume } from '@aws-cdk/aws-ec2';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct, IConstruct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Block format options for formatting a blank/new BlockVolume.
 *
 * @stability stable
 */
export declare enum BlockVolumeFormat {
    /**
     * See: https://en.wikipedia.org/wiki/Ext3.
     *
     * @stability stable
     */
    EXT3 = "ext3",
    /**
     * See: https://en.wikipedia.org/wiki/Ext4.
     *
     * @stability stable
     */
    EXT4 = "ext4",
    /**
     * See: https://en.wikipedia.org/wiki/XFS.
     *
     * @stability stable
     */
    XFS = "xfs"
}
/**
 * Properties that are required to create a {@link MountableBlockVolume}.
 *
 * @stability stable
 */
export interface MountableBlockVolumeProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-ec2.Volume.html|EBS Block Volume} that will be mounted by this object.
     *
     * @stability stable
     */
    readonly blockVolume: IVolume;
    /**
     * The filesystem format of the block volume.
     *
     * @default BlockVolumeFormat.XFS
     * @stability stable
     * @remark If the volume is already formatted, but does not match this format then
     * the mounting script employed by {@link MountableBlockVolume} will mount the volume as-is
     * if it is able. No formatting will be performed.
     */
    readonly volumeFormat?: BlockVolumeFormat;
    /**
     * Extra mount options that will be added to /etc/fstab for the file system.
     *
     * See the Linux man page for mounting the Volume's file system type for information
     * on available options.
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     * @stability stable
     */
    readonly extraMountOptions?: string[];
}
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an Amazon Elastic Block Store (EBS) Volume to itself when it is launched.
 *
 * The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *    **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
export declare class MountableBlockVolume implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableBlockVolumeProps;
    /**
     * @stability stable
     */
    constructor(scope: Construct, props: MountableBlockVolumeProps);
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * Returns whether the mounted file-system evaluates the UID/GID of the system user accessing the file-system.
     *
     * Some network file-systems provide features to fix a UID/GID for all access to the mounted file-system and ignore
     * the system user accessing the file. If this is the case, an implementing class must indicate this in the return
     * value.
     *
     * @stability stable
     * @inheritdoc true
     */
    usesUserPosixPermissions(): boolean;
    /**
     * Grant required permissions to the target.
     *
     * The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     *
     * @stability stable
     */
    protected grantRequiredPermissions(target: IMountingInstance): void;
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    protected mountAssetSingleton(scope: IConstruct): Asset;
}
