"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and executing commands against it.
 *
 * To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the MongoDB administrator credentials,
 *    and the MongoDB application port. An attacker that can find a way to modify and execute this lambda could use it to
 *    modify or read any data in the database. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g;
        super(scope, id);
        (_b = props.users.x509AuthUsers) === null || _b === void 0 ? void 0 : _b.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a MongoDbPostInstallSetup ${core_1.Names.uniqueId(this)} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        (_d = props.users.passwordAuthUsers) === null || _d === void 0 ? void 0 : _d.forEach(secret => secret.grantRead(lamdbaFunc));
        (_e = props.users.x509AuthUsers) === null || _e === void 0 ? void 0 : _e.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: (_f = props.users.passwordAuthUsers) === null || _f === void 0 ? void 0 : _f.map(secret => secret.secretArn),
            X509AuthUsers: (_g = props.users.x509AuthUsers) === null || _g === void 0 ? void 0 : _g.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
_a = JSII_RTTI_SYMBOL_1;
MongoDbPostInstallSetup[_a] = { fqn: "aws-rfdk.MongoDbPostInstallSetup", version: "0.39.0" };
//# sourceMappingURL=data:application/json;base64,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