"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 *
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *    1) Deadline client is not installed on it;
 *    2) Deadline client is installed but not running on it;
 *    3) RCS is not configured correctly for Deadline client;
 *    4) it is unable to connect to RCS due to any infrastructure issues;
 *    5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *    1) at least 1 instance is unhealthy for the configured grace period;
 *    2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * ![architecture diagram](/diagrams/core/HealthMonitor.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Application Load Balancer(s) doing frequent pings to the workers.
 * - An Amazon Simple Notification Service (SNS) topic for all unhealthy fleet notifications.
 * - An AWS Key Management Service (KMS) Key to encrypt SNS messages - If no encryption key is provided.
 * - An Amazon CloudWatch Alarm that triggers if a worker fleet is unhealthy for a long period.
 * - Another CloudWatch Alarm that triggers if the healthy host percentage of a worker fleet is lower than allowed.
 * - A single AWS Lambda function that sets fleet size to 0 when triggered in response to messages on the SNS Topic.
 * - Execution logs of the AWS Lambda function are published to a log group in Amazon CloudWatch.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is created by this construct to terminate unhealthy worker fleets has permission to
 *    UpdateAutoScalingGroup ( https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_UpdateAutoScalingGroup.html )
 *    on all of the fleets that this construct is monitoring. You should not grant any additional actors/principals the
 *    ability to modify or execute this Lambda.
 * - Execution of the AWS Lambda for terminating unhealthy workers is triggered by messages to the Amazon Simple
 *    Notification Service (SNS) Topic that is created by this construct. Any principal that is able to publish notification
 *    to this SNS Topic can cause the Lambda to execute and reduce one of your worker fleets to zero instances. You should
 *    not grant any additional principals permissions to publish to this SNS Topic.
 *
 * @stability stable
 */
class HealthMonitor extends HealthMonitorBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${core_1.Names.uniqueId(this)}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level monitoring.
     *
     * The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @stability stable
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
_a = JSII_RTTI_SYMBOL_1;
HealthMonitor[_a] = { fqn: "aws-rfdk.HealthMonitor", version: "0.39.0" };
/**
 * Default health check listening port.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 *
 * @stability stable
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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