"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type },
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type },
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
describe('default properties', () => {
    test('hash key only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
            KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('removalPolicy is DESTROY', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('hash + range key', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('hash + range key can also be specified in props', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('point-in-time recovery is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('server-side encryption is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('stream is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('ttl is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('can specify new and old images', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            TableName: 'MyTable',
        });
    });
    test('can specify new images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
            TableName: 'MyTable',
        });
    });
    test('can specify old images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.OLD_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
            TableName: 'MyTable',
        });
    });
});
test('when specifying every property', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        readCapacity: 42,
        writeCapacity: 1337,
        pointInTimeRecovery: true,
        serverSideEncryption: true,
        billingMode: lib_1.BillingMode.PROVISIONED,
        stream: lib_1.StreamViewType.KEYS_ONLY,
        timeToLiveAttribute: 'timeToLive',
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 42,
            WriteCapacityUnits: 1337,
        },
        PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
        SSESpecification: { SSEEnabled: true },
        StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
        TableName: 'MyTable',
        Tags: [{ Key: 'Environment', Value: 'Production' }],
        TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true },
    });
});
test('when specifying sse with customer managed CMK', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'MyTableKey8597C7A6',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying only encryptionKey', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying sse with customer managed CMK with encryptionKey provided by user', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('fails if encryption key is used with AWS managed CMK', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.AWS_MANAGED,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with default encryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with serverSideEncryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        serverSideEncryption: true,
        encryptionKey,
    })).toThrow(/encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead/);
});
test('fails if both encryption and serverSideEncryption is specified', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        serverSideEncryption: true,
    })).toThrow(/Only one of encryption and serverSideEncryption can be specified, but both were provided/);
});
test('fails if both replication regions used with customer managed CMK', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        replicationRegions: ['us-east-1', 'us-east-2', 'us-west-2'],
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    })).toThrow('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
});
test('if an encryption key is included, decrypt permissions are also added for grantStream', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryptionKey,
        stream: lib_1.StreamViewType.NEW_IMAGE,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantStreamRead(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'Key961B73FD': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'Key961B73FD',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_IMAGE',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'dynamodb:ListStreams',
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableA3D7B5AFA',
                                                    'Arn',
                                                ],
                                            },
                                            '/stream/*',
                                        ],
                                    ],
                                },
                            },
                            {
                                'Action': [
                                    'dynamodb:DescribeStream',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableA3D7B5AFA',
                                        'StreamArn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('if an encryption key is included, encrypt/decrypt permissions are also added both ways', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantReadWriteData(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'TableAKey07CC09EC': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::GetAtt': [
                                            'MyUserDC45028B',
                                            'Arn',
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'Description': 'Customer-managed key auto-created for encrypting DynamoDB table at Table A',
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'TableAKey07CC09EC',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': [
                                    'dynamodb:BatchGetItem',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                    'dynamodb:Query',
                                    'dynamodb:GetItem',
                                    'dynamodb:Scan',
                                    'dynamodb:BatchWriteItem',
                                    'dynamodb:PutItem',
                                    'dynamodb:UpdateItem',
                                    'dynamodb:DeleteItem',
                                ],
                                'Effect': 'Allow',
                                'Resource': [
                                    {
                                        'Fn::GetAtt': [
                                            'TableA3D7B5AFA',
                                            'Arn',
                                        ],
                                    },
                                    {
                                        'Ref': 'AWS::NoValue',
                                    },
                                ],
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableAKey07CC09EC',
                                        'Arn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('when specifying PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
        ],
        TableName: 'MyTable',
    });
});
test('error when specifying read or write capacity with a PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table B', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table C', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding a global secondary index with hash key only', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.ALL,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index on a table with PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    }).addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
    })).toThrow(/non-key attributes should be specified when using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type ALL, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    const gsiNonKeyAttributes = [];
    for (let i = 0; i < 21; i++) {
        gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
    }
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: gsiNonKeyAttributes,
    })).toThrow(/a maximum number of nonKeyAttributes across all of secondary indexes is 20/);
});
test('error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
    });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding multiple global secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiGenerator = GSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addGlobalSecondaryIndex(gsiGenerator.next().value);
    }
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey4', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI0',
                KeySchema: [
                    { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI1',
                KeySchema: [
                    { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI2',
                KeySchema: [
                    { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI3',
                KeySchema: [
                    { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI4',
                KeySchema: [
                    { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index without specifying read and write capacity', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a local secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value],
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
            },
        ],
    });
});
test('error when adding more than 5 local secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiGenerator = LSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addLocalSecondaryIndex(lsiGenerator.next().value);
    }
    expect(() => table.addLocalSecondaryIndex(lsiGenerator.next().value))
        .toThrow(/a maximum number of local secondary index per table is 5/);
});
test('error when adding a local secondary index with the name of a global secondary index', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: 'SecondaryIndex',
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(() => table.addLocalSecondaryIndex({
        indexName: 'SecondaryIndex',
        sortKey: LSI_SORT_KEY,
    })).toThrow(/a duplicate index name, SecondaryIndex, is not allowed/);
});
test('error when validating construct if a local secondary index exists without a sort key of the table', () => {
    var _a;
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    const errors = core_1.ConstructNode.validate(table.node);
    expect(errors.length).toBe(1);
    expect((_a = errors[0]) === null || _a === void 0 ? void 0 : _a.message).toBe('a sort key of the table must be specified to add local secondary indexes');
});
test('can enable Read AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('can enable Write AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('cannot enable AutoScaling twice on the same property', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/Read AutoScaling already enabled for this table/);
});
test('error when enabling AutoScaling on the PAY_PER_REQUEST table', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => {
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
        minCapacity: 1,
        maxCapacity: 5,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('error when specifying Read Auto Scaling with invalid scalingTargetValue < 10', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
    }).toThrow(/targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: 5/);
});
test('error when specifying Read Auto Scaling with invalid minimumCapacity', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }))
        .toThrow(/minCapacity \(10\) should be lower than maxCapacity \(5\)/);
});
test('can autoscale on a schedule', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        readCapacity: 42,
        writeCapacity: 1337,
        partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING },
    });
    // WHEN
    const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
    scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
        schedule: appscaling.Schedule.cron({}),
        maxCapacity: 10,
    });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        ScheduledActions: [
            {
                ScalableTargetAction: { 'MaxCapacity': 10 },
                Schedule: 'cron(* * * * ? *)',
                ScheduledActionName: 'SaveMoneyByNotScalingUp',
            },
        ],
    });
});
describe('metrics', () => {
    test('Can use metricConsumedReadCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedReadCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedReadCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricConsumedWriteCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedWriteCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedWriteCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricSystemErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSystemErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SystemErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricUserErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricUserErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'UserErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricConditionalCheckFailedRequests on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConditionalCheckFailedRequests())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConditionalCheckFailedRequests',
            statistic: 'Sum',
        });
    });
    test('Can use metricSuccessfulRequestLatency on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSuccessfulRequestLatency())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SuccessfulRequestLatency',
            statistic: 'Average',
        });
    });
});
describe('grants', () => {
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:action1', 'dynamodb:action2');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:action1',
                            'dynamodb:action2',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        testGrant(['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
    });
    test('"grantReadData" allows the principal to read data from the table', () => {
        testGrant(['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
    });
    test('"grantWriteData" allows the principal to write data to the table', () => {
        testGrant(['BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'], (p, t) => t.grantWriteData(p));
    });
    test('"grantReadWriteData" allows the principal to read/write data', () => {
        testGrant([
            'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
            'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
        ], (p, t) => t.grantReadWriteData(p));
    });
    test('"grantFullAccess" allows the principal to perform any action on the table ("*")', () => {
        testGrant(['*'], (p, t) => t.grantFullAccess(p));
    });
    test('"Table.grantListStreams" allows principal to list all streams', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const user = new iam.User(stack, 'user');
        // WHEN
        lib_1.Table.grantListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantTableListStreams" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantTableListStreams(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantTableListStreams" allows principal to list all streams for this table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantStreamRead" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantStreamRead(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantStreamRead" allows principal to read and describe the table stream"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantStreamRead(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                    {
                        'Action': [
                            'dynamodb:DescribeStream',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'StreamArn',
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('if table has an index grant gives access to the index', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
        table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'mytable0324D45C',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('grant for an imported table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = lib_1.Table.fromTableName(stack, 'MyTable', 'my-table');
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:*');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:*',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                Ref: 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            Users: [
                {
                    Ref: 'user2C2B57AE',
                },
            ],
        });
    });
});
describe('secondary indexes', () => {
    // See https://github.com/aws/aws-cdk/issues/4398
    test('attribute can be used as key attribute in one index, and non-key in another', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'pkey', type: lib_1.AttributeType.NUMBER },
        });
        // WHEN
        table.addGlobalSecondaryIndex({
            indexName: 'IndexA',
            partitionKey: { name: 'foo', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['bar'],
        });
        // THEN
        expect(() => table.addGlobalSecondaryIndex({
            indexName: 'IndexB',
            partitionKey: { name: 'baz', type: lib_1.AttributeType.STRING },
            sortKey: { name: 'bar', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['blah'],
        })).not.toThrow();
    });
});
describe('import', () => {
    test('report error when importing an external/existing table from invalid arn missing resource name', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
        // WHEN
        expect(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn)).toThrow(/ARN for DynamoDB table must be in the form: .../);
    });
    test('static fromTableArn(arn) allows importing an external/existing table from arn', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
        const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            tableArn,
                            { 'Ref': 'AWS::NoValue' },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe(tableArn);
        expect(stack.resolve(table.tableName)).toBe('MyTable');
    });
    test('static fromTableName(name) allows importing an external/existing table from table name', () => {
        const stack = new core_1.Stack();
        const tableName = 'MyTable';
        const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadWriteData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            'Ref': 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            'Ref': 'AWS::AccountId',
                                        },
                                        ':table/MyTable',
                                    ],
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe('arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
        expect(stack.resolve(table.tableName)).toBe(tableName);
    });
    describe('stream permissions on imported tables', () => {
        test('throw if no tableStreamArn is specified', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(() => table.grantTableListStreams(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
            expect(() => table.grantStreamRead(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
        });
        test('creates the correct list streams grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantTableListStreams(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct stream read grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantStreamRead(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                        {
                            Action: ['dynamodb:DescribeStream', 'dynamodb:GetRecords', 'dynamodb:GetShardIterator'],
                            Effect: 'Allow',
                            Resource: tableStreamArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
    });
});
describe('global', () => {
    test('create replicas', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-west-2',
            },
            Condition: 'TableStackRegionNotEqualseuwest2A03859E7',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-central-1',
            },
            Condition: 'TableStackRegionNotEqualseucentral199D46FC0',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toEqual({
            TableStackRegionNotEqualseuwest2A03859E7: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-west-2', { Ref: 'AWS::Region' }] },
                ],
            },
            TableStackRegionNotEqualseucentral199D46FC0: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-central-1', { Ref: 'AWS::Region' }] },
                ],
            },
        });
    });
    test('grantReadData', () => {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'TableCD117FA1',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'TableCD117FA1',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantReadData across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantTableListStreams across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:ListStreams',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('throws with PROVISIONED billing mode', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            billingMode: lib_1.BillingMode.PROVISIONED,
        })).toThrow(/`PAY_PER_REQUEST`/);
    });
    test('throws when stream is set and not set to NEW_AND_OLD_IMAGES', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            stream: lib_1.StreamViewType.OLD_IMAGE,
        })).toThrow(/`NEW_AND_OLD_IMAGES`/);
    });
    test('throws with replica in same region as stack', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'us-east-1' },
        });
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-1',
                'us-east-1',
                'eu-west-2',
            ],
        })).toThrow(/`replicationRegions` cannot include the region where this stack is deployed/);
    });
    test('no conditions when region is known', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'eu-west-1' },
        });
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toBeUndefined();
    });
});
function testGrant(expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': action,
                    'Effect': 'Allow',
                    'Resource': [
                        {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'Arn',
                            ],
                        },
                        {
                            'Ref': 'AWS::NoValue',
                        },
                    ],
                },
            ],
            'Version': '2012-10-17',
        },
        'PolicyName': 'userDefaultPolicy083DF682',
        'Users': [
            {
                'Ref': 'user2C2B57AE',
            },
        ],
    });
}
//# sourceMappingURL=data:application/json;base64,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