# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['easy_email_downloader']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'easy-email-downloader',
    'version': '2.0.0',
    'description': '',
    'long_description': '# easy-email-downloader 📨\n\n![PyPI](https://img.shields.io/pypi/v/easy-email-downloader)\n![PyPI - Python Version](https://img.shields.io/pypi/pyversions/easy-email-downloader)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/simple-email-sender)\n\n`easy-email-downloader` is a no nonsense and easy way to download emails from an IMAP server in Python with no\n3rd party dependencies.\n\n## Uses\n\n- Extract data/logs etc. from email content for further parsing or saving to disk.\n- Calculate general statistics from your inbox (who\'s sending me the most etc.)\n- Get a list of all senders so you can identify who to unsubsribe from.\n- Automatically download attachments (receipts/statements etc.) and save to disk.\n- Archive content of emails for personal use.\n\nSee the installation & examples below.\n\nPull requests/ideas/suggestions are welcome 👍🏻\n\n## Installation\n\n### PyPI\n\n```bash\npip install easy-email-downloader\n```\n\n### From Source\n\n#### Using [Poetry](https://python-poetry.org)\n\n👍🏻 recommended\n\nClone the repository and run:\n\n```bash\npoetry install\n```\n\n#### setuptools\n\nClone the repository and run in a virtual environment:\n\n```bash\npython setup.py install\n```\n\n## Usage\n\nTo download emails from an IMAP server:\n\n```python\nfrom easy_email_downloader import EmailFilter, EmailConfig, download_emails\n\nemail_config = EmailConfig(\n    host="mail.example.com",\n    email_address="example@example.com",\n    password="somepassword",\n    mailbox="INBOX",\n    port=993,\n)\n\n# The EmailFilter itself and all of its fields are optional. Sensible defaults are chosen if left out.\nemail_filter = EmailFilter(\n    subject="daily report",  # subject filtering is server specific - full string matches often won\'t work\n    sender="someone@gmail.com",\n    messages_to_download=1,  # setting this to -1 will download all emails\n    oldest_first=False,  # whether to retrive the oldest emails first\n    delete_after_download=False,  # be careful setting this to True - emails are permanently removed!\n)\n\ndownloaded_emails = download_emails(\n    email_config=email_config,\n    email_filter=email_filter,\n)\n\nfor downloaded_email in downloaded_emails:\n    print(f"sender: {downloaded_email.sender}")\n    print(f"date: {downloaded_email.date}")\n    print(f"subject: {downloaded_email.subject}")\n    print(f"content_type: {downloaded_email.content_type}")\n    print(f"body: {downloaded_email.body[:50]}")\n    for attachment in downloaded_email.attachments:\n        print(f"filename: {attachment.filename}")\n```\n\nWhich produces:\n\n```\nsender: Daniel Tomlinson <dtomlinson@panaetius.co.uk>\ndate: Sat, 23 Apr 2022 20:55:31 +0100\nsubject: Download me using easy-email-sender\ncontent_type: text/html\nbody: Successfully downloaded using easy-email-sender!\nfilename: hello.jpg\n```\n\nSee below for more information on `EmailFilter`, `EmailConfig`, `download_emails` or see the docstrings.\n### EmailConfig\n\nCreate your `EmailConfig` instance:\n\nIf you know the mailbox you want to download from:\n\n```python\nemail_config = EmailConfig(\n    host="mail.example.com", email_address="example@example.com", password="somepassword", mailbox="INBOX", port=993\n)\n```\n\nIf you don\'t know the mailbox to download from, use `EmailConfig.list_mailboxes`\n\n```\nemail_config = EmailConfig(\n    host="mail.example.com", email_address="example@example.com", password="somepassword", port=993\n)\nuser_mailboxes = email_config.list_mailboxes()\n```\n\nThis will return a list of strings of all the mailboxes in the user\'s account.\n\n`EmailConfig` supports the following arguments:\n\n```\nAttributes:\n    host (str): The host of the IMAP server.\n    email_address (str): The email address to login as.\n    password (str): The password.\n    mailbox (str): The mailbox to download emails from.\n    port (int): The IMAP port of the server.\n```\n\n### EmailFilter\n\nCreate an `EmailFilter` instance:\n\n```python\nemail_filter = EmailFilter(\n    subject="daily report",\n    sender="someone@gmail.com",\n    messages_to_download=1,\n    oldest_first=False,\n    delete_after_download=False,\n)\n```\n\n`subject` and `sender` are optional. Either both, one or none of `subject` and `sender` can be set. If neither are\nprovided all emails in the mailbox will be searched for.\n\n`EmailFilter` supports the following arguments:\n\n```\nAttributes:\n    subject (str | None, optional): The subject to filter by. Defaults to None.\n    sender (str | None, optional): The sender email address to filter by. Defaults to None.\n    messages_to_download (int, optional): The number of emails to download. If -1 will download all. Defaults to -1.\n    oldest_first (bool, optional): Whether to download emails starting with the oldest first. Defaults to False.\n    delete_after_download (bool, optional): Whether to delete the emails after successfully downloading. Defaults to False.\n```\n\n🚨 Be careful with `delete_after_download` - be sure you want to delete the email before you run it. This calls\n`imaplib.IMAP4_SSL.expunge` which **permanently** deletes the email from the server.\n\n### download_emails\n\nDownload emails using the `EmailConfig` and `EmailFilter`:\n\n```python\ndownloaded_emails = download_emails(\n    email_config=email_config,\n    email_filter=email_filter,\n)\n```\n\nThis returns a list of `Email` objects.\n\nThe available attributes on an `Email` object are:\n\n```\nAttributes:\n    sender (str): The sender in the form `first_name last_name <email_address>`.\n    subject (str): The email subject.\n    date (str): The date the email was sent.\n    body (str): The content of the email. This is either in plaintext or as HTML.\n    attachments (List[Optional[Attachment]]): A list of attachments as\n        [Attachment][easy_email_downloader.models.Attachment] objects. If no attachments this is an empty list.\n    content_type (str): The content type. Either `text/plain` or `text/html`.\n```\n\nAttachments are stored as `Attachment` objects containing the `filename` and `contents` (in bytes).\n\nThese can be saved to disk:\n\n```\nfor attachment in downloaded_emails.attachments:\n    with open(f"/some/path/{attachment.filename}", mode="wb") as attachment_contents:\n        attachment_contents.write(attachment.contents)\n```\n\n## TODO\n\n- ~~Extend attatchment logic to get filename of attatchment on server.~~ ✅\n  - ~~Useful to save attatchments to disk automatically without the user having to implement `file.open(mode="wb")`.~~ ✅\n- CLI to download emails to json/csv etc.\n  - Should include a Dockerfile and an image published to Dockerhub.\n- Tests 🥱.\n',
    'author': 'Daniel Tomlinson',
    'author_email': 'dtomlinson@panaetius.co.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dtomlinson91/easy-email-downloader',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
