import string
from abc import ABCMeta, abstractmethod
from datetime import datetime
from queue import Queue
from typing import Dict, List, Literal, Union

import numpy as np
import pandas as pd
import pytz
from loguru import logger

from bbstrader.btengine.data import DataHandler
from bbstrader.btengine.event import Events, FillEvent, SignalEvent
from bbstrader.config import BBSTRADER_DIR
from bbstrader.metatrader import (
    Account,
    AdmiralMarktsGroup,
    PepperstoneGroupLimited,
    TradeOrder,
    Rates,
    TradeSignal, 
    TradingMode,
    SymbolType
)
from bbstrader.models.optimization import optimized_weights

__all__ = ["Strategy", "MT5Strategy"]

logger.add(
    f"{BBSTRADER_DIR}/logs/strategy.log",
    enqueue=True,
    level="INFO",
    format="{time:YYYY-MM-DD HH:mm:ss} | {level} | {name} | {message}",
)


class Strategy(metaclass=ABCMeta):
    """
    A `Strategy()` object encapsulates all calculation on market data
    that generate advisory signals to a `Portfolio` object. Thus all of
    the "strategy logic" resides within this class. We opted to separate
    out the `Strategy` and `Portfolio` objects for this backtester,
    since we believe this is more amenable to the situation of multiple
    strategies feeding "ideas" to a larger `Portfolio`, which then can handle
    its own risk (such as sector allocation, leverage). In higher frequency trading,
    the strategy and portfolio concepts will be tightly coupled and extremely
    hardware dependent.

    At this stage in the event-driven backtester development there is no concept of
    an indicator or filter, such as those found in technical trading. These are also
    good candidates for creating a class hierarchy.

    The strategy hierarchy is relatively simple as it consists of an abstract
    base class with a single pure virtual method for generating `SignalEvent` objects.
    Other methods are provided to check for pending orders, update trades from fills,
    and get updates from the portfolio.
    """

    @abstractmethod
    def calculate_signals(self, *args, **kwargs) -> List[TradeSignal]:
        pass

    def check_pending_orders(self, *args, **kwargs): ...
    def get_update_from_portfolio(self, *args, **kwargs): ...
    def update_trades_from_fill(self, *args, **kwargs): ...
    def perform_period_end_checks(self, *args, **kwargs): ...


class MT5Strategy(Strategy):
    """
    A `MT5Strategy()` object is a subclass of `Strategy` that is used to
    calculate signals for the MetaTrader 5 trading platform. The signals
    are generated by the `MT5Strategy` object and sent to the the `Mt5ExecutionEngine`
    for live trading and `MT5BacktestEngine` objects for backtesting.

    # NOTE
    It is recommanded that every strategy specfic method to be a private method
    in order to avoid naming collusion.
    """
    tf: str
    max_trades: Dict[str, int]
    def __init__(
        self,
        events: Queue = None,
        symbol_list: List[str] = None,
        bars: DataHandler = None,
        mode: TradingMode = None,
        **kwargs,
    ):
        """
        Initialize the `MT5Strategy` object.

        Args:
            events : The event queue.
            symbol_list : The list of symbols for the strategy.
            bars : The data handler object.
            mode (TradingMode): The mode of operation for the strategy.
            **kwargs : Additional keyword arguments for other classes (e.g, Portfolio, ExecutionHandler).
                - max_trades : The maximum number of trades allowed per symbol.
                - time_frame : The time frame for the strategy.
                - logger : The logger object for the strategy.
        """
        self.events = events
        self.data = bars
        self.symbols = symbol_list
        self.mode = mode
        self._porfolio_value = None
        self.risk_budget = self._check_risk_budget(**kwargs)
        self.max_trades = kwargs.get("max_trades", {s: 1 for s in self.symbols})
        self.tf = kwargs.get("time_frame", "D1")
        self.logger = kwargs.get("logger") or logger
        if self.mode == TradingMode.BACKTEST:
            self._initialize_portfolio()
        self.kwargs = kwargs
        self.periodes = 0

    @property
    def account(self):
        if self.mode != TradingMode.LIVE:
            raise ValueError("account attribute is only allowed in Live mode")
        return Account(**self.kwargs)

    @property
    def cash(self) -> float:
        if self.mode == TradingMode.LIVE:
            return self.account.balance
        return self._porfolio_value

    @cash.setter
    def cash(self, value):
        if self.mode == TradingMode.LIVE:
            raise ValueError("Cannot set the account cash in live mode")
        self._porfolio_value = value

    @property
    def orders(self):
        if self.mode == TradingMode.LIVE:
            return self.account.get_orders() or []
        return self._orders

    @property
    def trades(self) -> Dict[str, Dict[str, int]]:
        if self.mode == TradingMode.LIVE:
            raise ValueError("Cannot call this methode in live mode")
        return self._trades

    @property
    def positions(self):
        if self.mode == TradingMode.LIVE:
            return self.account.get_positions() or []
        return self._positions

    @property
    def holdings(self) -> Dict[str, float]:
        if self.mode == TradingMode.LIVE:
            raise ValueError("Cannot call this methode in live mode")
        return self._holdings

    def _check_risk_budget(self, **kwargs):
        weights = kwargs.get("risk_weights")
        if weights is not None and isinstance(weights, dict):
            for asset in self.symbols:
                if asset not in weights:
                    raise ValueError(f"Risk budget for asset {asset} is missing.")
            total_risk = float(round(sum(weights.values())))
            if not np.isclose(total_risk, 1.0):
                raise ValueError(f"Risk budget weights must sum to 1. got {total_risk}")
            return weights
        elif isinstance(weights, str):
            return weights

    def _initialize_portfolio(self):
        positions = ["LONG", "SHORT"]
        orders = ["BLMT", "BSTP", "BSTPLMT", "SLMT", "SSTP", "SSTPLMT"]
        self._orders: Dict[str, Dict[str, List[SignalEvent]]] = {}
        self._positions: Dict[str, Dict[str, int | float]] = {}
        self._trades: Dict[str, Dict[str, int]] = {}
        for symbol in self.symbols:
            self._positions[symbol] = {}
            self._orders[symbol] = {}
            self._trades[symbol] = {}
            for position in positions:
                self._trades[symbol][position] = 0
                self._positions[symbol][position] = 0.0
            for order in orders:
                self._orders[symbol][order] = []
        self._holdings = {s: 0.0 for s in self.symbols}

    def get_update_from_portfolio(self, positions, holdings):
        """
        Update the positions and holdings for the strategy from the portfolio.

        Positions are the number of shares of a security that are owned in long or short.
        Holdings are the value (postions * price) of the security that are owned in long or short.

        Args:
            positions : The positions for the symbols in the strategy.
            holdings : The holdings for the symbols in the strategy.
        """
        for symbol in self.symbols:
            if symbol in positions:
                if positions[symbol] > 0:
                    self._positions[symbol]["LONG"] = positions[symbol]
                elif positions[symbol] < 0:
                    self._positions[symbol]["SHORT"] = positions[symbol]
                else:
                    self._positions[symbol]["LONG"] = 0
                    self._positions[symbol]["SHORT"] = 0
            if symbol in holdings:
                self._holdings[symbol] = holdings[symbol]

    def update_trades_from_fill(self, event: FillEvent):
        """
        This method updates the trades for the strategy based on the fill event.
        It is used to keep track of the number of trades executed for each order.
        """
        if event.type == Events.FILL:
            if event.order != "EXIT":
                self._trades[event.symbol][event.order] += 1
            elif event.order == "EXIT" and event.direction == "BUY":
                self._trades[event.symbol]["SHORT"] = 0
            elif event.order == "EXIT" and event.direction == "SELL":
                self._trades[event.symbol]["LONG"] = 0

    def calculate_signals(self, *args, **kwargs) -> List[TradeSignal]:
        """
        Provides the mechanisms to calculate signals for the strategy.
        This methods should return a list of signals for the strategy.

        Each signal must be a ``TradeSignal`` object with the following attributes:
        - ``action``: The order to execute on the symbol (LONG, SHORT, EXIT, etc.), see `bbstrader.core.utils.TradeAction`.
        - ``price``: The price at which to execute the action, used for pending orders.
        - ``stoplimit``: The stop-limit price for STOP-LIMIT orders, used for pending stop limit orders.
        - ``id``: The unique identifier for the strategy or order.
        - ``comment``: An optional comment or description related to the trade signal.
        """
        pass

    def perform_period_end_checks(self, *args, **kwargs):
        """
        Some strategies may require additional checks at the end of the period,
        such as closing all positions or orders or tracking the performance of the strategy etc.

        This method is called at the end of the period to perform such checks.
        """
        pass

    def apply_risk_management(
        self, optimer, symbols=None, freq=252
    ) -> Dict[str, float] | None:
        """
        Apply risk management rules to the strategy.
        """
        if optimer is None:
            return None
        symbols = symbols or self.symbols
        prices = self.get_asset_values(
            symbol_list=symbols,
            bars=self.data,
            mode=self.mode,
            window=freq,
            value_type="close",
            array=False,
            tf=self.tf,
        )
        prices = pd.DataFrame(prices)
        prices = prices.dropna(axis=0, how="any")
        try:
            weights = optimized_weights(prices=prices, freq=freq, method=optimer)
            return {symbol: abs(weight) for symbol, weight in weights.items()}
        except Exception:
            return {symbol: 0.0 for symbol in symbols}

    def get_quantity(self, symbol, weight, price=None, volume=None, maxqty=None) -> int:
        """
        Calculate the quantity to buy or sell for a given symbol based on the dollar value provided.
        The quantity calculated can be used to evalute a strategy's performance for each symbol
        given the fact that the dollar value is the same for all symbols.

        Args:
            symbol : The symbol for the trade.

        Returns:
            qty : The quantity to buy or sell for the symbol.
        """
        if (
            self._porfolio_value is None
            or weight == 0
            or self._porfolio_value == 0
            or np.isnan(self._porfolio_value)
        ):
            return 0
        if volume is None:
            volume = round(self._porfolio_value * weight)
        if price is None:
            price = self.data.get_latest_bar_value(symbol, "close")
        if (
            price is None
            or not isinstance(price, (int, float, np.number))
            or volume is None
            or not isinstance(volume, (int, float, np.number))
            or np.isnan(float(price))
            or np.isnan(float(volume))
        ):
            if weight != 0:
                return 1
            return 0
        qty = round(volume / price, 2)
        qty = max(qty, 0) / self.max_trades[symbol]
        if maxqty is not None:
            qty = min(qty, maxqty)
        return max(round(qty, 2), 0)

    def get_quantities(self, quantities: Union[None, dict, int]) -> dict:
        """
        Get the quantities to buy or sell for the symbols in the strategy.
        This method is used when whe need to assign different quantities to the symbols.

        Args:
            quantities : The quantities for the symbols in the strategy.
        """
        if quantities is None:
            return {symbol: None for symbol in self.symbols}
        if isinstance(quantities, dict):
            return quantities
        elif isinstance(quantities, int):
            return {symbol: quantities for symbol in self.symbols}

    def _send_order(
        self,
        id,
        symbol: str,
        signal: str,
        strength: float,
        price: float,
        quantity: int,
        dtime: datetime | pd.Timestamp,
    ):
        position = SignalEvent(
            id, symbol, dtime, signal, quantity=quantity, strength=strength, price=price
        )
        log = False
        if signal in ["LONG", "SHORT"]:
            if self._trades[symbol][signal] < self.max_trades[symbol] and quantity > 0:
                self.events.put(position)
                log = True
        elif signal == "EXIT":
            if (
                self._positions[symbol]["LONG"] > 0
                or self._positions[symbol]["SHORT"] < 0
            ):
                self.events.put(position)
                log = True
        if log:
            self.logger.info(
                f"{signal} ORDER EXECUTED: SYMBOL={symbol}, QUANTITY={quantity}, PRICE @{round(price, 5)}",
                custom_time=dtime,
            )

    def buy_mkt(
        self,
        id: int,
        symbol: str,
        price: float,
        quantity: int,
        strength: float = 1.0,
        dtime: datetime | pd.Timestamp = None,
    ):
        """
        Open a long position

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        self._send_order(id, symbol, "LONG", strength, price, quantity, dtime)

    def sell_mkt(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Open a short position

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        self._send_order(id, symbol, "SHORT", strength, price, quantity, dtime)

    def close_positions(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Close a position or exit all positions

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        self._send_order(id, symbol, "EXIT", strength, price, quantity, dtime)

    def buy_stop(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Open a pending order to buy at a stop price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price <= current_price:
            raise ValueError(
                "The buy_stop price must be greater than the current price."
            )
        order = SignalEvent(
            id, symbol, dtime, "LONG", quantity=quantity, strength=strength, price=price
        )
        self._orders[symbol]["BSTP"].append(order)

    def sell_stop(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Open a pending order to sell at a stop price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price >= current_price:
            raise ValueError("The sell_stop price must be less than the current price.")
        order = SignalEvent(
            id,
            symbol,
            dtime,
            "SHORT",
            quantity=quantity,
            strength=strength,
            price=price,
        )
        self._orders[symbol]["SSTP"].append(order)

    def buy_limit(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Open a pending order to buy at a limit price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price >= current_price:
            raise ValueError("The buy_limit price must be less than the current price.")
        order = SignalEvent(
            id, symbol, dtime, "LONG", quantity=quantity, strength=strength, price=price
        )
        self._orders[symbol]["BLMT"].append(order)

    def sell_limit(self, id, symbol, price, quantity, strength=1.0, dtime=None):
        """
        Open a pending order to sell at a limit price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price <= current_price:
            raise ValueError(
                "The sell_limit price must be greater than the current price."
            )
        order = SignalEvent(
            id,
            symbol,
            dtime,
            "SHORT",
            quantity=quantity,
            strength=strength,
            price=price,
        )
        self._orders[symbol]["SLMT"].append(order)

    def buy_stop_limit(
        self,
        id: int,
        symbol: str,
        price: float,
        stoplimit: float,
        quantity: int,
        strength: float = 1.0,
        dtime: datetime | pd.Timestamp = None,
    ):
        """
        Open a pending order to buy at a stop-limit price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price <= current_price:
            raise ValueError(
                f"The stop price {price} must be greater than the current price {current_price}."
            )
        if price >= stoplimit:
            raise ValueError(
                f"The stop-limit price {stoplimit} must be greater than the price {price}."
            )
        order = SignalEvent(
            id,
            symbol,
            dtime,
            "LONG",
            quantity=quantity,
            strength=strength,
            price=price,
            stoplimit=stoplimit,
        )
        self._orders[symbol]["BSTPLMT"].append(order)

    def sell_stop_limit(
        self, id, symbol, price, stoplimit, quantity, strength=1.0, dtime=None
    ):
        """
        Open a pending order to sell at a stop-limit price

        See `bbstrader.btengine.event.SignalEvent` for more details on arguments.
        """
        current_price = self.data.get_latest_bar_value(symbol, "close")
        if price >= current_price:
            raise ValueError(
                f"The stop price {price} must be less than the current price {current_price}."
            )
        if price <= stoplimit:
            raise ValueError(
                f"The stop-limit price {stoplimit} must be less than the price {price}."
            )
        order = SignalEvent(
            id,
            symbol,
            dtime,
            "SHORT",
            quantity=quantity,
            strength=strength,
            price=price,
            stoplimit=stoplimit,
        )
        self._orders[symbol]["SSTPLMT"].append(order)

    def check_pending_orders(self):
        """
        Check for pending orders and handle them accordingly.
        """

        def logmsg(order, type, symbol, dtime):
            return self.logger.info(
                f"{type} ORDER EXECUTED: SYMBOL={symbol}, QUANTITY={order.quantity}, "
                f"PRICE @ {round(order.price, 5)}",
                custom_time=dtime,
            )

        def process_orders(order_type, condition, execute_fn, log_label, symbol, dtime):
            for order in self._orders[symbol][order_type].copy():
                if condition(order):
                    execute_fn(order)
                    try:
                        self._orders[symbol][order_type].remove(order)
                        assert order not in self._orders[symbol][order_type]
                    except AssertionError:
                        self._orders[symbol][order_type] = [
                            o for o in self._orders[symbol][order_type] if o != order
                        ]
                    logmsg(order, log_label, symbol, dtime)

            for symbol in self.symbols:
                dtime = self.data.get_latest_bar_datetime(symbol)
                latest_close = self.data.get_latest_bar_value(symbol, "close")

                process_orders(
                    "BLMT",
                    lambda o: latest_close <= o.price,
                    lambda o: self.buy_mkt(
                        o.strategy_id, symbol, o.price, o.quantity, dtime
                    ),
                    "BUY LIMIT",
                    symbol,
                    dtime,
                )

                process_orders(
                    "SLMT",
                    lambda o: latest_close >= o.price,
                    lambda o: self.sell_mkt(
                        o.strategy_id, symbol, o.price, o.quantity, dtime
                    ),
                    "SELL LIMIT",
                    symbol,
                    dtime,
                )

                process_orders(
                    "BSTP",
                    lambda o: latest_close >= o.price,
                    lambda o: self.buy_mkt(
                        o.strategy_id, symbol, o.price, o.quantity, dtime
                    ),
                    "BUY STOP",
                    symbol,
                    dtime,
                )

                process_orders(
                    "SSTP",
                    lambda o: latest_close <= o.price,
                    lambda o: self.sell_mkt(
                        o.strategy_id, symbol, o.price, o.quantity, dtime
                    ),
                    "SELL STOP",
                    symbol,
                    dtime,
                )

                process_orders(
                    "BSTPLMT",
                    lambda o: latest_close >= o.price,
                    lambda o: self.buy_limit(
                        o.strategy_id, symbol, o.stoplimit, o.quantity, dtime
                    ),
                    "BUY STOP LIMIT",
                    symbol,
                    dtime,
                )

                process_orders(
                    "SSTPLMT",
                    lambda o: latest_close <= o.price,
                    lambda o: self.sell_limit(
                        o.strategy_id, symbol, o.stoplimit, o.quantity, dtime
                    ),
                    "SELL STOP LIMIT",
                    symbol,
                    dtime,
                )

    @staticmethod
    def calculate_pct_change(current_price, lh_price) -> float:
        return ((current_price - lh_price) / lh_price) * 100

    def get_asset_values(
        self,
        symbol_list: List[str],
        window: int,
        value_type: str = "returns",
        array: bool = True,
        bars: DataHandler = None,
        mode: TradingMode = TradingMode.BACKTEST,
        tf: str = "D1",
        error: Literal["ignore", "raise"] = None,
    ) -> Dict[str, np.ndarray | pd.Series] | None:
        """
        Get the historical OHLCV value or returns or custum value
        based on the DataHandker of the assets in the symbol list.

        Args:
            bars : DataHandler for market data handling, required for backtest mode.
            symbol_list : List of ticker symbols for the pairs trading strategy.
            value_type : The type of value to get (e.g., returns, open, high, low, close, adjclose, volume).
            array : If True, return the values as numpy arrays, otherwise as pandas Series.
            mode : Mode of operation for the strategy.
            window : The lookback period for resquesting the data.
            tf : The time frame for the strategy.
            error : The error handling method for the function.

        Returns:
            asset_values : Historical values of the assets in the symbol list.

        Note:
            In Live mode, the `bbstrader.metatrader.rates.Rates` class is used to get the historical data
            so the value_type must be 'returns', 'open', 'high', 'low', 'close', 'adjclose', 'volume'.
        """
        if mode not in [TradingMode.BACKTEST, TradingMode.LIVE]:
            raise ValueError("Mode must be an instance of TradingMode")
        asset_values = {}
        if mode == TradingMode.BACKTEST:
            if bars is None:
                raise ValueError("DataHandler is required for backtest mode.")
            for asset in symbol_list:
                if array:
                    values = bars.get_latest_bars_values(asset, value_type, N=window)
                    asset_values[asset] = values[~np.isnan(values)]
                else:
                    values = bars.get_latest_bars(asset, N=window)
                    asset_values[asset] = getattr(values, value_type)
        elif mode == TradingMode.LIVE:
            for asset in symbol_list:
                rates = Rates(asset, timeframe=tf, count=window + 1, **self.kwargs)
                if array:
                    values = getattr(rates, value_type).to_numpy()
                    asset_values[asset] = values[~np.isnan(values)]
                else:
                    values = getattr(rates, value_type)
                    asset_values[asset] = values
        if all(len(values) >= window for values in asset_values.values()):
            return {a: v[-window:] for a, v in asset_values.items()}
        else:
            if error == "raise":
                raise ValueError("Not enough data to calculate the values.")
            elif error == "ignore":
                return asset_values
            return None

    @staticmethod
    def is_signal_time(period_count, signal_inverval) -> bool:
        """
        Check if we can generate a signal based on the current period count.
        We use the signal interval as a form of periodicity or rebalancing period.

        Args:
            period_count : The current period count (e.g., number of bars).
            signal_inverval : The signal interval for generating signals (e.g., every 5 bars).

        Returns:
            bool : True if we can generate a signal, False otherwise
        """
        if period_count == 0 or period_count is None:
            return True
        return period_count % signal_inverval == 0

    @staticmethod
    def stop_time(time_zone: str, stop_time: str) -> bool:
        now = datetime.now(pytz.timezone(time_zone)).time()
        stop_time = datetime.strptime(stop_time, "%H:%M").time()
        return now >= stop_time

    def ispositions(
        self, symbol, strategy_id, position, max_trades, one_true=False, account=None
    ) -> bool:
        """
        This function is use for live trading to check if there are open positions
        for a given symbol and strategy. It is used to prevent opening more trades
        than the maximum allowed trades per symbol.

        Args:
            symbol : The symbol for the trade.
            strategy_id : The unique identifier for the strategy.
            position : The position type (1: short, 0: long).
            max_trades : The maximum number of trades allowed per symbol.
            one_true : If True, return True if there is at least one open position.
            account : The `bbstrader.metatrader.Account` object for the strategy.

        Returns:
            bool : True if there are open positions, False otherwise
        """
        account = account or self.account
        positions = account.get_positions(symbol=symbol)
        if positions is not None:
            open_positions = [
                pos.ticket
                for pos in positions
                if pos.type == position and pos.magic == strategy_id
            ]
            if one_true:
                return len(open_positions) in range(1, max_trades + 1)
            return len(open_positions) >= max_trades
        return False

    def get_positions_prices(self, symbol, strategy_id, position, account=None):
        """
        Get the buy or sell prices for open positions of a given symbol and strategy.

        Args:
            symbol : The symbol for the trade.
            strategy_id : The unique identifier for the strategy.
            position : The position type (1: short, 0: long).
            account : The `bbstrader.metatrader.Account` object for the strategy.

        Returns:
            prices : numpy array of buy or sell prices for open positions if any or an empty array.
        """
        account = account or self.account
        positions = account.get_positions(symbol=symbol)
        if positions is not None:
            prices = np.array(
                [
                    pos.price_open
                    for pos in positions
                    if pos.type == position and pos.magic == strategy_id
                ]
            )
            return prices
        return np.array([])
    
    def get_active_orders(self, symbol: str, strategy_id: int, order_type: int = None) -> List[TradeOrder]:
        """
        Get the active orders for a given symbol and strategy.

        Args:
            symbol : The symbol for the trade.
            strategy_id : The unique identifier for the strategy.
            order_type : The type of order to filter by (optional):
                    "BUY_LIMIT": 2
                    "SELL_LIMIT": 3
                    "BUY_STOP": 4
                    "SELL_STOP": 5
                    "BUY_STOP_LIMIT": 6
                    "SELL_STOP_LIMIT": 7

        Returns:
            List[TradeOrder] : A list of active orders for the given symbol and strategy.
        """
        orders = [o for o in self.orders if o.symbol == symbol and o.magic == strategy_id]
        if order_type is not None and len(orders) > 0:
            orders = [o for o in orders if o.type == order_type]
        return orders

    def exit_positions(self, position, prices, asset, th: float = 0.01):
        if len(prices) == 0:
            return False
        tick_info = self.account.get_tick_info(asset)
        bid, ask = tick_info.bid, tick_info.ask
        if len(prices) == 1:
            price = prices[0]
        elif len(prices) in range(2, self.max_trades[asset] + 1):
            price = np.mean(prices)
        if (
            position == 0
            and self.calculate_pct_change(ask, price) >= th
            or position == 1
            and abs(self.calculate_pct_change(bid, price)) >= th
        ):
            return True
        return False

    @staticmethod
    def get_current_dt(time_zone: str = "US/Eastern") -> datetime:
        return datetime.now(pytz.timezone(time_zone))

    @staticmethod
    def convert_time_zone(
        dt: datetime | int | pd.Timestamp,
        from_tz: str = "UTC",
        to_tz: str = "US/Eastern",
    ) -> pd.Timestamp:
        """
        Convert datetime from one timezone to another.

        Args:
            dt : The datetime to convert.
            from_tz : The timezone to convert from.
            to_tz : The timezone to convert to.

        Returns:
            dt_to : The converted datetime.
        """
        from_tz = pytz.timezone(from_tz)
        if isinstance(dt, datetime):
            dt = pd.to_datetime(dt, unit="s")
        elif isinstance(dt, int):
            dt = pd.to_datetime(dt, unit="s")
        if dt.tzinfo is None:
            dt = dt.tz_localize(from_tz)
        else:
            dt = dt.tz_convert(from_tz)

        dt_to = dt.tz_convert(pytz.timezone(to_tz))
        return dt_to

    @staticmethod
    def get_mt5_equivalent(
        symbols, symbol_type: str | SymbolType = SymbolType.STOCKS, **kwargs
    ) -> List[str]:
        """
        Get the MetaTrader 5 equivalent symbols for the symbols in the list.
        This method is used to get the symbols that are available on the MetaTrader 5 platform.

        Args:
            symbols : The list of symbols to get the MetaTrader 5 equivalent symbols for.
            symbol_type : The type of symbols to get (See `bbstrader.metatrader.utils.SymbolType`).
            **kwargs : Additional keyword arguments for the `bbstrader.metatrader.Account` object.

        Returns:
            mt5_equivalent : The MetaTrader 5 equivalent symbols for the symbols in the list.
        """
        account = Account(**kwargs)
        mt5_symbols = account.get_symbols(symbol_type=symbol_type)
        mt5_equivalent = []
        if account.broker == AdmiralMarktsGroup():
            for s in mt5_symbols:
                _s = s[1:] if s[0] in string.punctuation else s
                for symbol in symbols:
                    if _s.split(".")[0] == symbol or _s.split("_")[0] == symbol:
                        mt5_equivalent.append(s)
        elif account.broker == PepperstoneGroupLimited():
            for s in mt5_symbols:
                for symbol in symbols:
                    if s.split(".")[0] == symbol:
                        mt5_equivalent.append(s)
        return mt5_equivalent


class TWSStrategy(Strategy): ...
