# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['cut', 'detection']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.1.3',
 'dask>=2021.10.0',
 'netCDF4>=1.5.7',
 'numba>=0.54.1',
 'numpy>=1.20.3',
 'pandas>=1.3.5',
 'scipy>=1.7.3',
 'xarray>=2022.6.0']

entry_points = \
{'console_scripts': ['mhw-cut = src.mhw_detect.cut.cut:cut',
                     'mhw-detect = '
                     'src.mhw_detect.detection.mhw:extreme_events']}

setup_kwargs = {
    'name': 'mhw-detect',
    'version': '0.1.7',
    'description': 'Fast marine heatwaves and extrem events detector based on https://github.com/ecjoliver/marineHeatWaves',
    'long_description': "# MHW Detector\n\nMarine heatwaves detector based on https://github.com/ecjoliver/marineHeatWaves.\n\nThis package integrates a numba optimised version of ecjoliver's implementation for MHW detection with multiprocessing capabilities to compute detection over every coordinates of the dataset.\n\nThis code is not only for detecting MHW. It can also be used to detect extrem events of any variables like chla, pH, O2, etc...\n\n## Installation\n> pip install mhw-detect\n\n\n### Dependencies\n- xarray\n- numba\n- scipy\n- dask\n- numpy\n- pandas\n- netcdf4\n- click\n\n## Usage\n### Configuration file\nWith mhw-detect no need for kilometers of parameters in command line. You just need to write a configuration file in which you put every parameters like an identity card of your next detection.\n\n```yaml\ndata:\n  data :\n    path : '/folder/sst.nc'\n    var : 'sst'\n  clim :\n    path : '/folder_clim/clim.nc'\n    var : 'sst'\n  percent :\n    path : '/folder_percent/percentile.nc'\n    var : 'sst'\n\nparams:\n    depth : 0\n    climatologyPeriod : [null, null]\n    pctile : 90\n    windowHalfWidth : 5\n    smoothPercentile : True\n    smoothPercentileWidth : 31\n    minDuration : 5\n    joinAcrossGaps : True\n    maxGap : 2\n    maxPadLength: False\n    coldSpells : False\n    Ly : False\n\ncut:\n  nb_lat : 157\n  nb_lon : 72\n\noutput_detection : '/my/path/to/folder_result/'\n```\n\n- `data` : specifies the paths and variables you want to use. Do not specify clim and percent if you want them to be computed during the detection.\n- `params` : specifies the parameters of the detection. See section below.\n- `cut` : specifies the number of latitude and longitude for geospatial dataset cutting.\n- `output_detection` : specifies the folder in which to save the results.\n\n\n### Detection parameters\nFrom https://github.com/ecjoliver/marineHeatWaves.\n\n```\nclimatologyPeriod      Period over which climatology is calculated, specified\n                        as list of start and end years. Default ([null, null]) is to calculate\n                        over the full range of years in the supplied time series.\n                        Alternate periods suppled as a list e.g. [1983,2012].\n                        Unused if precalculated clim and percentile are set.\n\npctile                 Threshold percentile (%) for detection of extreme values\n                        (DEFAULT = 90)\n\nwindowHalfWidth        Width of window (one sided) about day-of-year used for\n                        the pooling of values and calculation of threshold percentile\n                        (DEFAULT = 5 [days])\n\nsmoothPercentile       Boolean switch indicating whether to smooth the threshold\n                        percentile timeseries with a moving average (DEFAULT = True)\n\nsmoothPercentileWidth  Width of moving average window for smoothing threshold\n                        (DEFAULT = 31 [days])\n\nminDuration            Minimum duration for acceptance detected MHWs\n                        (DEFAULT = 5 [days])\n\njoinAcrossGaps         Boolean switch indicating whether to join MHWs\n                        which occur before/after a short gap (DEFAULT = True)\n\nmaxGap                 Maximum length of gap allowed for the joining of MHWs\n                        (DEFAULT = 2 [days])\n\nmaxPadLength           Specifies the maximum length [days] over which to interpolate\n                        (pad) missing data (specified as nans) in input temp time series.\n                        i.e., any consecutive blocks of NaNs with length greater\n                        than maxPadLength will be left as NaN. Set as an integer.\n                        (DEFAULT = False, interpolates over all missing values).\n\ncoldSpells             Specifies if the code should detect cold events instead of\n                        heat events. (DEFAULT = False)\n```\n\n### Datasets coordinates\nAll datasets must have lat/latitude, lon/longitude and time as coordinates. `depth` coordinate is allowed for the main dataset. Currently, the depth as to be specified via its index in the coordinate array. Giving directly the wanted depth will be added later.\n\nThe percentile dataset must have a `quantile` coordinate as a dimension for the variable. It is useful in the case you want to do the detection with different quantile (90, 99).\n\n### Step 1 : Geospatial cutting (optionnal but recommended)\nTo use multiprocessing efficiently, the datasets must be cut in several smaller datasets over the lat/lon dimensions. Call `mhw-cut` with your config file to make it happen. Each sub-datasets will be called Cut_X.nc where X is the number of the cut (that is why your datasets (data, clim, percentile) must be in different folders).\n\nThe number of cuts does not matter, chunk size does. To find suitables nb_lat and nb_lon, it is better to use a notebook.\n>import xarray as xr\nds = xr.open_dataset('dataset.nc', chunks={'latitude': nb_lat, 'longitude': nb_lon})\nds\n\nnb_lat and nb_lon should be multiples of the latitude and longitude dimensions and choose carefully to have chunks of size over 10Mb (see Dask documentation for more details). Printing ds on a notebook gives you the size of the chunk (cut).\n\nPlease note that this step will double the space used in your disk.\n\n\n### Step 2 : Detection\nCall `mhw-detect` to detect MHW. With multiprocessing, each cut is processed in parallel. For a cut, you will get in `output_detection` a text file with the results of the detection. Finally, when all the detections are done, every text files are concatenated into one csv (with ; as a separator).\n\nIf you do not want to use multiprocessing just to make a detection on a small geospatial subset, you can give the option `-g lat_min lat_max lon_min lon_max` to the command.\n\n\n### Commands\n#### Geospatial cut\n> mhw-cut -c config.yml\n\n#### Detection\nCompute detection with multiprocessing:\n> mhw-detect -c config.yml\n\nCompute detection on a geospatial subset (no multiprocessing):\n> mhw-detect -c config.yml -g lat_min lat_max lon_min lon_max\n\n## To do\n- Add plot command.\n- Add shapefile usage.\n- Add an option to remove text files.\n- Add specify depth and not its index.\n\n\n\n## References\n\nHobday, A.J. et al. (2016), A hierarchical approach to defining marine heatwaves, Progress in Oceanography, 141, pp. 227-238, doi: 10.1016/j.pocean.2015.12.014 [pdf](http://passage.phys.ocean.dal.ca/~olivere/docs/Hobdayetal_2016_PO_HierarchMHWDefn.pdf)\n",
    'author': 'John Brouillet',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<3.12',
}


setup(**setup_kwargs)
