import torch
import torch.nn as nn
import math
import torch.utils.model_zoo as model_zoo
from Losses.Mask_bce_loss import Attention_loss
import numpy as np
from Losses.Mask_bce_loss import UnNormalizer
unnormalize = UnNormalizer()
import matplotlib.pyplot as plt
import cv2
import os

pwd = os.path.abspath(__file__+'../../../')

__all__ = ['ResNet', 'resnet18_cbam', 'resnet34_cbam', 'resnet50_cbam', 'resnet101_cbam',
           'resnet152_cbam']


model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-5c106cde.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-333f7ec4.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-19c8e357.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-5d3b4d8f.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-b121ed2d.pth',
}


def conv3x3(in_planes, out_planes, stride=1):
    "3x3 convolution with padding"
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)

class ChannelAttention(nn.Module):
    def __init__(self, in_planes, ratio=16):
        super(ChannelAttention, self).__init__()
        self.avg_pool = nn.AdaptiveAvgPool2d(1)
        self.max_pool = nn.AdaptiveMaxPool2d(1)

        self.fc1   = nn.Conv2d(in_planes, in_planes // 16, 1, bias=False)
        self.relu1 = nn.ReLU()
        self.fc2   = nn.Conv2d(in_planes // 16, in_planes, 1, bias=False)

        self.sigmoid = nn.Sigmoid()

    def forward(self, x):
        avg_out = self.fc2(self.relu1(self.fc1(self.avg_pool(x))))
        max_out = self.fc2(self.relu1(self.fc1(self.max_pool(x))))
        out = avg_out + max_out
        return self.sigmoid(out)

class SpatialAttention(nn.Module):
    def __init__(self, kernel_size=7):
        super(SpatialAttention, self).__init__()

        assert kernel_size in (3, 7), 'kernel size must be 3 or 7'
        padding = 3 if kernel_size == 7 else 1

        self.conv1 = nn.Conv2d(2, 1, kernel_size, padding=padding, bias=False)
        self.sigmoid = nn.Sigmoid()

    def forward(self, x):
        avg_out = torch.mean(x, dim=1, keepdim=True)
        max_out, _ = torch.max(x, dim=1, keepdim=True)
        x = torch.cat([avg_out, max_out], dim=1)
        x = self.conv1(x)
        return self.sigmoid(x)


class FACEAttention(nn.Module):
    def __init__(self, channel_in, channel_size):
        super(FACEAttention, self).__init__()

        self.conv1 = nn.Conv2d(channel_in, channel_size, kernel_size=3, padding=1)
        self.relu1 = nn.ReLU()
        self.bn1 = nn.BatchNorm2d(channel_size)

        self.conv2 = nn.Conv2d(channel_size, channel_size, kernel_size=3, padding=1)
        self.relu2 = nn.ReLU()
        self.bn2 = nn.BatchNorm2d(channel_size)

        self.conv3 = nn.Conv2d(channel_size, 1, kernel_size=3, padding=1)
        self.sigmoid = nn.Sigmoid()

    def forward(self, x):
        out = self.conv1(x)
        out = self.relu1(out)
        out = self.bn1(out)
        out = self.conv2(out)
        out = self.relu2(out)
        out = self.bn2(out)
        out = self.conv3(out)

        # w = list(np.unique(out.detach().cpu().numpy()))
        # w.sort()
        # print('outoutout5: ', out.size(), w)
        out = self.sigmoid(out)
        # w = list(np.unique(out.detach().cpu().numpy()))
        # w.sort()
        # print('outoutout6: ', out.size(), w[:3], w[-3:],w)
        return out
    # def __init__(self, channel_in, channel_size):
    #     super(FACEAttention, self).__init__()
    #     # self.conv1 = nn.Conv2d(channel_in+2, 1, kernel_size=3, padding=1,bias=False)
    #     self.conv1 = nn.Conv2d(channel_in+2, channel_in, kernel_size=3, padding=1,bias=False)
    #     self.conv2 = nn.Conv2d(channel_in, 1, kernel_size=3, padding=1,bias=False)
    #     self.relu1 = nn.ReLU()
    #     self.bn1 = nn.BatchNorm2d(channel_in)
    #
    #     self.sigmoid = nn.Sigmoid()
    #
    # def forward(self, x):
    #     avg_out = torch.mean(x, dim=1, keepdim=True)
    #     max_out, _ = torch.max(x, dim=1, keepdim=True)
    #     # x = self.bn1(x)
    #     x = torch.cat([avg_out, max_out, x], dim=1)
    #     out = self.conv1(x)
    #     out = self.bn1(out)
    #     # out = self.relu1(out)
    #     out = self.conv2(out)
    #     w = list(np.unique(out.detach().cpu().numpy()))
    #     w.sort()
    #     print('outoutout5: ', out.size(), w)
    #     out = self.sigmoid(out)
    #     w = list(np.unique(out.detach().cpu().numpy()))
    #     w.sort()
    #     print('outoutout6: ', out.size(), w[:3], w[-3:])
    #     return out

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)

        self.ca = ChannelAttention(planes)
        self.sa = SpatialAttention()

        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        out = self.ca(out) * out
        out = self.sa(out) * out

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes * 4)
        self.relu = nn.ReLU(inplace=True)

        self.ca = ChannelAttention(planes * 4)
        self.sa = SpatialAttention()

        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        out = self.ca(out) * out
        out = self.sa(out) * out

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(self, block, layers, showlayer= False, num_classes=128):
        self.inplanes = 64
        self.showlayer = showlayer
        super(ResNet, self).__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2)
        self.avgpool = nn.AvgPool2d(7, stride=1)
        self.Fa = FACEAttention(512, 512)
        self.fc = nn.Linear(2048, num_classes)
        self.aloss = Attention_loss()

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes))

        return nn.Sequential(*layers)

    def forward(self, inputs):
        if self.training:
            img, mask = inputs
        else:
            img = inputs
        x = self.conv1(img)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        # W=list(np.unique(x.detach().cpu().numpy()))
        # W.sort()
        # print(111111, x.size(), W)
        xs = self.Fa(x)
        if self.training:
            loss = self.aloss(xs, mask)
        x = x * torch.exp(xs)
        tmp = torch.exp(xs)
        if self.showlayer:
            i = 0
            for level in [x]:
                i += 1
                level = level.squeeze(0)
                # levelnpy = level.detach().cpu().numpy()
                if torch.cuda.is_available():
                    level = np.array(255 * unnormalize(level).detach().cpu().numpy()).copy()
                else:
                    level = np.array(255 * unnormalize(level).detach().numpy()).copy()
                level = np.transpose(level, (1, 2, 0))
                version = input('输入版本号：V*')
                version = version.upper()
                layers = level.shape[2]
                plt.imsave(os.path.join(pwd, 'Layer_show', version + '.jpg'), level[:, :, 0])
                # for ij in range(layers):
                #     plt.imsave(os.path.join(pwd, 'Layer_show', version+ '_layer%s.jpg'%ij), level[:, :, ij])
        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)
        # x = torch.div(x, torch.norm(x))*50
        if self.training:
            return x, loss
        else:
            return x

def resnet18_cbam(pretrained=True, **kwargs):
    """Constructs a ResNet-18 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet(BasicBlock, [2, 2, 2, 2], **kwargs)
    if pretrained:
        pretrained_state_dict = model_zoo.load_url(model_urls['resnet18'])
        now_state_dict        = model.state_dict()
        state_dict = {k: v for k, v in pretrained_state_dict.items() if (k in now_state_dict.keys()) and \
                      ('fc.weight' not in now_state_dict.keys())}
        now_state_dict.update(state_dict)
        # now_state_dict.update(pretrained_state_dict)
        model.load_state_dict(now_state_dict)
    return model


def resnet34_cbam(pretrained=True, **kwargs):
    """Constructs a ResNet-34 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet(BasicBlock, [3, 4, 6, 3], **kwargs)
    if pretrained:
        pretrained_state_dict = model_zoo.load_url(model_urls['resnet34'])
        now_state_dict = model.state_dict()
        state_dict = {k: v for k, v in pretrained_state_dict.items() if (k in now_state_dict.keys()) and \
                      ('fc.weight' not in now_state_dict.keys())}
        now_state_dict.update(state_dict)
        # now_state_dict.update(pretrained_state_dict)
        model.load_state_dict(now_state_dict)
    return model


def resnet50_cbam(pretrained=True, **kwargs):
    """Constructs a ResNet-50 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet(Bottleneck, [3, 4, 6, 3], **kwargs)
    if pretrained:
        pretrained_state_dict = model_zoo.load_url(model_urls['resnet50'])
        now_state_dict = model.state_dict()
        state_dict = {k: v for k, v in pretrained_state_dict.items() if (k in now_state_dict.keys()) and \
                      ('fc.weight' not in now_state_dict.keys())}
        now_state_dict.update(state_dict)
        # now_state_dict.update(pretrained_state_dict)
        model.load_state_dict(now_state_dict)
    return model


def resnet101_cbam(pretrained=True, **kwargs):
    """Constructs a ResNet-101 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet(Bottleneck, [3, 4, 23, 3], **kwargs)
    if pretrained:
        pretrained_state_dict = model_zoo.load_url(model_urls['resnet101'])
        now_state_dict = model.state_dict()
        state_dict = {k: v for k, v in pretrained_state_dict.items() if (k in now_state_dict.keys()) and \
                      ('fc.weight' not in now_state_dict.keys())}
        now_state_dict.update(state_dict)
        # now_state_dict.update(pretrained_state_dict)
        model.load_state_dict(now_state_dict)
    return model


def resnet152_cbam(pretrained=True, **kwargs):
    """Constructs a ResNet-152 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet(Bottleneck, [3, 8, 36, 3], **kwargs)
    if pretrained:
        pretrained_state_dict = model_zoo.load_url(model_urls['resnet152'])
        now_state_dict = model.state_dict()
        state_dict = {k: v for k, v in pretrained_state_dict.items() if (k in now_state_dict.keys()) and \
                      ('fc.weight' not in now_state_dict.keys())}
        now_state_dict.update(state_dict)
        # now_state_dict.update(pretrained_state_dict)
        model.load_state_dict(now_state_dict)
    return model