#
#  ive(nn, x)  - exponentially weighted modified bessel first kind
#
#   x : real
#   nn : 0 to 20
#
#  computated using
#     smallx : expansion
#     largex : asymptotic series
#     mid-range : rational approximation
#
#   

import scipy.special as scs
from scipy.special import ive
import numpy as np
from numpy.linalg import norm
import warnings

import petram.debug as debug
dprint1, dprint2, dprint3 = debug.init_dprints('NUMBA_IVE')

from numba import njit

factorial_table = np.array([
    1, 1, 2, 6, 24, 120, 720, 5040, 40320,
    362880, 3628800, 39916800, 479001600,
    6227020800, 87178291200, 1307674368000,
    20922789888000, 355687428096000, 6402373705728000,
    121645100408832000, 2432902008176640000], dtype='int64')

@njit
def factorial(n):
    if n > 20:
        raise ValueError
    return factorial_table[n]

@njit("float64(int32, float64)")
def ive_largex(nn, x):

    nnn = 4*nn**2
    term = (1
            - (nnn-1)/(8*x)
            + (nnn-1)*(nnn-9)/2/(8*x)**2
            - (nnn-1)*(nnn-9)*(nnn-25)/6/(8*x)**3
            + (nnn-1)*(nnn-9)*(nnn-25)*(nnn-49)/24/(8*x)**4
            - (nnn-1)*(nnn-9)*(nnn-25)*(nnn-49)*(nnn-81)/120/(8*x)**5)

    return term/np.sqrt(2*np.pi*x)

@njit("float64(int32, float64)")
def ive_smallx(nn, x):
    facn = factorial(nn)
    a = np.exp(-x)*(x/2)**nn
    terms = (1 + (x/2)**2/(nn+1) + (x/2)**4/(nn+1)/(nn+2)/2 + (x/2)**6/(nn+1)/(nn+2)/(nn+3)/6)
    return a*terms/facn

#
#   pre-computed table for n=0 to 20
#
#   output from
#     for i in range(21):
#         ans = petram.phys.common.numba_ive.search_approx_range(i)
#

c0 = [None]*21
c = [None]*21
d = [None]*21
x1 = [None]*21
x2 = [None]*21

c0[0] = (-3.4404917401919457+0j)
c[0] = [(45.49512033407257+0j), (0.4763433889789821-5.903584756561189j), (0.4763433889789821+5.903584756561189j), (-0.6157463594845617+0.5419598427597438j), (-0.6157463594845617-0.5419598427597438j)]
d[0] = [(-25.339056945578736+0j), (-3.0285812338396143+2.4435559942515175j), (-3.0285812338396143-2.4435559942515175j), (-0.20369168494342582+2.602692067450225j), (-0.20369168494342582-2.602692067450225j)]
x1[0] = 0.7886728615614156
x2[0] = 10.915359353313933
c0[1] = (-3.9226564942216022+0j)
c[1] = [(114.0487976834727+0j), (-0.04750180798666049-0.5583777108013944j), (-0.04750180798666049+0.5583777108013944j), (15.508991757533629-0j), (-1.4115660603557918+0.16609443814392044j), (-1.4115660603557918-0.16609443814392044j), (-2.466629618380557+0j), (-0.534499192371193-0j), (-0.2105369077167612+0j), (-0.08832201479937296-0j), (-0.029640490689724387+0j)]
d[1] = [(-79.50620403321611+0j), (-0.17432162788258576+5.713628079027035j), (-0.17432162788258576-5.713628079027035j), (-4.923878644915301+0j), (-0.5019544577511512+3.959957684122994j), (-0.5019544577511512-3.959957684122994j), (-1.662951947154347+0j), (-0.550995232931602+0j), (-0.20828968491161526+0j), (-0.06721848790300193+0j), (-0.011205796123132007+0j)]
x1[1] = 0.9704808877380307
x2[1] = 11.169868184678249
c0[2] = (-4.0616395320983045+0j)
c[2] = [(145.53675431637024-0j), (15.643215366796065+0j), (-0.39829783374250705-0.4867815171499894j), (-0.39829783374250705+0.4867815171499894j), (-5.384824801782728-0j), (-1.0788657647919764+0j), (-0.34717732641487725-0j), (-0.09945638859494366+0j)]
d[2] = [(-94.00335813653716+0j), (-5.093591544456992+0j), (-0.3216332108659989+4.3536425426732j), (-0.3216332108659989-4.3536425426732j), (-1.6499937902774175+0j), (-0.44070317978943946+0j), (-0.12078279983711207+0j), (-0.018418389798021826+0j)]
x1[2] = 1.1144152514667882
x2[2] = 12.248646137509333
c0[3] = (-3.927138582734833+0j)
c[3] = [(114.9985122198393-0j), (-0.30782028669407424-0.4444004609657096j), (-0.30782028669407424+0.4444004609657096j), (5.09072074532317+10.801748074331952j), (5.09072074532317-10.801748074331952j), (-3.0584285157763724+0j), (-1.0185447769267988-0j), (-0.386569117223627+0j), (-0.12137233137404108-0j)]
d[3] = [(-79.39624698642193+0j), (-0.20523478637429748+5.448747046843189j), (-0.20523478637429748-5.448747046843189j), (-3.3741072484777153+1.653957964867992j), (-3.3741072484777153-1.653957964867992j), (-0.914652009082425+0j), (-0.31182048524322975+0j), (-0.09442396490657952+0j), (-0.015158297654829617+0j)]
x1[3] = 1.2220446866314887
x2[3] = 14.393226447194094
c0[4] = (-4.924963534707903+0j)
c[4] = [(581.7993750219019-0j), (26.482146879163174+0j), (0.7801855741599533-1.6697245685587563j), (0.7801855741599533+1.6697245685587563j), (-10.972552218355075-0j), (-2.735156538081982+0j), (-0.8955761314297972-0j), (-0.2509801132234983+0j)]
d[4] = [(-288.33946247032117+0j), (-13.070639540506411+0j), (-1.3655521593374473+5.733705320003338j), (-1.3655521593374473-5.733705320003338j), (-1.9711994816371259+0j), (-0.5645567678583572+0j), (-0.15420107733127533+0j), (-0.023105965881576578+0j)]
x1[4] = 1.3400688963639507
x2[4] = 18.124175473742394
c0[5] = (-5.1198607193162555+0j)
c[5] = [(823.0330064653945-0j), (31.680730072879044+0j), (2.3933324401534497-1.5297264512663524j), (2.3933324401534497+1.5297264512663524j), (-14.755676970295827-0j), (-3.595657502197998+0j), (-1.158892438027093-0j), (-0.3245092944864868+0j)]
d[5] = [(-400.05746118101825+0j), (-20.120367136283225+0j), (-1.8906382535179316+6.092593948747974j), (-1.8906382535179316-6.092593948747974j), (-2.0889321692198655+0j), (-0.58764751557358+0j), (-0.15948402194779396+0j), (-0.02390694261264517+0j)]
x1[5] = 1.4360089846512607
x2[5] = 25.026400964179217
c0[6] = (-4.3021706985169175+0j)
c[6] = [(242.9573623618798-0j), (32.88572472885924+0j), (1.6508644484147121-1.7961161249342557j), (1.6508644484147121+1.7961161249342557j), (-24.676860619497553-0j), (-5.98152571165835+0j), (-1.9396010726408173-0j), (-0.5276988737285443+0j)]
d[6] = [(-169.71290399791127+0j), (-13.011515372553962+0j), (-1.7516489327468685+7.445003387913739j), (-1.7516489327468685-7.445003387913739j), (-2.89008856858203+0j), (-0.8158393539358795+0j), (-0.2195654961611431+0j), (-0.03209972650927598+0j)]
x1[6] = 1.5388177500383464
x2[6] = 41.554553347188794
c0[7] = (-5.1420034631525695+0j)
c[7] = [(1244.623708380224+0j), (80.51902618076582-0j), (47.122107930808916+0j), (1.5071834489213027-1.3998244394716934j), (1.5071834489213027+1.3998244394716934j), (-44.56519225811861-0j), (-9.087492477268635+0j), (-2.9305300207321996-0j), (-0.7821473814311941+0j)]
d[7] = [(-823.4095497187869+0j), (-91.1707551102828+0j), (-9.435237935840346+0j), (-1.5593046475008419+8.373879009266492j), (-1.5593046475008419-8.373879009266492j), (-3.9236718688134897+0j), (-1.0557173799637838+0j), (-0.2820616776987428+0j), (-0.04043080039664251+0j)]
x1[7] = 1.6114142772530198
x2[7] = 68.99837121430025
c0[8] = (-4.932363675469713+0j)
c[8] = [(855.6667302240703+0j), (70.23904086837817-0j), (1.7516449889449233-1.3240695286155977j), (1.7516449889449233+1.3240695286155977j), (88.57672872305521+0j), (-88.76249037732985-0j), (-12.602567232197543+0j), (-4.067763249116888-0j), (-1.072496437374869+0j)]
d[8] = [(-624.6279397813506+0j), (-86.57521343442353+0j), (-1.6490523665452363+9.262668801504518j), (-1.6490523665452363-9.262668801504518j), (-8.408353351398151+0j), (-5.099876718038103+0j), (-1.2787732696104783+0j), (-0.3410934441535474+0j), (-0.048116657064266134+0j)]
x1[8] = 1.6874356777273758
x2[8] = 99.76977642363211
c0[9] = (-4.996363061678075+0j)
c[9] = [(972.1901888786524-0j), (79.38135008630523+0j), (2.483548149610432-1.3789027839160226j), (2.483548149610432+1.3789027839160226j), (75.75837269649696-0j), (-80.93613556023405+0j), (-14.72716297790413-0j), (-4.572692945060632+0j), (-1.1516628494579033-0j)]
d[9] = [(-708.4111638255167+0j), (-97.71664298245389+0j), (-1.9939173675475568+10.032070381710318j), (-1.9939173675475568-10.032070381710318j), (-10.223302879931273+0j), (-5.0486971990121186+0j), (-1.2946386344483247+0j), (-0.33257949251970054+0j), (-0.0457580274682975+0j)]
x1[9] = 1.7670435260889465
x2[9] = 134.6260579298911
c0[10] = (-5.369435403759731+0j)
c[10] = [(2052.144357990143+0j), (169.75959429056377-0j), (42.25736889379038+0j), (1.6183622646500202-0.6648307879330295j), (1.6183622646500202+0.6648307879330295j), (0.46335768005864214+32.98032213787702j), (0.46335768005864214-32.98032213787702j), (-28.43747565898945-0j), (-8.136488513266386+0j), (-1.9924314327768913-0j)]
d[10] = [(-1533.241980183614+0j), (-240.4557740007149+0j), (-65.55157031047727+0j), (-1.292809881043428+11.019107465305481j), (-1.292809881043428-11.019107465305481j), (-5.980093991658271+4.377841613324338j), (-5.980093991658271-4.377841613324338j), (-2.1095798592938215+0j), (-0.5273288267491508+0j), (-0.06955369978716044+0j)]
x1[10] = 1.8504070195423021
x2[10] = 169.52323415541213
c0[11] = (-5.292918564378738+0j)
c[11] = [(1757.7309605834146-0j), (140.79995719508412+0j), (42.03585344403605-0j), (8.330691489587931+1.4252916858837257j), (8.330691489587931-1.4252916858837257j), (-65.70285933551133+0j), (-14.91759653110043-0j), (-3.1792397064937163+0j)]
d[11] = [(-1273.6136479423506+0j), (-172.97740384031493+0j), (-28.896514410777165+0j), (-3.6859910135089766+10.378612814879656j), (-3.6859910135089766-10.378612814879656j), (-3.7985411351948284+0j), (-0.8211722779024773+0j), (-0.09707004381290245+0j)]
x1[11] = 1.9377033374779888
x2[11] = 213.46630333242456
c0[12] = (-5.319334762710483+0j)
c[12] = [(1853.842944842827-0j), (149.1979037009601+0j), (47.135208768247715-0j), (9.551428676227205+1.5249941023775289j), (9.551428676227205-1.5249941023775289j), (-78.80026146289512+0j), (-18.13375997766123-0j), (-3.8491792524056074+0j)]
d[12] = [(-1345.3622278053397+0j), (-183.34868076550237+0j), (-29.795506592110513+0j), (-4.0309098328498845+11.281507167536187j), (-4.0309098328498845-11.281507167536187j), (-4.204437750849335+0j), (-0.9151715206535841+0j), (-0.1066882937649886+0j)]
x1[12] = 1.9828839491270713
x2[12] = 256.6902715491952
c0[13] = (-5.339121165152264+0j)
c[13] = [(1928.170722157643+0j), (154.67292113578367-0j), (53.96697613508976+0j), (10.411260229403975+1.213718565994823j), (10.411260229403975-1.213718565994823j), (-94.44569687721756-0j), (-21.918814337530584+0j), (-4.617586214210594-0j)]
d[13] = [(-1396.683946271346+0j), (-188.95389299887194+0j), (-28.97948194362039+0j), (-4.283034661180806+12.289599150034068j), (-4.283034661180806-12.289599150034068j), (-4.669002926854251+0j), (-1.0194013397158954+0j), (-0.11691287084948983+0j)]
x1[13] = 2.0764301072557747
x2[13] = 301.6334347259197
c0[14] = (-5.387032469038644+0j)
c[14] = [(2121.316227652467+0j), (169.08414811316263-0j), (61.17510581378472+0j), (11.75968151978576+1.16168839451587j), (11.75968151978576-1.16168839451587j), (-110.39449868932022-0j), (-25.92812741419512+0j), (-5.444584945193489-0j)]
d[14] = [(-1531.5192461117074+0j), (-204.3805307801515+0j), (-30.03873311626323+0j), (-4.634213513756359+13.224558040572372j), (-4.634213513756359-13.224558040572372j), (-5.098407324484497+0j), (-1.120092762691376+0j), (-0.1267876027113258+0j)]
x1[14] = 2.124845352498883
x2[14] = 354.44556739704353
c0[15] = (-5.818801787604116+0j)
c[15] = [(5027.654345750393+0j), (400.9779926708568-0j), (91.84297387492076+0j), (83.3907917040657-0j), (9.490371841148209+0.5802659709951578j), (9.490371841148209-0.5802659709951578j), (-143.74404804009936+0j), (-32.25493530824474-0j), (-6.624402923878525+0j)]
d[15] = [(-3683.3525818811454+0j), (-539.5237227108577+0j), (-137.88967718543446+0j), (-20.590421229675133+0j), (-4.179760089403365+14.389640369988388j), (-4.179760089403365-14.389640369988388j), (-5.994418100685252+0j), (-1.2888447109399188+0j), (-0.14180057609323432+0j)]
x1[15] = 2.2250887981283696
x2[15] = 416.5044248545185
c0[16] = (-5.7633602017976955+0j)
c[16] = [(4509.747426526107+0j), (369.1385034839494-0j), (88.19009636930468+0j), (109.90006907835355-0j), (9.371994629173747+0.6661696238202184j), (9.371994629173747-0.6661696238202184j), (-178.55918042781+0j), (-38.31207041541802-0j), (-7.751019412498489+0j)]
d[16] = [(-3346.523239636981+0j), (-510.7138896476909+0j), (-135.08892701987043+0j), (-18.82159513357819+0j), (-4.182798282164736+15.356161017239893j), (-4.182798282164736-15.356161017239893j), (-6.749891818297188+0j), (-1.4267516747438018+0j), (-0.15375851235590368+0j)]
x1[16] = 2.27697025538168
x2[16] = 467.37951079924636
c0[17] = (-5.782511938417699+0j)
c[17] = [(4684.648464668492+0j), (381.5532569878865-0j), (89.94860580650618+0j), (9.06200540129118+0.7861652999861884j), (9.06200540129118-0.7861652999861884j), (162.99604678527265-0j), (-238.6191195524644+0j), (-45.738845078264625-0j), (-9.102875943702918+0j)]
d[17] = [(-3466.1944407763176+0j), (-523.0342678897858+0j), (-136.591810273937+0j), (-4.122249463289526+16.33265806103862j), (-4.122249463289526-16.33265806103862j), (-16.880730130748827+0j), (-7.789318778676851+0j), (-1.592404947725051+0j), (-0.1677748943456356+0j)]
x1[17] = 2.3300614106969246
x2[17] = 536.6976945540476
c0[18] = (-5.811239172325813+0j)
c[18] = [(4961.3817245503515-0j), (403.9079757640223+0j), (95.3557269503359-0j), (10.445111475395963+1.1153001381454097j), (10.445111475395963-1.1153001381454097j), (170.22907285329623+0j), (-257.7232868474-0j), (-50.95743634528417+0j), (-10.157509843664283-0j)]
d[18] = [(-3671.0482405559615+0j), (-554.4408802645771+0j), (-145.24575585315188+0j), (-4.501203838846138+17.14788095118241j), (-4.501203838846138-17.14788095118241j), (-18.166617680679245+0j), (-8.119917177850454+0j), (-1.6788099116926596+0j), (-0.17567907822226622+0j)]
x1[18] = 2.3843904700937206
x2[18] = 602.2541201461928
c0[19] = (-5.852792575989611+0j)
c[19] = [(5390.750382823547-0j), (437.9360738781031+0j), (102.66679298083929-0j), (11.584482564184011+1.258865599041151j), (11.584482564184011-1.258865599041151j), (191.86959213839108+0j), (-290.41760767441826-0j), (-57.302624095794805+0j), (-11.39916636125876-0j)]
d[19] = [(-3982.9956775818127+0j), (-597.9407795951726+0j), (-155.2611517453332+0j), (-4.807832639768364+18.043204631617378j), (-4.807832639768364-18.043204631617378j), (-18.928832892451304+0j), (-8.636864239144504+0j), (-1.7889221613856978+0j), (-0.1853061856837876+0j)]
x1[19] = 2.43998629725955
x2[19] = 675.818116816111
c0[20] = (-5.926671841122811+0j)
c[20] = [(6249.910766139546-0j), (509.06234921459776+0j), (120.1841022193695-0j), (89.55556068497998+0j), (17.886893839683005+7.419088291056715j), (17.886893839683005-7.419088291056715j), (-226.13710272130461-0j), (-47.30606998188518+0j), (-7.381762154745213-0j)]
d[20] = [(-4627.884038701318+0j), (-701.5398442266414+0j), (-185.63374324364827+0j), (-32.064212232109384+0j), (-5.909445250525356+17.63116005811569j), (-5.909445250525356-17.63116005811569j), (-6.773384456443728+0j), (-1.263565908375743+0j), (-0.11582111185042332+0j)]
x1[20] = 2.496878428884327
x2[20] = 758.367791499719


def make_func(nn, c0, c_arr, d_arr, x1, x2):
    c_arr = np.array(c_arr)
    d_arr = np.array(d_arr)
    @njit("float64(float64)")
    def ive_ra(x):
        value = c0
        for c, d in zip(c_arr, d_arr):            
            value = value + c / (x - d)
        return value.real
    
    @njit("float64(float64)")
    def func(x):
        if x < x1:
            return ive_smallx(nn, x)
        if x> x2:
            return ive_largex(nn, x)
        return np.exp(ive_ra(x))
    return func

func_table = [make_func(_nn, _c0, _c, _d, _x1, _x2)
              for _nn, _c0, _c, _d, _x1, _x2 in zip(range(len(c0)),c0, c, d, x1, x2)]

def make_numbafunc_from_table(i=0, table=func_table, limit=len(func_table)):
    func = table[i]
    if i == len(table)-1:
       @njit("float64(int32, float64)")
       def fend(nn, x):
           if nn % 2 == 0:
               return func(np.abs(x))
           else:
               if x < 0:
                   return -func(-x)
               else:
                   return -func(x)
       return fend

    wrapp = make_numbafunc_from_table(i+1)           
    @njit("float64(int32, float64)")
    def wrap(nn, x):
        if nn < 0:
            nn = -nn
        if nn > limit:
            raise ValueError
        if nn == i:
           if nn % 2 == 0:
               return func(np.abs(x))
           else:
               if x < 0:
                   return -func(-x)
               else:
                   return func(x)
        return wrapp(nn, x)
    return wrap

ive = make_numbafunc_from_table(i=0, table=func_table)

def build_logive_caller(n=1):
    def func(x, nn=n):
        return np.log(ive(nn, x))
    return func

def build_ive_caller(n=1):
    def func(x, nn=n):
        return ive(nn, x)
    return func


def search_approx_range(nn, do_print=True):
    xx = np.logspace(-7, 3, 1000)
    tol = 1e-6

    do_smallx = True
    do_largex = False
    irange = [-1, -1]
    for i, x in  enumerate(xx):
        if do_smallx:
            check = np.abs(ive(nn, x)/ive_smallx(nn,x)-1)
            #print(check)
            if check > tol:
                #print("small x <= ", xx[i-1])
                x1 = xx[i-1]
                do_smallx=False
                do_largex=True
                irange[0] = i-1

        if do_largex:
            check = np.abs(ive(nn, x)/ive_largex(nn,x)-1)            
            #print(check)
            if check < tol:
                #print("large x >= ", xx[i])
                irange[1] = i
                break
    for ii in range(10):
        iii = irange[1] + ii + 4
        fits, narr, flag = find_fits(kmin=0.1, kmax=xx[iii], narr=[nn],
                           ngrid=300, tol=tol, verbose=False, fac=1)
        if flag:
            x2 = xx[iii]
            if do_print:
                print("c0["+str(nn)+"] = " + fits[0].c0.__str__())
                print("c["+str(nn)+"] = " + list(fits[0].c_arr).__str__())
                print("d["+str(nn)+"] = " + list(fits[0].d_arr).__str__())
                print("x1["+str(nn)+"] = " + str(x1))
                print("x2["+str(nn)+"] = " + str(x2))
            return fits, narr, x1, x2
    assert False, "Did not find approximation"

                
    
def find_fits(kmin=1e-7, kmax=200, nmax=5, narr=None,
              mmin=3, mmax=15, ngrid=300, tol=1e-7,
              verbose=False, fac=2):
    '''
    build term decomposition using the same d
    '''
    L = (np.linspace(0, 1, ngrid)**fac)*(kmax-kmin) + kmin

    from petram.helper.ra import find_decomposition

    if narr is None:
        narr = list(range(nmax+1))
    funcs = [build_logive_caller(n=i) for i in narr]
    fits = []
    successes = []
    errs = []
    for func in funcs:
        fit, success, err = find_decomposition(func, L,
                                               mmin=mmin, mmax=mmax,
                                               tol=tol,
                                               verbose=verbose)

        fits.append(fit)
        successes.append(success)
        errs.append(err)

    if not all(successes):
        print("can not find proper rational approximation")
        print(successes)
        flag = False
    else:
        flag = True

    return fits, narr, flag

def plot_fits(fits, narr, kmax=20, ngrid=300, pmax=-1, fac=2):

    nmax = len(fits)
    funcs = [build_ive_caller(n=i) for i in narr]
    
    if pmax < 0:
        pmax = kmax
    L = (np.linspace(0, 1, ngrid*30)**fac)*kmax + 1e-7        

    print(L)

    from ifigure.interactive import figure

    v = figure()
    v.nsec(nmax)
    for i, func in enumerate(funcs):
        v.isec(i)
        v.title("exp(-l)In(l) (n="+str(i+1)+")")
        v.plot(L, func(L), 'b')
        v.plot(L, np.exp(fits[i](L)), 'r--')
        #v.ylog()

