"""Simple tox plugin to facilitate working with abstract and concrete dependencies"""
import dataclasses
import itertools
from collections import defaultdict
from pathlib import Path
from typing import Dict, Set

import toml
import tox  # type: ignore

REQ_PATH = Path("requirements")

HEADER = """\
#
# This file is autogenerated by tox-constraints
# It is updated every time tox runs
#
"""


@dataclasses.dataclass
class Config:  # pylint: disable=too-few-public-methods
    """A collection of configurable parameters for this tool.

    This class serves as a single source of documentation for what parameters exist and
    what types and values they may take.
    """

    @staticmethod
    def read():
        """Read from default path"""
        pyproject_toml = toml.load("pyproject.toml")
        section = pyproject_toml["tool"]["tox-constraints"]

        # If any configuration of this plugin exists, enable the plugin unless
        # explicitly disabled to reduce the configuration needed.
        plugin_enabled = section.get("plugin_enabled", True)

        # Additional concrete requirements that cannot be captured in the constraints
        # file. This could be because they point to a path relative to the package
        # under test.
        concrete = section.get("concrete", {})

        return Config(plugin_enabled=plugin_enabled, concrete=concrete)

    plugin_enabled: bool
    concrete: Dict[str, str]


_SAMPLE = """
# alpha
bravo
charlie > 3
delta ; python_version >= '3.7'
echo-foxtrot
golf # hotel
"""


def _parse_requirements(text: str) -> Set[str]:
    """
    >>> sorted(_parse_requirements(_SAMPLE))
    ['bravo', 'charlie > 3', "delta ; python_version >= '3.7'", 'echo-foxtrot', 'golf']
    """
    return {
        req for req in (line.split("#")[0].strip() for line in text.splitlines()) if req
    }


def _expands_requirements(tool_config: Config, envconfig) -> None:
    filenames = ["install_requires.txt"] + [
        f"extras_require-{extra}.txt" for extra in envconfig.extras
    ]

    dep_names = set()
    for filename in filenames:
        filepath = REQ_PATH / filename
        dep_names.update(_parse_requirements(filepath.read_text()))

    new = {
        dep_name: tox.config.DepConfig(tool_config.concrete.get(dep_name, dep_name))
        for dep_name in dep_names
    }
    old = {dep.name: i for i, dep in enumerate(envconfig.deps)}

    for dep_name in new:
        if dep_name in old:
            envconfig.deps[old[dep_name]] = new[dep_name]
        else:
            envconfig.deps.append(new[dep_name])


def _find_upwards(start: Path, filename: str) -> Path:
    for dirpath in itertools.chain([start], start.parents):
        filepath = dirpath / filename
        if filepath.exists():
            return filepath

    raise FileNotFoundError


def _patch_envconfigs(tool_config: Config, config):
    for name, envconfig in config.envconfigs.items():
        if name == ".package":
            # Don't patch isolated packaging environment because some parsing will fail
            # on -cconstraints.txt
            continue
        if envconfig.skip_install is True and not envconfig.deps:
            # Avoid attempting to install no packages as pip does not like this:
            # > ERROR: You must give at least one requirement to install
            continue

        for dep in envconfig.deps:
            if dep.name.startswith("-c"):
                break
        else:
            constraints_filepath = _find_upwards(
                Path(config.toxinidir), "constraints.txt"
            )
            envconfig.deps.append(tox.config.DepConfig(f"-c{constraints_filepath}"))

        if envconfig.skip_install is True:
            pass
        elif envconfig.skip_install is False:
            _expands_requirements(tool_config, envconfig)
        else:
            raise ValueError


def _save_if_different(path: Path, new_content: str):
    """Save `new_content` to file if it is different to the existing content.

    Useful e.g. for only touching files when they change to avoid triggering make.
    """
    try:
        with path.open("r") as fp:
            old_content = fp.read()
    except FileNotFoundError:
        old_content = None

    if old_content != new_content:
        with path.open("w") as fp:
            fp.write(new_content)


def _export_deps(envconfigs):
    if not REQ_PATH.exists():
        REQ_PATH.mkdir()

    dep2names = defaultdict(list)
    for name, envconfig in envconfigs.items():
        for dep in envconfig.deps:
            if dep.name.startswith("-r"):
                continue  # Potential loop

            if dep.name.startswith("-c"):
                continue  # Constraint override

            dep2names[str(dep)].append(name)

    dep_len = max(len(dep) for dep in dep2names)
    lines = [
        f"{dep:<{dep_len}}  # from {', '.join(sorted(dep2names[dep]))}\n"
        for dep in sorted(dep2names)
    ]
    _save_if_different(REQ_PATH / "tox.txt", HEADER + "".join(lines))


@tox.hookimpl
def tox_configure(config):
    """Apply concrete constraints and export abstract dependencies"""
    try:
        tool_config = Config.read()
    except (FileNotFoundError, KeyError):
        # Disable plugin by default to make it less disruptive in a development
        # environment that is shared by multiple projects
        return

    if tool_config.plugin_enabled:
        _export_deps(config.envconfigs)
        _patch_envconfigs(tool_config, config)
