"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDeploymentGroup = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
/**
 * @stability stable
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class LambdaDeploymentGroup extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.LambdaApplication(this, 'Application');
        this.alarms = props.alarms || [];
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
        });
        this.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRoleForLambdaLimited'));
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            serviceRoleArn: this.role.roleArn,
            deploymentGroupName: this.physicalName,
            deploymentConfigName: this.deploymentConfig.deploymentConfigName,
            deploymentStyle: {
                deploymentType: 'BLUE_GREEN',
                deploymentOption: 'WITH_TRAFFIC_CONTROL',
            },
            alarmConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.any({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
        if (props.preHook) {
            this.addPreHook(props.preHook);
        }
        if (props.postHook) {
            this.addPostHook(props.postHook);
        }
        props.alias.node.defaultChild.cfnOptions.updatePolicy = {
            codeDeployLambdaAliasUpdate: {
                applicationName: this.application.applicationName,
                deploymentGroupName: resource.ref,
                beforeAllowTrafficHook: cdk.Lazy.string({ produce: () => this.preHook && this.preHook.functionName }),
                afterAllowTrafficHook: cdk.Lazy.string({ produce: () => this.postHook && this.postHook.functionName }),
            },
        };
        // If the deployment config is a construct, add a dependency to ensure the deployment config
        // is created before the deployment group is.
        if (this.deploymentConfig instanceof constructs_1.Construct) {
            this.node.addDependency(this.deploymentConfig);
        }
    }
    /**
     * Import an Lambda Deployment Group defined either outside the CDK app, or in a different AWS region.
     *
     * @param scope the parent Construct for this new Construct.
     * @param id the logical ID of this new Construct.
     * @param attrs the properties of the referenced Deployment Group.
     * @returns a Construct representing a reference to an existing Deployment Group
     * @stability stable
     */
    static fromLambdaDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedLambdaDeploymentGroup(scope, id, attrs);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group.
     * @stability stable
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    /**
     * Associate a function to run before deployment begins.
     *
     * @param preHook function to run before deployment beings.
     * @stability stable
     * @throws an error if a pre-hook function is already configured
     */
    addPreHook(preHook) {
        if (this.preHook !== undefined) {
            throw new Error('A pre-hook function is already defined for this deployment group');
        }
        this.preHook = preHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.preHook);
        this.preHook.grantInvoke(this.role);
    }
    /**
     * Associate a function to run after deployment completes.
     *
     * @param postHook function to run after deployment completes.
     * @stability stable
     * @throws an error if a post-hook function is already configured
     */
    addPostHook(postHook) {
        if (this.postHook !== undefined) {
            throw new Error('A post-hook function is already defined for this deployment group');
        }
        this.postHook = postHook;
        this.grantPutLifecycleEventHookExecutionStatus(this.postHook);
        this.postHook.grantInvoke(this.role);
    }
    /**
     * Grant a principal permission to codedeploy:PutLifecycleEventHookExecutionStatus on this deployment group resource.
     *
     * @param grantee to grant permission to.
     * @stability stable
     */
    grantPutLifecycleEventHookExecutionStatus(grantee) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.deploymentGroupArn],
            actions: ['codedeploy:PutLifecycleEventHookExecutionStatus'],
        });
    }
}
exports.LambdaDeploymentGroup = LambdaDeploymentGroup;
_a = JSII_RTTI_SYMBOL_1;
LambdaDeploymentGroup[_a] = { fqn: "@aws-cdk/aws-codedeploy.LambdaDeploymentGroup", version: "1.99.0" };
class ImportedLambdaDeploymentGroup extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
        this.deploymentConfig = props.deploymentConfig || deployment_config_1.LambdaDeploymentConfig.CANARY_10PERCENT_5MINUTES;
    }
}
//# sourceMappingURL=data:application/json;base64,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