"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenerateCSV = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_ = require("aws-cdk-lib/aws-lambda");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Generates a output based on Textract Forms and Queries.
 * Supported output_types: "LINES" | "CSV"
 *
 * Input: "Payload"."textract_result"."TextractOutputJsonPath"
 * Output: "TextractOutputCSVPath" TODO: rename
 *
 *
 * Output as LINES
 * Example (Python)
 * ```python
 *         generate_text = tcdk.TextractGenerateCSV(
            self,
            "GenerateText",
            csv_s3_output_bucket=document_bucket.bucket_name,
            csv_s3_output_prefix=s3_txt_output_prefix,
            output_type='LINES',
            lambda_log_level="DEBUG",
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            input=sfn.TaskInput.from_object({
                "Token":
                sfn.JsonPath.task_token,
                "ExecutionId":
                sfn.JsonPath.string_at('$$.Execution.Id'),
                "Payload":
                sfn.JsonPath.entire_payload,
            }),
            result_path="$.txt_output_location")
   ```
 */
class TextractGenerateCSV extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 300 : props.lambdaTimeout;
        var lambdaMemoryMB = props.lambdaMemoryMB === undefined ? 1048 : props.lambdaMemoryMB;
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var outputType = props.outputType === undefined ? 'CSV' : props.outputType;
        var metaDataToAppend = props.metaDataToAppend === undefined ? '' : props.metaDataToAppend;
        const csvGeneratorFunction = new lambda_.DockerImageFunction(this, 'TextractCSVGenerator', {
            code: lambda_.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/generatecsv/')),
            memorySize: lambdaMemoryMB,
            architecture: lambda_.Architecture.X86_64,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                CSV_S3_OUTPUT_BUCKET: props.csvS3OutputBucket,
                CSV_S3_OUTPUT_PREFIX: props.csvS3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                OUTPUT_TYPE: outputType,
                TEXTRACT_API: textractAPI,
                META_DATA_TO_APPEND: metaDataToAppend?.toString(),
            },
        });
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:PutObject', 's3:Get*', 's3:List*'], resources: ['*'] }));
        csvGeneratorFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskSuccess', 'states:SendTaskFailure',
            ],
            resources: ['*'],
        }));
        this.generateCSVLambda = csvGeneratorFunction;
        this.generateCSVLogGroup = csvGeneratorFunction.logGroup;
        const csvGeneratorLambdaInvoke = new tasks.LambdaInvoke(this, 'csvGeneratorInvoke', {
            lambdaFunction: csvGeneratorFunction,
        });
        csvGeneratorLambdaInvoke.addCatch(new sfn.Fail(this, 'csvGenerationFailure'), {
            errors: [sfn.Errors.ALL],
        });
        const workflow_chain = sfn.Chain.start(csvGeneratorLambdaInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            tracingEnabled: true,
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
           * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
           *
           * the scoped access policy should be generated accordingly.
           *
           * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
           */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenerateCSV = TextractGenerateCSV;
_a = JSII_RTTI_SYMBOL_1;
TextractGenerateCSV[_a] = { fqn: "amazon-textract-idp-cdk-constructs.TextractGenerateCSV", version: "0.0.13" };
TextractGenerateCSV.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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