"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    /**
     * The ARN of the key.
     */
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    /**
     * The ID of the key (the part that looks something like: 1234abcd-12ab-34cd-56ef-1234567890ab).
     */
    get keyId() {
        return this.aliasName;
    }
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    /**
     * Adds a statement to the KMS key resource policy.
     */
    addToResourcePolicy(statement, allowNoOp) {
        return this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    /**
     * Grant the indicated permissions on this key to the given principal.
     */
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    /**
     * Grant decryption permisisons using this key to the given principal.
     */
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    /**
     * Grant encryption permisisons using this key to the given principal.
     */
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal.
     */
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management Service (AWS KMS).
 *
 * Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    /**
     *
     */
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error('Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-');
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn,
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs the properties of the referenced KMS Alias.
     */
    static fromAliasAttributes(scope, id, attrs) {
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    /**
     * Import an existing KMS Alias defined outside the CDK app, by the alias name.
     *
     * This method should be used
     * instead of 'fromAliasAttributes' when the underlying KMS Key ARN is not available.
     * This Alias will not have a direct reference to the KMS Key, so addAlias and grant* methods are not supported.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param aliasName The full name of the KMS Alias (e.g., 'alias/aws/s3', 'alias/myKeyAlias').
     */
    static fromAliasName(scope, id, aliasName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.keyArn = core_1.Stack.of(this).formatArn({ service: 'kms', resource: aliasName });
                this.keyId = aliasName;
                this.aliasName = aliasName;
            }
            get aliasTargetKey() { throw new Error('Cannot access aliasTargetKey on an Alias imnported by Alias.fromAliasName().'); }
            addAlias(_alias) { throw new Error('Cannot call addAlias on an Alias imported by Alias.fromAliasName().'); }
            addToResourcePolicy(_statement, _allowNoOp) {
                return { statementAdded: false };
            }
            grant(grantee, ..._actions) { return iam.Grant.drop(grantee, ''); }
            grantDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncrypt(grantee) { return iam.Grant.drop(grantee, ''); }
            grantEncryptDecrypt(grantee) { return iam.Grant.drop(grantee, ''); }
        }
        return new Import(scope, id);
    }
    /**
     *
     */
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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