"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeFunction = exports.InvocationType = exports.LogType = exports.AwsApiCall = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("aws-cdk-lib/core");
const api_call_base_1 = require("./api-call-base");
const providers_1 = require("./providers");
const waiter_state_machine_1 = require("./waiter-state-machine");
/**
 * Construct that creates a custom resource that will perform
 * a query using the AWS SDK
 */
class AwsApiCall extends api_call_base_1.ApiCallBase {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@aws-cdk/integ-tests-alpha.AwsApiCall", version: "2.230.0-alpha.0" };
    provider;
    /**
     * access the AssertionsProvider for the waiter state machine.
     * This can be used to add additional IAM policies
     * the the provider role policy
     *
     * @example
     * declare const apiCall: AwsApiCall;
     * apiCall.waiterProvider?.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: ['s3:GetObject'],
     *   Resource: ['*'],
     * });
     */
    waiterProvider;
    apiCallResource;
    name;
    _assertAtPath;
    api;
    service;
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AwsApiCallProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AwsApiCall);
            }
            throw error;
        }
        this.provider = new providers_1.AssertionsProvider(this, 'SdkProvider', {
            logRetention: props.parameters?.RetentionDays,
        });
        this.provider.addPolicyStatementFromSdkCall(props.service, props.api);
        this.name = `${props.service}${props.api}`;
        this.api = props.api;
        this.service = props.service;
        if (props.outputPaths) {
            this.outputPaths = [...props.outputPaths];
        }
        this.apiCallResource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.provider.serviceToken,
            properties: {
                service: props.service,
                api: props.api,
                expected: core_1.Lazy.any({ produce: () => this.expectedResult }),
                actualPath: core_1.Lazy.string({ produce: () => this._assertAtPath }),
                stateMachineArn: core_1.Lazy.string({ produce: () => this.stateMachineArn }),
                parameters: this.provider.encode(props.parameters),
                flattenResponse: core_1.Lazy.string({ produce: () => this.flattenResponse }),
                outputPaths: core_1.Lazy.list({ produce: () => this.outputPaths }),
                salt: Date.now().toString(),
            },
            // Remove the slash from the resource type because when using the v3 package name as the service name,
            // the `service` props includes the slash, but the resource type name cannot contain the slash
            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cloudformation-customresource.html#aws-resource-cloudformation-customresource--remarks
            resourceType: `${providers_1.SDK_RESOURCE_TYPE_PREFIX}${this.name}`.substring(0, 60).replace(/[\/]/g, ''),
        });
        // Needed so that all the policies set up by the provider should be available before the custom resource is provisioned.
        this.apiCallResource.node.addDependency(this.provider);
        // if expectedResult has been configured then that means
        // we are making assertions and we should output the results
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall) {
                    if (node.expectedResult) {
                        const result = node.apiCallResource.getAttString('assertion');
                        new core_1.CfnOutput(node, 'AssertionResults', {
                            value: result,
                            // Remove the at sign, slash, and hyphen because when using the v3 package name or client name as the service name,
                            // the `id` includes them, but they are not allowed in the `CfnOutput` logical id
                            // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/outputs-section-structure.html#outputs-section-syntax
                        }).overrideLogicalId(`AssertionResults${id}`.replace(/[\@\/\-]/g, ''));
                    }
                }
            },
        }, { priority: core_1.AspectPriority.MUTATING });
    }
    assertAtPath(path, expected) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_ExpectedResult(expected);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.assertAtPath);
            }
            throw error;
        }
        this._assertAtPath = path;
        (this.outputPaths ??= []).push(path);
        this.expectedResult = expected.result;
        this.flattenResponse = 'true';
        return this;
    }
    waitForAssertions(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.waitForAssertions);
            }
            throw error;
        }
        const waiter = new waiter_state_machine_1.WaiterStateMachine(this, 'WaitFor', {
            ...options,
        });
        this.stateMachineArn = waiter.stateMachineArn;
        this.provider.addPolicyStatementFromSdkCall('states', 'StartExecution');
        waiter.isCompleteProvider.addPolicyStatementFromSdkCall(this.service, this.api);
        this.waiterProvider = waiter.isCompleteProvider;
        return this;
    }
}
exports.AwsApiCall = AwsApiCall;
/**
 * Set to Tail to include the execution log in the response.
 * Applies to synchronously invoked functions only.
 */
var LogType;
(function (LogType) {
    /**
     * The log messages are not returned in the response
     */
    LogType["NONE"] = "None";
    /**
     * The log messages are returned in the response
     */
    LogType["TAIL"] = "Tail";
})(LogType || (exports.LogType = LogType = {}));
/**
 * The type of invocation. Default is REQUEST_RESPONSE
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke the function asynchronously.
     * Send events that fail multiple times to the function's
     * dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * Invoke the function synchronously.
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Validate parameter values and verify that the user
     * or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType || (exports.InvocationType = InvocationType = {}));
/**
 * An AWS Lambda Invoke function API call.
 * Use this instead of the generic AwsApiCall in order to
 * invoke a lambda function. This will automatically create
 * the correct permissions to invoke the function
 */
class LambdaInvokeFunction extends AwsApiCall {
    static [JSII_RTTI_SYMBOL_1] = { fqn: "@aws-cdk/integ-tests-alpha.LambdaInvokeFunction", version: "2.230.0-alpha.0" };
    constructor(scope, id, props) {
        super(scope, id, {
            api: 'invoke',
            service: 'Lambda',
            parameters: {
                FunctionName: props.functionName,
                InvocationType: props.invocationType,
                LogType: props.logType,
                Payload: props.payload,
                RetentionDays: props.logRetention,
            },
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_LambdaInvokeFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LambdaInvokeFunction);
            }
            throw error;
        }
        const stack = core_1.Stack.of(this);
        // need to give the assertion lambda permission to invoke
        new core_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: props.functionName,
                Principal: this.provider.handlerRoleArn,
            },
        });
        // the api call is 'invoke', but the permission is 'invokeFunction'
        // so need to handle it specially
        this.provider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                service: 'lambda',
                resource: 'function',
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                resourceName: props.functionName,
            })]);
        // If using `waitForAssertions`, do the same for `waiterProvider` as above.
        // Aspects are used here because we do not know if the user is using `waitForAssertions` at this point.
        core_1.Aspects.of(this).add({
            visit(node) {
                if (node instanceof AwsApiCall && node.waiterProvider) {
                    node.waiterProvider.addPolicyStatementFromSdkCall('Lambda', 'invokeFunction', [stack.formatArn({
                            service: 'lambda',
                            resource: 'function',
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                            resourceName: props.functionName,
                        })]);
                }
            },
        }, { priority: core_1.AspectPriority.MUTATING });
    }
}
exports.LambdaInvokeFunction = LambdaInvokeFunction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2RrLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic2RrLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMkNBQTJIO0FBRTNILG1EQUF3RDtBQUV4RCwyQ0FBMkU7QUFDM0UsaUVBQXVGO0FBeUN2Rjs7O0dBR0c7QUFDSCxNQUFhLFVBQVcsU0FBUSwyQkFBVzs7SUFDekIsUUFBUSxDQUFxQjtJQUU3Qzs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxjQUFjLENBQXNCO0lBRXhCLGVBQWUsQ0FBaUI7SUFDbEMsSUFBSSxDQUFTO0lBRXRCLGFBQWEsQ0FBVTtJQUNkLEdBQUcsQ0FBUztJQUNaLE9BQU8sQ0FBUztJQUVqQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Ozs7OzsrQ0ExQlIsVUFBVTs7OztRQTRCbkIsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLDhCQUFrQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDMUQsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVLEVBQUUsYUFBYTtTQUM5QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxDQUFDLDZCQUE2QixDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxJQUFJLEdBQUcsR0FBRyxLQUFLLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7UUFDckIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQ3RCLElBQUksQ0FBQyxXQUFXLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM1QyxDQUFDO1FBRUQsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLHFCQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUN6RCxZQUFZLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZO1lBQ3hDLFVBQVUsRUFBRTtnQkFDVixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3RCLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztnQkFDZCxRQUFRLEVBQUUsV0FBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFLENBQUM7Z0JBQzFELFVBQVUsRUFBRSxXQUFJLENBQUMsTUFBTSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQztnQkFDOUQsZUFBZSxFQUFFLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO2dCQUNyRSxVQUFVLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztnQkFDbEQsZUFBZSxFQUFFLFdBQUksQ0FBQyxNQUFNLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO2dCQUNyRSxXQUFXLEVBQUUsV0FBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7Z0JBQzNELElBQUksRUFBRSxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFO2FBQzVCO1lBQ0Qsc0dBQXNHO1lBQ3RHLDhGQUE4RjtZQUM5Rix5S0FBeUs7WUFDekssWUFBWSxFQUFFLEdBQUcsb0NBQXdCLEdBQUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7U0FDOUYsQ0FBQyxDQUFDO1FBQ0gsd0hBQXdIO1FBQ3hILElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFdkQsd0RBQXdEO1FBQ3hELDREQUE0RDtRQUM1RCxjQUFPLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQztZQUNuQixLQUFLLENBQUMsSUFBZ0I7Z0JBQ3BCLElBQUksSUFBSSxZQUFZLFVBQVUsRUFBRSxDQUFDO29CQUMvQixJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUUsQ0FBQzt3QkFDeEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxZQUFZLENBQUMsV0FBVyxDQUFDLENBQUM7d0JBRTlELElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7NEJBQ3RDLEtBQUssRUFBRSxNQUFNOzRCQUNiLG1IQUFtSDs0QkFDbkgsaUZBQWlGOzRCQUNqRiwySEFBMkg7eUJBQzVILENBQUMsQ0FBQyxpQkFBaUIsQ0FBQyxtQkFBbUIsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO29CQUN6RSxDQUFDO2dCQUNILENBQUM7WUFDSCxDQUFDO1NBQ0YsRUFBRSxFQUFFLFFBQVEsRUFBRSxxQkFBYyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDM0M7SUFFTSxZQUFZLENBQUMsSUFBWSxFQUFFLFFBQXdCOzs7Ozs7Ozs7O1FBQ3hELElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDO1FBQzFCLENBQUMsSUFBSSxDQUFDLFdBQVcsS0FBSyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLGNBQWMsR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxlQUFlLEdBQUcsTUFBTSxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0tBQ2I7SUFFTSxpQkFBaUIsQ0FBQyxPQUFtQzs7Ozs7Ozs7OztRQUMxRCxNQUFNLE1BQU0sR0FBRyxJQUFJLHlDQUFrQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDckQsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGVBQWUsR0FBRyxNQUFNLENBQUMsZUFBZSxDQUFDO1FBQzlDLElBQUksQ0FBQyxRQUFRLENBQUMsNkJBQTZCLENBQUMsUUFBUSxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFDeEUsTUFBTSxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2hGLElBQUksQ0FBQyxjQUFjLEdBQUcsTUFBTSxDQUFDLGtCQUFrQixDQUFDO1FBQ2hELE9BQU8sSUFBSSxDQUFDO0tBQ2I7O0FBakdILGdDQWtHQztBQUVEOzs7R0FHRztBQUNILElBQVksT0FVWDtBQVZELFdBQVksT0FBTztJQUNqQjs7T0FFRztJQUNILHdCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILHdCQUFhLENBQUE7QUFDZixDQUFDLEVBVlcsT0FBTyx1QkFBUCxPQUFPLFFBVWxCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLGNBcUJYO0FBckJELFdBQVksY0FBYztJQUN4Qjs7Ozs7T0FLRztJQUNILGlDQUFlLENBQUE7SUFFZjs7OztPQUlHO0lBQ0gsc0RBQW9DLENBQUE7SUFFcEM7OztPQUdHO0lBQ0gsb0NBQWtCLENBQUE7QUFDcEIsQ0FBQyxFQXJCVyxjQUFjLDhCQUFkLGNBQWMsUUFxQnpCO0FBd0NEOzs7OztHQUtHO0FBQ0gsTUFBYSxvQkFBcUIsU0FBUSxVQUFVOztJQUNsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWdDO1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsR0FBRyxFQUFFLFFBQVE7WUFDYixPQUFPLEVBQUUsUUFBUTtZQUNqQixVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUNoQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7Z0JBQ3BDLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO2dCQUN0QixhQUFhLEVBQUUsS0FBSyxDQUFDLFlBQVk7YUFDbEM7U0FDRixDQUFDLENBQUM7Ozs7OzsrQ0FaTSxvQkFBb0I7Ozs7UUFjN0IsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3Qix5REFBeUQ7UUFDekQsSUFBSSxrQkFBVyxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUU7WUFDOUIsSUFBSSxFQUFFLHlCQUF5QjtZQUMvQixVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLHVCQUF1QjtnQkFDL0IsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFZO2dCQUNoQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxjQUFjO2FBQ3hDO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsbUVBQW1FO1FBQ25FLGlDQUFpQztRQUNqQyxJQUFJLENBQUMsUUFBUSxDQUFDLDZCQUE2QixDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7Z0JBQ3ZGLE9BQU8sRUFBRSxRQUFRO2dCQUNqQixRQUFRLEVBQUUsVUFBVTtnQkFDcEIsU0FBUyxFQUFFLGdCQUFTLENBQUMsbUJBQW1CO2dCQUN4QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7YUFDakMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVMLDJFQUEyRTtRQUMzRSx1R0FBdUc7UUFDdkcsY0FBTyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUM7WUFDbkIsS0FBSyxDQUFDLElBQWdCO2dCQUNwQixJQUFJLElBQUksWUFBWSxVQUFVLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRSxDQUFDO29CQUN0RCxJQUFJLENBQUMsY0FBYyxDQUFDLDZCQUE2QixDQUFDLFFBQVEsRUFBRSxnQkFBZ0IsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7NEJBQzdGLE9BQU8sRUFBRSxRQUFROzRCQUNqQixRQUFRLEVBQUUsVUFBVTs0QkFDcEIsU0FBUyxFQUFFLGdCQUFTLENBQUMsbUJBQW1COzRCQUN4QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7eUJBQ2pDLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ1AsQ0FBQztZQUNILENBQUM7U0FDRixFQUFFLEVBQUUsUUFBUSxFQUFFLHFCQUFjLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztLQUMzQzs7QUFoREgsb0RBaURDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXJuRm9ybWF0LCBDZm5SZXNvdXJjZSwgQ3VzdG9tUmVzb3VyY2UsIExhenksIFN0YWNrLCBBc3BlY3RzLCBDZm5PdXRwdXQsIEFzcGVjdFByaW9yaXR5IH0gZnJvbSAnYXdzLWNkay1saWIvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IEFwaUNhbGxCYXNlLCBJQXBpQ2FsbCB9IGZyb20gJy4vYXBpLWNhbGwtYmFzZSc7XG5pbXBvcnQgeyBFeHBlY3RlZFJlc3VsdCB9IGZyb20gJy4vY29tbW9uJztcbmltcG9ydCB7IEFzc2VydGlvbnNQcm92aWRlciwgU0RLX1JFU09VUkNFX1RZUEVfUFJFRklYIH0gZnJvbSAnLi9wcm92aWRlcnMnO1xuaW1wb3J0IHsgV2FpdGVyU3RhdGVNYWNoaW5lLCBXYWl0ZXJTdGF0ZU1hY2hpbmVPcHRpb25zIH0gZnJvbSAnLi93YWl0ZXItc3RhdGUtbWFjaGluZSc7XG5pbXBvcnQgeyBSZXRlbnRpb25EYXlzIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuXG4vKipcbiAqIE9wdGlvbnMgdG8gcGVyZm9ybSBhbiBBV1MgSmF2YVNjcmlwdCBWMiBBUEkgY2FsbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF3c0FwaUNhbGxPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBBV1Mgc2VydmljZSwgaS5lLiBTM1xuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgYXBpIGNhbGwgdG8gbWFrZSwgaS5lLiBnZXRCdWNrZXRMaWZlY3ljbGVcbiAgICovXG4gIHJlYWRvbmx5IGFwaTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbnkgcGFyYW1ldGVycyB0byBwYXNzIHRvIHRoZSBhcGkgY2FsbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHBhcmFtZXRlcnNcbiAgICovXG4gIHJlYWRvbmx5IHBhcmFtZXRlcnM/OiBhbnk7XG5cbiAgLyoqXG4gICAqIFJlc3RyaWN0IHRoZSBkYXRhIHJldHVybmVkIGJ5IHRoZSBBUEkgY2FsbCB0byBzcGVjaWZpYyBwYXRocyBpblxuICAgKiB0aGUgQVBJIHJlc3BvbnNlLiBVc2UgdGhpcyB0byBsaW1pdCB0aGUgZGF0YSByZXR1cm5lZCBieSB0aGUgY3VzdG9tXG4gICAqIHJlc291cmNlIGlmIHdvcmtpbmcgd2l0aCBBUEkgY2FsbHMgdGhhdCBjb3VsZCBwb3RlbnRpYWxseSByZXN1bHQgaW4gY3VzdG9tXG4gICAqIHJlc3BvbnNlIG9iamVjdHMgZXhjZWVkaW5nIHRoZSBoYXJkIGxpbWl0IG9mIDQwOTYgYnl0ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcmV0dXJuIGFsbCBkYXRhXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRQYXRocz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIENvbnN0cnVjdCB0aGF0IGNyZWF0ZXMgYSBjdXN0b20gcmVzb3VyY2UgdGhhdCB3aWxsIHBlcmZvcm1cbiAqIGEgcXVlcnkgdXNpbmcgdGhlIEFXUyBTREtcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBd3NBcGlDYWxsUHJvcHMgZXh0ZW5kcyBBd3NBcGlDYWxsT3B0aW9ucyB7IH1cblxuLyoqXG4gKiBDb25zdHJ1Y3QgdGhhdCBjcmVhdGVzIGEgY3VzdG9tIHJlc291cmNlIHRoYXQgd2lsbCBwZXJmb3JtXG4gKiBhIHF1ZXJ5IHVzaW5nIHRoZSBBV1MgU0RLXG4gKi9cbmV4cG9ydCBjbGFzcyBBd3NBcGlDYWxsIGV4dGVuZHMgQXBpQ2FsbEJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvdmlkZXI6IEFzc2VydGlvbnNQcm92aWRlcjtcblxuICAvKipcbiAgICogYWNjZXNzIHRoZSBBc3NlcnRpb25zUHJvdmlkZXIgZm9yIHRoZSB3YWl0ZXIgc3RhdGUgbWFjaGluZS5cbiAgICogVGhpcyBjYW4gYmUgdXNlZCB0byBhZGQgYWRkaXRpb25hbCBJQU0gcG9saWNpZXNcbiAgICogdGhlIHRoZSBwcm92aWRlciByb2xlIHBvbGljeVxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKiBkZWNsYXJlIGNvbnN0IGFwaUNhbGw6IEF3c0FwaUNhbGw7XG4gICAqIGFwaUNhbGwud2FpdGVyUHJvdmlkZXI/LmFkZFRvUm9sZVBvbGljeSh7XG4gICAqICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgKiAgIEFjdGlvbjogWydzMzpHZXRPYmplY3QnXSxcbiAgICogICBSZXNvdXJjZTogWycqJ10sXG4gICAqIH0pO1xuICAgKi9cbiAgcHVibGljIHdhaXRlclByb3ZpZGVyPzogQXNzZXJ0aW9uc1Byb3ZpZGVyO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlDYWxsUmVzb3VyY2U6IEN1c3RvbVJlc291cmNlO1xuICBwcml2YXRlIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICBwcml2YXRlIF9hc3NlcnRBdFBhdGg/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgYXBpOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgc2VydmljZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBd3NBcGlDYWxsUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5wcm92aWRlciA9IG5ldyBBc3NlcnRpb25zUHJvdmlkZXIodGhpcywgJ1Nka1Byb3ZpZGVyJywge1xuICAgICAgbG9nUmV0ZW50aW9uOiBwcm9wcy5wYXJhbWV0ZXJzPy5SZXRlbnRpb25EYXlzLFxuICAgIH0pO1xuICAgIHRoaXMucHJvdmlkZXIuYWRkUG9saWN5U3RhdGVtZW50RnJvbVNka0NhbGwocHJvcHMuc2VydmljZSwgcHJvcHMuYXBpKTtcbiAgICB0aGlzLm5hbWUgPSBgJHtwcm9wcy5zZXJ2aWNlfSR7cHJvcHMuYXBpfWA7XG4gICAgdGhpcy5hcGkgPSBwcm9wcy5hcGk7XG4gICAgdGhpcy5zZXJ2aWNlID0gcHJvcHMuc2VydmljZTtcbiAgICBpZiAocHJvcHMub3V0cHV0UGF0aHMpIHtcbiAgICAgIHRoaXMub3V0cHV0UGF0aHMgPSBbLi4ucHJvcHMub3V0cHV0UGF0aHNdO1xuICAgIH1cblxuICAgIHRoaXMuYXBpQ2FsbFJlc291cmNlID0gbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdEZWZhdWx0Jywge1xuICAgICAgc2VydmljZVRva2VuOiB0aGlzLnByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgc2VydmljZTogcHJvcHMuc2VydmljZSxcbiAgICAgICAgYXBpOiBwcm9wcy5hcGksXG4gICAgICAgIGV4cGVjdGVkOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuZXhwZWN0ZWRSZXN1bHQgfSksXG4gICAgICAgIGFjdHVhbFBhdGg6IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5fYXNzZXJ0QXRQYXRoIH0pLFxuICAgICAgICBzdGF0ZU1hY2hpbmVBcm46IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5zdGF0ZU1hY2hpbmVBcm4gfSksXG4gICAgICAgIHBhcmFtZXRlcnM6IHRoaXMucHJvdmlkZXIuZW5jb2RlKHByb3BzLnBhcmFtZXRlcnMpLFxuICAgICAgICBmbGF0dGVuUmVzcG9uc2U6IExhenkuc3RyaW5nKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5mbGF0dGVuUmVzcG9uc2UgfSksXG4gICAgICAgIG91dHB1dFBhdGhzOiBMYXp5Lmxpc3QoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLm91dHB1dFBhdGhzIH0pLFxuICAgICAgICBzYWx0OiBEYXRlLm5vdygpLnRvU3RyaW5nKCksXG4gICAgICB9LFxuICAgICAgLy8gUmVtb3ZlIHRoZSBzbGFzaCBmcm9tIHRoZSByZXNvdXJjZSB0eXBlIGJlY2F1c2Ugd2hlbiB1c2luZyB0aGUgdjMgcGFja2FnZSBuYW1lIGFzIHRoZSBzZXJ2aWNlIG5hbWUsXG4gICAgICAvLyB0aGUgYHNlcnZpY2VgIHByb3BzIGluY2x1ZGVzIHRoZSBzbGFzaCwgYnV0IHRoZSByZXNvdXJjZSB0eXBlIG5hbWUgY2Fubm90IGNvbnRhaW4gdGhlIHNsYXNoXG4gICAgICAvLyBTZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLWNsb3VkZm9ybWF0aW9uLWN1c3RvbXJlc291cmNlLmh0bWwjYXdzLXJlc291cmNlLWNsb3VkZm9ybWF0aW9uLWN1c3RvbXJlc291cmNlLS1yZW1hcmtzXG4gICAgICByZXNvdXJjZVR5cGU6IGAke1NES19SRVNPVVJDRV9UWVBFX1BSRUZJWH0ke3RoaXMubmFtZX1gLnN1YnN0cmluZygwLCA2MCkucmVwbGFjZSgvW1xcL10vZywgJycpLFxuICAgIH0pO1xuICAgIC8vIE5lZWRlZCBzbyB0aGF0IGFsbCB0aGUgcG9saWNpZXMgc2V0IHVwIGJ5IHRoZSBwcm92aWRlciBzaG91bGQgYmUgYXZhaWxhYmxlIGJlZm9yZSB0aGUgY3VzdG9tIHJlc291cmNlIGlzIHByb3Zpc2lvbmVkLlxuICAgIHRoaXMuYXBpQ2FsbFJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLnByb3ZpZGVyKTtcblxuICAgIC8vIGlmIGV4cGVjdGVkUmVzdWx0IGhhcyBiZWVuIGNvbmZpZ3VyZWQgdGhlbiB0aGF0IG1lYW5zXG4gICAgLy8gd2UgYXJlIG1ha2luZyBhc3NlcnRpb25zIGFuZCB3ZSBzaG91bGQgb3V0cHV0IHRoZSByZXN1bHRzXG4gICAgQXNwZWN0cy5vZih0aGlzKS5hZGQoe1xuICAgICAgdmlzaXQobm9kZTogSUNvbnN0cnVjdCkge1xuICAgICAgICBpZiAobm9kZSBpbnN0YW5jZW9mIEF3c0FwaUNhbGwpIHtcbiAgICAgICAgICBpZiAobm9kZS5leHBlY3RlZFJlc3VsdCkge1xuICAgICAgICAgICAgY29uc3QgcmVzdWx0ID0gbm9kZS5hcGlDYWxsUmVzb3VyY2UuZ2V0QXR0U3RyaW5nKCdhc3NlcnRpb24nKTtcblxuICAgICAgICAgICAgbmV3IENmbk91dHB1dChub2RlLCAnQXNzZXJ0aW9uUmVzdWx0cycsIHtcbiAgICAgICAgICAgICAgdmFsdWU6IHJlc3VsdCxcbiAgICAgICAgICAgICAgLy8gUmVtb3ZlIHRoZSBhdCBzaWduLCBzbGFzaCwgYW5kIGh5cGhlbiBiZWNhdXNlIHdoZW4gdXNpbmcgdGhlIHYzIHBhY2thZ2UgbmFtZSBvciBjbGllbnQgbmFtZSBhcyB0aGUgc2VydmljZSBuYW1lLFxuICAgICAgICAgICAgICAvLyB0aGUgYGlkYCBpbmNsdWRlcyB0aGVtLCBidXQgdGhleSBhcmUgbm90IGFsbG93ZWQgaW4gdGhlIGBDZm5PdXRwdXRgIGxvZ2ljYWwgaWRcbiAgICAgICAgICAgICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL291dHB1dHMtc2VjdGlvbi1zdHJ1Y3R1cmUuaHRtbCNvdXRwdXRzLXNlY3Rpb24tc3ludGF4XG4gICAgICAgICAgICB9KS5vdmVycmlkZUxvZ2ljYWxJZChgQXNzZXJ0aW9uUmVzdWx0cyR7aWR9YC5yZXBsYWNlKC9bXFxAXFwvXFwtXS9nLCAnJykpO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSxcbiAgICB9LCB7IHByaW9yaXR5OiBBc3BlY3RQcmlvcml0eS5NVVRBVElORyB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhc3NlcnRBdFBhdGgocGF0aDogc3RyaW5nLCBleHBlY3RlZDogRXhwZWN0ZWRSZXN1bHQpOiBJQXBpQ2FsbCB7XG4gICAgdGhpcy5fYXNzZXJ0QXRQYXRoID0gcGF0aDtcbiAgICAodGhpcy5vdXRwdXRQYXRocyA/Pz0gW10pLnB1c2gocGF0aCk7XG4gICAgdGhpcy5leHBlY3RlZFJlc3VsdCA9IGV4cGVjdGVkLnJlc3VsdDtcbiAgICB0aGlzLmZsYXR0ZW5SZXNwb25zZSA9ICd0cnVlJztcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHB1YmxpYyB3YWl0Rm9yQXNzZXJ0aW9ucyhvcHRpb25zPzogV2FpdGVyU3RhdGVNYWNoaW5lT3B0aW9ucyk6IElBcGlDYWxsIHtcbiAgICBjb25zdCB3YWl0ZXIgPSBuZXcgV2FpdGVyU3RhdGVNYWNoaW5lKHRoaXMsICdXYWl0Rm9yJywge1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgICB0aGlzLnN0YXRlTWFjaGluZUFybiA9IHdhaXRlci5zdGF0ZU1hY2hpbmVBcm47XG4gICAgdGhpcy5wcm92aWRlci5hZGRQb2xpY3lTdGF0ZW1lbnRGcm9tU2RrQ2FsbCgnc3RhdGVzJywgJ1N0YXJ0RXhlY3V0aW9uJyk7XG4gICAgd2FpdGVyLmlzQ29tcGxldGVQcm92aWRlci5hZGRQb2xpY3lTdGF0ZW1lbnRGcm9tU2RrQ2FsbCh0aGlzLnNlcnZpY2UsIHRoaXMuYXBpKTtcbiAgICB0aGlzLndhaXRlclByb3ZpZGVyID0gd2FpdGVyLmlzQ29tcGxldGVQcm92aWRlcjtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxufVxuXG4vKipcbiAqIFNldCB0byBUYWlsIHRvIGluY2x1ZGUgdGhlIGV4ZWN1dGlvbiBsb2cgaW4gdGhlIHJlc3BvbnNlLlxuICogQXBwbGllcyB0byBzeW5jaHJvbm91c2x5IGludm9rZWQgZnVuY3Rpb25zIG9ubHkuXG4gKi9cbmV4cG9ydCBlbnVtIExvZ1R5cGUge1xuICAvKipcbiAgICogVGhlIGxvZyBtZXNzYWdlcyBhcmUgbm90IHJldHVybmVkIGluIHRoZSByZXNwb25zZVxuICAgKi9cbiAgTk9ORSA9ICdOb25lJyxcblxuICAvKipcbiAgICogVGhlIGxvZyBtZXNzYWdlcyBhcmUgcmV0dXJuZWQgaW4gdGhlIHJlc3BvbnNlXG4gICAqL1xuICBUQUlMID0gJ1RhaWwnLFxufVxuXG4vKipcbiAqIFRoZSB0eXBlIG9mIGludm9jYXRpb24uIERlZmF1bHQgaXMgUkVRVUVTVF9SRVNQT05TRVxuICovXG5leHBvcnQgZW51bSBJbnZvY2F0aW9uVHlwZSB7XG4gIC8qKlxuICAgKiBJbnZva2UgdGhlIGZ1bmN0aW9uIGFzeW5jaHJvbm91c2x5LlxuICAgKiBTZW5kIGV2ZW50cyB0aGF0IGZhaWwgbXVsdGlwbGUgdGltZXMgdG8gdGhlIGZ1bmN0aW9uJ3NcbiAgICogZGVhZC1sZXR0ZXIgcXVldWUgKGlmIGl0J3MgY29uZmlndXJlZCkuXG4gICAqIFRoZSBBUEkgcmVzcG9uc2Ugb25seSBpbmNsdWRlcyBhIHN0YXR1cyBjb2RlLlxuICAgKi9cbiAgRVZFTlQgPSAnRXZlbnQnLFxuXG4gIC8qKlxuICAgKiBJbnZva2UgdGhlIGZ1bmN0aW9uIHN5bmNocm9ub3VzbHkuXG4gICAqIEtlZXAgdGhlIGNvbm5lY3Rpb24gb3BlbiB1bnRpbCB0aGUgZnVuY3Rpb24gcmV0dXJucyBhIHJlc3BvbnNlIG9yIHRpbWVzIG91dC5cbiAgICogVGhlIEFQSSByZXNwb25zZSBpbmNsdWRlcyB0aGUgZnVuY3Rpb24gcmVzcG9uc2UgYW5kIGFkZGl0aW9uYWwgZGF0YS5cbiAgICovXG4gIFJFUVVFU1RfUkVTUE9OU0UgPSAnUmVxdWVzdFJlc3BvbnNlJyxcblxuICAvKipcbiAgICogVmFsaWRhdGUgcGFyYW1ldGVyIHZhbHVlcyBhbmQgdmVyaWZ5IHRoYXQgdGhlIHVzZXJcbiAgICogb3Igcm9sZSBoYXMgcGVybWlzc2lvbiB0byBpbnZva2UgdGhlIGZ1bmN0aW9uLlxuICAgKi9cbiAgRFJZX1JVTiA9ICdEcnlSdW4nLFxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gcGFzcyB0byB0aGUgTGFtYmRhIGludm9rZUZ1bmN0aW9uIEFQSSBjYWxsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhSW52b2tlRnVuY3Rpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZnVuY3Rpb24gdG8gaW52b2tlXG4gICAqL1xuICByZWFkb25seSBmdW5jdGlvbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgaW52b2NhdGlvbiB0byB1c2VcbiAgICpcbiAgICogQGRlZmF1bHQgSW52b2NhdGlvblR5cGUuUkVRVUVTVF9SRVNQT05TRVxuICAgKi9cbiAgcmVhZG9ubHkgaW52b2NhdGlvblR5cGU/OiBJbnZvY2F0aW9uVHlwZTtcblxuICAvKipcbiAgICogV2hldGhlciB0byByZXR1cm4gdGhlIGxvZ3MgYXMgcGFydCBvZiB0aGUgcmVzcG9uc2VcbiAgICpcbiAgICogQGRlZmF1bHQgTG9nVHlwZS5OT05FXG4gICAqL1xuICByZWFkb25seSBsb2dUeXBlPzogTG9nVHlwZTtcblxuICAvKipcbiAgICogSG93IGxvbmcsIGluIGRheXMsIHRoZSBsb2cgY29udGVudHMgd2lsbCBiZSByZXRhaW5lZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyByZXRlbnRpb24gZGF5cyBzcGVjaWZpZWRcbiAgICovXG4gIHJlYWRvbmx5IGxvZ1JldGVudGlvbj86IFJldGVudGlvbkRheXM7XG5cbiAgLyoqXG4gICAqIFBheWxvYWQgdG8gc2VuZCBhcyBwYXJ0IG9mIHRoZSBpbnZva2VcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBwYXlsb2FkXG4gICAqL1xuICByZWFkb25seSBwYXlsb2FkPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEFuIEFXUyBMYW1iZGEgSW52b2tlIGZ1bmN0aW9uIEFQSSBjYWxsLlxuICogVXNlIHRoaXMgaW5zdGVhZCBvZiB0aGUgZ2VuZXJpYyBBd3NBcGlDYWxsIGluIG9yZGVyIHRvXG4gKiBpbnZva2UgYSBsYW1iZGEgZnVuY3Rpb24uIFRoaXMgd2lsbCBhdXRvbWF0aWNhbGx5IGNyZWF0ZVxuICogdGhlIGNvcnJlY3QgcGVybWlzc2lvbnMgdG8gaW52b2tlIHRoZSBmdW5jdGlvblxuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhSW52b2tlRnVuY3Rpb24gZXh0ZW5kcyBBd3NBcGlDYWxsIHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IExhbWJkYUludm9rZUZ1bmN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIGFwaTogJ2ludm9rZScsXG4gICAgICBzZXJ2aWNlOiAnTGFtYmRhJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgRnVuY3Rpb25OYW1lOiBwcm9wcy5mdW5jdGlvbk5hbWUsXG4gICAgICAgIEludm9jYXRpb25UeXBlOiBwcm9wcy5pbnZvY2F0aW9uVHlwZSxcbiAgICAgICAgTG9nVHlwZTogcHJvcHMubG9nVHlwZSxcbiAgICAgICAgUGF5bG9hZDogcHJvcHMucGF5bG9hZCxcbiAgICAgICAgUmV0ZW50aW9uRGF5czogcHJvcHMubG9nUmV0ZW50aW9uLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG4gICAgLy8gbmVlZCB0byBnaXZlIHRoZSBhc3NlcnRpb24gbGFtYmRhIHBlcm1pc3Npb24gdG8gaW52b2tlXG4gICAgbmV3IENmblJlc291cmNlKHRoaXMsICdJbnZva2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpMYW1iZGE6OlBlcm1pc3Npb24nLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBBY3Rpb246ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nLFxuICAgICAgICBGdW5jdGlvbk5hbWU6IHByb3BzLmZ1bmN0aW9uTmFtZSxcbiAgICAgICAgUHJpbmNpcGFsOiB0aGlzLnByb3ZpZGVyLmhhbmRsZXJSb2xlQXJuLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIHRoZSBhcGkgY2FsbCBpcyAnaW52b2tlJywgYnV0IHRoZSBwZXJtaXNzaW9uIGlzICdpbnZva2VGdW5jdGlvbidcbiAgICAvLyBzbyBuZWVkIHRvIGhhbmRsZSBpdCBzcGVjaWFsbHlcbiAgICB0aGlzLnByb3ZpZGVyLmFkZFBvbGljeVN0YXRlbWVudEZyb21TZGtDYWxsKCdMYW1iZGEnLCAnaW52b2tlRnVuY3Rpb24nLCBbc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdsYW1iZGEnLFxuICAgICAgcmVzb3VyY2U6ICdmdW5jdGlvbicsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgcmVzb3VyY2VOYW1lOiBwcm9wcy5mdW5jdGlvbk5hbWUsXG4gICAgfSldKTtcblxuICAgIC8vIElmIHVzaW5nIGB3YWl0Rm9yQXNzZXJ0aW9uc2AsIGRvIHRoZSBzYW1lIGZvciBgd2FpdGVyUHJvdmlkZXJgIGFzIGFib3ZlLlxuICAgIC8vIEFzcGVjdHMgYXJlIHVzZWQgaGVyZSBiZWNhdXNlIHdlIGRvIG5vdCBrbm93IGlmIHRoZSB1c2VyIGlzIHVzaW5nIGB3YWl0Rm9yQXNzZXJ0aW9uc2AgYXQgdGhpcyBwb2ludC5cbiAgICBBc3BlY3RzLm9mKHRoaXMpLmFkZCh7XG4gICAgICB2aXNpdChub2RlOiBJQ29uc3RydWN0KSB7XG4gICAgICAgIGlmIChub2RlIGluc3RhbmNlb2YgQXdzQXBpQ2FsbCAmJiBub2RlLndhaXRlclByb3ZpZGVyKSB7XG4gICAgICAgICAgbm9kZS53YWl0ZXJQcm92aWRlci5hZGRQb2xpY3lTdGF0ZW1lbnRGcm9tU2RrQ2FsbCgnTGFtYmRhJywgJ2ludm9rZUZ1bmN0aW9uJywgW3N0YWNrLmZvcm1hdEFybih7XG4gICAgICAgICAgICBzZXJ2aWNlOiAnbGFtYmRhJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAnZnVuY3Rpb24nLFxuICAgICAgICAgICAgYXJuRm9ybWF0OiBBcm5Gb3JtYXQuQ09MT05fUkVTT1VSQ0VfTkFNRSxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogcHJvcHMuZnVuY3Rpb25OYW1lLFxuICAgICAgICAgIH0pXSk7XG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgfSwgeyBwcmlvcml0eTogQXNwZWN0UHJpb3JpdHkuTVVUQVRJTkcgfSk7XG4gIH1cbn1cbiJdfQ==