"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceHandler = void 0;
/* eslint-disable no-console */
const https = require("https");
const url = require("url");
const client_sfn_1 = require("@aws-sdk/client-sfn");
class CustomResourceHandler {
    event;
    context;
    physicalResourceId;
    timeout;
    timedOut = false;
    constructor(event, context) {
        this.event = event;
        this.context = context;
        this.timeout = setTimeout(async () => {
            await this.respond({
                status: 'FAILED',
                reason: 'Lambda Function Timeout',
                data: this.context.logStreamName,
            });
            this.timedOut = true;
        }, context.getRemainingTimeInMillis() - 1200);
        this.event = event;
        this.physicalResourceId = extractPhysicalResourceId(event);
    }
    /**
     * Handles executing the custom resource event. If `stateMachineArn` is present
     * in the props then trigger the waiter statemachine
     */
    async handle() {
        try {
            if ('stateMachineArn' in this.event.ResourceProperties) {
                const req = {
                    stateMachineArn: this.event.ResourceProperties.stateMachineArn,
                    name: this.event.RequestId,
                    input: JSON.stringify(this.event),
                };
                await this.startExecution(req);
                return;
            }
            else {
                const response = await this.processEvent(this.event.ResourceProperties);
                return response;
            }
        }
        catch (e) {
            console.log(e);
            throw e;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Handle async requests from the waiter state machine
     */
    async handleIsComplete() {
        try {
            const result = await this.processEvent(this.event.ResourceProperties);
            return result;
        }
        catch (e) {
            console.log(e);
            return;
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    /**
     * Start a step function state machine which will wait for the request
     * to be successful.
     */
    async startExecution(req) {
        try {
            const sfn = new client_sfn_1.SFN({});
            await sfn.startExecution(req);
        }
        finally {
            clearTimeout(this.timeout);
        }
    }
    respond(response) {
        if (this.timedOut) {
            return;
        }
        const cfResponse = {
            Status: response.status,
            Reason: response.reason,
            PhysicalResourceId: this.physicalResourceId,
            StackId: this.event.StackId,
            RequestId: this.event.RequestId,
            LogicalResourceId: this.event.LogicalResourceId,
            NoEcho: false,
            Data: response.data,
        };
        const responseBody = JSON.stringify(cfResponse);
        console.log('Responding to CloudFormation', responseBody);
        const parsedUrl = url.parse(this.event.ResponseURL);
        const requestOptions = {
            hostname: parsedUrl.hostname,
            path: parsedUrl.path,
            method: 'PUT',
            headers: {
                'content-type': '',
                'content-length': Buffer.byteLength(responseBody, 'utf8'),
            },
        };
        return new Promise((resolve, reject) => {
            try {
                const request = https.request(requestOptions, resolve);
                request.on('error', reject);
                request.write(responseBody);
                request.end();
            }
            catch (e) {
                reject(e);
            }
            finally {
                clearTimeout(this.timeout);
            }
        });
    }
}
exports.CustomResourceHandler = CustomResourceHandler;
function extractPhysicalResourceId(event) {
    switch (event.RequestType) {
        case 'Create':
            return event.LogicalResourceId;
        case 'Update':
        case 'Delete':
            return event.PhysicalResourceId;
    }
}
//# sourceMappingURL=data:application/json;base64,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