# Generated E-commerce Checkout Flow Test
# Framework: Playwright Python
# Generated by Browse-to-Test with production optimizations

import asyncio
import pytest
from playwright.async_api import async_playwright, Page, Browser, BrowserContext
from typing import Dict, Any
import os
from datetime import datetime


class TestEcommerceCheckout:
    """E-commerce checkout flow test with comprehensive error handling and assertions."""
    
    @pytest.fixture(scope="session")
    async def browser(self):
        """Set up browser for testing."""
        playwright = await async_playwright().start()
        browser = await playwright.chromium.launch(
            headless=bool(os.getenv("HEADLESS", "true").lower() == "true"),
            args=["--no-sandbox", "--disable-dev-shm-usage"]
        )
        yield browser
        await browser.close()
        await playwright.stop()
    
    @pytest.fixture
    async def context(self, browser: Browser):
        """Create a new browser context for each test."""
        context = await browser.new_context(
            viewport={"width": 1280, "height": 720},
            user_agent="Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
        )
        yield context
        await context.close()
    
    @pytest.fixture
    async def page(self, context: BrowserContext):
        """Create a new page for each test."""
        page = await context.new_page()
        yield page
        await page.close()
    
    @pytest.mark.asyncio
    async def test_complete_checkout_flow(self, page: Page):
        """Test complete e-commerce checkout flow with comprehensive validations."""
        
        try:
            # Step 1: Navigate to demo shop
            await page.goto("https://demo-shop.example.com", timeout=30000)
            await page.wait_for_load_state("networkidle", timeout=10000)
            
            # Verify homepage loaded correctly
            await page.wait_for_selector("[data-testid='product-card']", timeout=10000)
            assert await page.title() == "Demo Shop - Home", "Homepage title incorrect"
            
            # Step 2: Add product to cart
            add_to_cart_btn = page.locator("[data-testid='product-card']:first-child .add-to-cart-btn")
            await add_to_cart_btn.wait_for(state="visible", timeout=10000)
            
            # Verify product information before adding to cart
            product_name = await add_to_cart_btn.get_attribute("aria-label")
            assert "Wireless Headphones" in product_name, "Product name not found"
            
            await add_to_cart_btn.click()
            
            # Step 3: Wait for cart notification
            cart_notification = page.locator(".cart-notification.success")
            await cart_notification.wait_for(state="visible", timeout=5000)
            
            notification_text = await cart_notification.text_content()
            assert "Wireless Headphones added to cart" in notification_text, "Cart notification incorrect"
            
            # Step 4: Navigate to cart
            cart_link = page.locator(".header-cart-link")
            await cart_link.wait_for(state="visible")
            
            # Verify cart count updated
            cart_text = await cart_link.text_content()
            assert "Cart (1)" in cart_text, "Cart count not updated"
            
            await cart_link.click()
            await page.wait_for_url("**/cart", timeout=10000)
            
            # Step 5: Proceed to checkout
            checkout_btn = page.locator(".checkout-btn")
            await checkout_btn.wait_for(state="enabled", timeout=5000)
            await checkout_btn.click()
            
            await page.wait_for_url("**/checkout", timeout=10000)
            
            # Verify checkout page elements
            await page.wait_for_selector("#checkout-email", timeout=10000)
            
            # Step 6-8: Fill out checkout form
            checkout_form_data = {
                "#checkout-email": "john.doe@example.com",
                "#checkout-fullname": "John Doe", 
                "#checkout-address": "123 Main Street"
            }
            
            for selector, value in checkout_form_data.items():
                field = page.locator(selector)
                await field.wait_for(state="visible")
                await field.fill(value)
                
                # Verify field was filled correctly
                filled_value = await field.input_value()
                assert filled_value == value, f"Field {selector} not filled correctly"
            
            # Step 9: Select payment method
            payment_method = page.locator("#payment-method")
            await payment_method.select_option("credit_card")
            
            selected_value = await payment_method.input_value()
            assert selected_value == "credit_card", "Payment method not selected"
            
            # Step 10-12: Fill payment information
            # Note: In real tests, use test credit card numbers
            await page.locator("#card-number").fill("4111111111111111")
            await page.locator("#card-expiry").fill("12/25")
            await page.locator("#card-cvv").fill("123")
            
            # Verify form validation
            card_number_field = page.locator("#card-number")
            is_valid = await card_number_field.evaluate("el => el.checkValidity()")
            assert is_valid, "Credit card number validation failed"
            
            # Step 13: Submit order
            place_order_btn = page.locator(".place-order-btn")
            
            # Verify order total is displayed
            order_total = await place_order_btn.text_content()
            assert "$199.99" in order_total, "Order total not displayed correctly"
            
            # Click place order and handle loading state
            await place_order_btn.click()
            
            # Wait for loading state to appear and disappear
            loading_btn = page.locator(".place-order-btn.loading")
            await loading_btn.wait_for(state="visible", timeout=2000)  # May not appear immediately
            await loading_btn.wait_for(state="hidden", timeout=10000)
            
            # Step 14: Verify order confirmation
            await page.wait_for_url("**/order-confirmation", timeout=15000)
            
            order_confirmation = page.locator(".order-confirmation")
            await order_confirmation.wait_for(state="visible", timeout=10000)
            
            confirmation_text = await order_confirmation.text_content()
            assert "Order Confirmed!" in confirmation_text, "Order confirmation message not found"
            assert "ORD-2024-" in confirmation_text, "Order number not found"
            
            # Verify page title changed
            final_title = await page.title()
            assert "Order Confirmation" in final_title, "Confirmation page title incorrect"
            
            # Additional success validations
            success_elements = await page.locator("[data-testid='order-confirmation']").count()
            assert success_elements > 0, "Order confirmation element not found"
            
            print("✓ E-commerce checkout flow completed successfully")
            
        except Exception as e:
            # Take screenshot on failure for debugging
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            screenshot_path = f"test_failure_checkout_{timestamp}.png"
            await page.screenshot(path=screenshot_path, full_page=True)
            
            # Log page state for debugging
            current_url = page.url
            page_title = await page.title()
            
            print(f"Test failed at URL: {current_url}")
            print(f"Page title: {page_title}")
            print(f"Screenshot saved: {screenshot_path}")
            
            raise AssertionError(f"Checkout flow test failed: {str(e)}")


if __name__ == "__main__":
    # Run the test
    pytest.main([__file__, "-v", "--tb=short"])
