# Generated Dynamic Content Testing Suite
# Framework: Selenium Python
# Generated by Browse-to-Test with advanced SPA and mobile testing capabilities

import time
import json
import unittest
from typing import List, Dict, Any, Optional
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.chrome.options import Options as ChromeOptions
from selenium.webdriver.firefox.options import Options as FirefoxOptions
from selenium.common.exceptions import (
    TimeoutException, NoSuchElementException, 
    StaleElementReferenceException, WebDriverException
)
from selenium.webdriver.common.desired_capabilities import DesiredCapabilities
import logging
from datetime import datetime
import os


class DynamicContentTestSuite(unittest.TestCase):
    """
    Comprehensive test suite for Single Page Applications with dynamic content.
    
    Features tested:
    - AJAX loading and waiting
    - Dynamic content filtering
    - Infinite scroll functionality
    - Modal interactions
    - JavaScript execution
    - Mobile responsiveness
    - Touch interactions
    """
    
    @classmethod
    def setUpClass(cls):
        """Set up logging and test configuration."""
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
        )
        cls.logger = logging.getLogger(__name__)
        
        # Performance tracking
        cls.performance_metrics = {
            'ajax_requests': [],
            'load_times': [],
            'animation_durations': []
        }
    
    def setUp(self):
        """Set up WebDriver with optimized configuration for SPA testing."""
        self.setup_driver()
        self.wait = WebDriverWait(self.driver, 10)
        self.long_wait = WebDriverWait(self.driver, 30)
        self.actions = ActionChains(self.driver)
        
        # Enable performance logging
        self.enable_performance_logging()
        
        # Track network requests
        self.network_requests = []
        
    def tearDown(self):
        """Clean up after each test."""
        if hasattr(self, 'driver'):
            try:
                # Log final performance metrics
                self.log_performance_metrics()
                self.driver.quit()
            except Exception as e:
                self.logger.error(f"Error during cleanup: {e}")
    
    def setup_driver(self):
        """Configure Chrome driver with SPA-optimized settings."""
        chrome_options = ChromeOptions()
        
        # Performance and debugging options
        chrome_options.add_argument('--no-sandbox')
        chrome_options.add_argument('--disable-dev-shm-usage')
        chrome_options.add_argument('--disable-blink-features=AutomationControlled')
        chrome_options.add_experimental_option('useAutomationExtension', False)
        chrome_options.add_experimental_option('excludeSwitches', ['enable-automation'])
        
        # Enable logging for network and performance
        chrome_options.add_argument('--enable-logging')
        chrome_options.add_argument('--log-level=0') 
        
        # Mobile emulation capability
        mobile_emulation = {
            "deviceMetrics": {"width": 375, "height": 667, "pixelRatio": 2.0},
            "userAgent": "Mozilla/5.0 (iPhone; CPU iPhone OS 14_6 like Mac OS X) AppleWebKit/605.1.15"
        }
        
        # Set up capabilities for network monitoring
        caps = DesiredCapabilities.CHROME
        caps['goog:loggingPrefs'] = {'performance': 'ALL', 'browser': 'ALL'}
        
        self.driver = webdriver.Chrome(options=chrome_options, desired_capabilities=caps)
        self.driver.set_window_size(1280, 720)  # Start with desktop
        
        # Set implicit wait for dynamic content
        self.driver.implicitly_wait(5)
        
        # Store mobile emulation config for later use
        self.mobile_config = mobile_emulation
        
    def enable_performance_logging(self):
        """Enable comprehensive performance and network logging."""
        # Inject performance monitoring script
        performance_script = """
        window.performanceData = {
            ajaxRequests: [],
            loadTimes: [],
            networkRequests: []
        };
        
        // Override XMLHttpRequest to track AJAX
        const originalXHR = window.XMLHttpRequest;
        window.XMLHttpRequest = function() {
            const xhr = new originalXHR();
            const originalOpen = xhr.open;
            const originalSend = xhr.send;
            
            xhr.open = function(method, url) {
                xhr._method = method;
                xhr._url = url;
                xhr._startTime = performance.now();
                return originalOpen.apply(this, arguments);
            };
            
            xhr.send = function() {
                const self = this;
                xhr.addEventListener('loadend', function() {
                    const duration = performance.now() - self._startTime;
                    window.performanceData.ajaxRequests.push({
                        method: self._method,
                        url: self._url,
                        status: self.status,
                        duration: duration,
                        timestamp: Date.now()
                    });
                });
                return originalSend.apply(this, arguments);
            };
            
            return xhr;
        };
        
        // Track page load performance
        window.addEventListener('load', function() {
            const perfData = performance.getEntriesByType('navigation')[0];
            window.performanceData.loadTimes.push({
                domContentLoaded: perfData.domContentLoadedEventEnd - perfData.domContentLoadedEventStart,
                loadComplete: perfData.loadEventEnd - perfData.loadEventStart,
                totalTime: perfData.loadEventEnd - perfData.navigationStart
            });
        });
        """
        
        self.driver.execute_script(performance_script)
        
    def wait_for_ajax_complete(self, timeout: int = 10) -> bool:
        """Wait for all AJAX requests to complete."""
        try:
            WebDriverWait(self.driver, timeout).until(
                lambda driver: driver.execute_script(
                    "return (typeof jQuery !== 'undefined' ? jQuery.active == 0 : true) && "
                    "document.readyState === 'complete'"
                )
            )
            return True
        except TimeoutException:
            self.logger.warning(f"AJAX requests did not complete within {timeout} seconds")
            return False
    
    def wait_for_loading_spinner_gone(self, timeout: int = 10) -> bool:
        """Wait for loading spinner to disappear."""
        try:
            spinner = self.driver.find_element(By.CSS_SELECTOR, ".loading-spinner")
            WebDriverWait(self.driver, timeout).until(
                EC.invisibility_of_element(spinner)
            )
            return True
        except (NoSuchElementException, TimeoutException):
            return True  # Spinner not found or already gone
    
    def wait_for_dynamic_elements(self, selector: str, expected_count: int, timeout: int = 10) -> List:
        """Wait for dynamic elements to appear with expected count."""
        try:
            WebDriverWait(self.driver, timeout).until(
                lambda driver: len(driver.find_elements(By.CSS_SELECTOR, selector)) >= expected_count
            )
            return self.driver.find_elements(By.CSS_SELECTOR, selector)
        except TimeoutException:
            current_count = len(self.driver.find_elements(By.CSS_SELECTOR, selector))
            raise AssertionError(
                f"Expected at least {expected_count} elements with selector '{selector}', "
                f"but found {current_count}"
            )
    
    def trigger_infinite_scroll(self, target_new_items: int = 5) -> bool:
        """Trigger infinite scroll and wait for new content."""
        initial_items = len(self.driver.find_elements(By.CSS_SELECTOR, ".content-item"))
        
        # Scroll to bottom
        self.driver.execute_script("window.scrollTo(0, document.body.scrollHeight);")
        
        # Wait for new items to load
        try:
            WebDriverWait(self.driver, 10).until(
                lambda driver: len(driver.find_elements(By.CSS_SELECTOR, ".content-item")) >= 
                              initial_items + target_new_items
            )
            return True
        except TimeoutException:
            current_items = len(self.driver.find_elements(By.CSS_SELECTOR, ".content-item"))
            self.logger.warning(f"Infinite scroll loaded {current_items - initial_items} items, expected {target_new_items}")
            return current_items > initial_items
    
    def switch_to_mobile_viewport(self):
        """Switch to mobile viewport for responsive testing."""
        self.driver.set_window_size(375, 667)
        
        # Wait for responsive layout changes
        time.sleep(1)
        
        # Verify mobile layout is active
        try:
            mobile_indicator = self.wait.until(
                EC.presence_of_element_located((By.CSS_SELECTOR, ".mobile-layout, [data-mobile='true']"))
            )
            return True
        except TimeoutException:
            self.logger.warning("Mobile layout indicator not found")
            return False
    
    def perform_touch_scroll(self, element, direction: str = "down", distance: int = 500):
        """Simulate touch scrolling on mobile devices."""
        if direction == "down":
            self.driver.execute_script(f"""
                arguments[0].scrollTop += {distance};
                
                // Trigger touch events for better mobile simulation
                const touchStart = new TouchEvent('touchstart', {{
                    touches: [new Touch({{
                        identifier: 0,
                        target: arguments[0],
                        clientX: arguments[0].offsetWidth / 2,
                        clientY: arguments[0].offsetHeight / 2
                    }})]
                }});
                
                const touchEnd = new TouchEvent('touchend', {{
                    touches: []
                }});
                
                arguments[0].dispatchEvent(touchStart);
                setTimeout(() => arguments[0].dispatchEvent(touchEnd), 100);
            """, element)
    
    def verify_mobile_responsive_elements(self, selectors: List[str]) -> Dict[str, bool]:
        """Verify elements are properly responsive on mobile."""
        results = {}
        viewport_width = 375  # Mobile width
        
        for selector in selectors:
            try:
                elements = self.driver.find_elements(By.CSS_SELECTOR, selector)
                if not elements:
                    results[selector] = False
                    continue
                
                element = elements[0]
                element_width = element.size['width']
                
                # Check if element fits within mobile viewport (with some tolerance)
                is_responsive = element_width <= viewport_width + 20  # 20px tolerance
                results[selector] = is_responsive
                
            except Exception as e:
                self.logger.error(f"Error checking responsiveness for {selector}: {e}")
                results[selector] = False
        
        return results
    
    def get_performance_metrics(self) -> Dict[str, Any]:
        """Get comprehensive performance metrics from the browser."""
        try:
            return self.driver.execute_script("""
                return {
                    ajaxRequests: window.performanceData.ajaxRequests,
                    loadTimes: window.performanceData.loadTimes,
                    currentPageMetrics: {
                        domElements: document.getElementsByTagName('*').length,
                        scripts: document.scripts.length,
                        stylesheets: document.styleSheets.length
                    },
                    memoryUsage: performance.memory ? {
                        usedJSHeapSize: performance.memory.usedJSHeapSize,
                        totalJSHeapSize: performance.memory.totalJSHeapSize,
                        jsHeapSizeLimit: performance.memory.jsHeapSizeLimit
                    } : null
                };
            """)
        except Exception as e:
            self.logger.error(f"Error getting performance metrics: {e}")
            return {}
    
    def log_performance_metrics(self):
        """Log performance metrics for analysis."""
        metrics = self.get_performance_metrics()
        if metrics:
            self.logger.info(f"Performance Metrics: {json.dumps(metrics, indent=2)}")
    
    def test_dynamic_content_spa_workflow(self):
        """
        Main test for dynamic content SPA workflow.
        
        Tests:
        1. Initial page load and AJAX completion
        2. Content grid loading
        3. Dynamic filtering with AJAX
        4. Infinite scroll functionality
        5. Modal interactions
        6. JavaScript execution
        7. Mobile responsive behavior
        """
        try:
            start_time = time.time()
            
            # Step 1: Navigate to SPA demo
            self.driver.get("https://spa-demo.example.com/")
            
            # Verify initial page load
            self.assertEqual(self.driver.title, "Dynamic Content Demo - SPA")
            
            # Step 2: Wait for AJAX loading to complete
            self.assertTrue(self.wait_for_loading_spinner_gone(timeout=10))
            self.assertTrue(self.wait_for_ajax_complete(timeout=10))
            
            # Step 3: Wait for content grid to load
            content_grid = self.wait.until(
                EC.presence_of_element_located((By.CSS_SELECTOR, "[data-testid='content-grid']"))
            )
            self.assertTrue(content_grid.is_displayed())
            
            # Verify initial content count
            initial_items = self.wait_for_dynamic_elements(".content-item", 10)
            self.assertGreaterEqual(len(initial_items), 10)
            
            # Step 4: Apply technology filter
            tech_filter = self.wait.until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "[data-category='technology']"))
            )
            tech_filter.click()
            
            # Wait for filter results
            self.assertTrue(self.wait_for_ajax_complete(timeout=5))
            filtered_items = self.wait_for_dynamic_elements(".content-item.technology-category", 8)
            self.assertEqual(len(filtered_items), 8)
            
            # Step 5-7: Test infinite scroll
            self.assertTrue(self.trigger_infinite_scroll(target_new_items=5))
            
            # Verify new content loaded
            all_items_after_scroll = self.driver.find_elements(By.CSS_SELECTOR, ".content-item")
            self.assertGreaterEqual(len(all_items_after_scroll), 13)
            
            # Step 8-12: Test modal interaction
            view_details_btn = self.wait.until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "[data-item-id='tech-5'] .view-details-btn"))
            )
            view_details_btn.click()
            
            # Wait for modal to appear
            modal = self.wait.until(
                EC.visibility_of_element_located((By.CSS_SELECTOR, ".detail-modal"))
            )
            self.assertTrue(modal.is_displayed())
            self.assertEqual(modal.get_attribute("aria-modal"), "true")
            
            # Execute JavaScript to get modal content height
            modal_height = self.driver.execute_script(
                "return document.querySelector('.detail-modal .content').scrollHeight"
            )
            self.assertGreater(modal_height, 0)
            self.logger.info(f"Modal content height: {modal_height}px")
            
            # Close modal
            close_btn = self.wait.until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, ".close-modal-btn"))
            )
            close_btn.click()
            
            # Wait for modal to disappear
            self.wait.until(EC.invisibility_of_element_located((By.CSS_SELECTOR, ".detail-modal")))
            
            # Step 13-16: Test mobile responsiveness
            self.switch_to_mobile_viewport()
            
            # Wait for responsive layout
            mobile_grid = self.wait.until(
                EC.presence_of_element_located((By.CSS_SELECTOR, ".content-grid.mobile-layout"))
            )
            self.assertTrue(mobile_grid.is_displayed())
            
            # Test touch scrolling
            self.perform_touch_scroll(mobile_grid, "down", 500)
            
            # Verify mobile responsiveness
            responsive_results = self.verify_mobile_responsive_elements([
                ".content-item", ".filter-btn", ".header"
            ])
            
            for selector, is_responsive in responsive_results.items():
                self.assertTrue(is_responsive, f"Element {selector} is not mobile responsive")
            
            # Performance validation
            final_metrics = self.get_performance_metrics()
            if final_metrics.get('loadTimes'):
                load_time = final_metrics['loadTimes'][0].get('totalTime', 0)
                self.assertLess(load_time, 30000, "Page load time exceeds 30 seconds")  # 30s max
            
            ajax_requests = final_metrics.get('ajaxRequests', [])
            self.assertGreater(len(ajax_requests), 0, "No AJAX requests detected")
            
            # Log successful completion
            total_time = time.time() - start_time
            self.logger.info(f"✓ Dynamic content SPA workflow completed in {total_time:.2f} seconds")
            
            print(f"✓ Successfully tested dynamic SPA with {len(ajax_requests)} AJAX requests")
            print(f"✓ Mobile responsiveness verified for all key elements")
            print(f"✓ Performance metrics within acceptable ranges")
            
        except Exception as e:
            # Enhanced error handling with context
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            screenshot_path = f"spa_test_failure_{timestamp}.png"
            
            try:
                self.driver.save_screenshot(screenshot_path)
                self.logger.error(f"Screenshot saved: {screenshot_path}")
            except Exception:
                pass
            
            # Log detailed error context
            error_context = {
                "current_url": self.driver.current_url,
                "page_title": self.driver.title,
                "window_size": self.driver.get_window_size(),
                "error": str(e),
                "timestamp": timestamp
            }
            
            self.logger.error(f"SPA test failed: {json.dumps(error_context, indent=2)}")
            
            # Try to get final performance data
            try:
                final_metrics = self.get_performance_metrics()
                self.logger.info(f"Final metrics at failure: {json.dumps(final_metrics, indent=2)}")
            except Exception:
                pass
            
            raise AssertionError(f"Dynamic content SPA test failed: {str(e)}")
    
    def test_ajax_error_handling(self):
        """Test handling of AJAX errors and network failures."""
        self.driver.get("https://spa-demo.example.com/")
        
        # Simulate network failure by blocking requests
        self.driver.execute_cdp_cmd('Network.enable', {})
        self.driver.execute_cdp_cmd('Network.setBlockedURLs', {
            'urls': ['**/api/content/filter*']
        })
        
        try:
            # Try to trigger a filter that will fail
            tech_filter = self.wait.until(
                EC.element_to_be_clickable((By.CSS_SELECTOR, "[data-category='technology']"))
            )
            tech_filter.click()
            
            # Check for error handling UI
            error_message = self.wait.until(
                EC.presence_of_element_located((By.CSS_SELECTOR, ".error-message, .ajax-error"))
            )
            self.assertTrue(error_message.is_displayed())
            
        finally:
            # Re-enable network
            self.driver.execute_cdp_cmd('Network.setBlockedURLs', {'urls': []})
            self.driver.execute_cdp_cmd('Network.disable', {})


if __name__ == '__main__':
    # Configure test execution
    unittest.TestLoader.testMethodPrefix = 'test_'
    
    # Run tests with detailed output
    unittest.main(verbosity=2, buffer=True)