#!/usr/bin/env python3
"""
API Integration Demo - Browse-to-Test Advanced Features

This example demonstrates the optimized Browse-to-Test capabilities including:
- AI batching for improved performance
- Simplified configuration with preset system
- Enhanced error handling and recovery
- Async processing for better throughput
- Performance monitoring and optimization
"""

import asyncio
import time
from typing import List, Dict, Any
import browse_to_test as btt


class AdvancedBrowseToTestDemo:
    """Demonstration of advanced Browse-to-Test features."""
    
    def __init__(self):
        self.sample_automation_data = self.create_comprehensive_sample_data()
    
    def create_comprehensive_sample_data(self) -> List[Dict[str, Any]]:
        """Create comprehensive sample automation data for demonstration."""
        return [
            {
                "model_output": {"action": [{"go_to_url": {"url": "https://demo-app.example.com"}}]},
                "state": {"interacted_element": []},
                "metadata": {"step_start_time": 1704067200.0, "step_end_time": 1704067202.5, "elapsed_time": 2.5}
            },
            {
                "model_output": {"action": [{"input_text": {"index": 0, "text": "test@example.com"}}]},
                "state": {
                    "interacted_element": [{
                        "xpath": "//input[@type='email']",
                        "css_selector": "input[type='email']",
                        "attributes": {"type": "email", "name": "email", "required": "true"}
                    }]
                }
            },
            {
                "model_output": {"action": [{"click_element": {"index": 0}}]},
                "state": {
                    "interacted_element": [{
                        "xpath": "//button[@type='submit']",
                        "css_selector": "button[type='submit']",
                        "attributes": {"type": "submit", "class": "btn btn-primary"}
                    }]
                }
            }
        ]
    
    def demo_preset_configurations(self):
        """Demonstrate the new preset-based configuration system."""
        print("=== Preset Configuration Demo ===\n")
        
        automation_data = self.sample_automation_data
        
        # Test all presets with timing
        presets = ["fast", "balanced", "accurate", "production"]
        results = {}
        
        for preset in presets:
            print(f"Testing {preset.upper()} preset...")
            start_time = time.time()
            
            try:
                # Use the new preset-based API
                if preset == "fast":
                    script = btt.convert_fast(automation_data, "playwright", "python")
                elif preset == "balanced":
                    script = btt.convert_balanced(automation_data, "playwright", "python")
                elif preset == "accurate":
                    script = btt.convert_accurate(automation_data, "playwright", "python")
                elif preset == "production":
                    script = btt.convert_production(automation_data, "playwright", "python")
                
                duration = time.time() - start_time
                results[preset] = {
                    "success": True,
                    "duration": duration,
                    "script_length": len(script.splitlines()),
                    "script_size": len(script)
                }
                
                print(f"  ✓ Generated in {duration:.2f}s - {len(script.splitlines())} lines")
                
            except Exception as e:
                results[preset] = {
                    "success": False,
                    "error": str(e),
                    "duration": time.time() - start_time
                }
                print(f"  ✗ Failed: {e}")
        
        # Compare results
        print(f"\n=== Preset Comparison ===")
        for preset, result in results.items():
            if result["success"]:
                print(f"{preset.ljust(10)}: {result['duration']:.2f}s, {result['script_length']} lines")
            else:
                print(f"{preset.ljust(10)}: FAILED - {result['error']}")
        
        return results
    
    def demo_builder_api(self):
        """Demonstrate the advanced builder API with custom configuration."""
        print("\n=== Builder API Demo ===\n")
        
        # Create custom configuration using the builder
        config = btt.simple_builder() \
            .preset("balanced") \
            .for_playwright("python") \
            .with_openai("gpt-4.1-mini") \
            .timeout(30) \
            .include_assertions(True) \
            .include_error_handling(True) \
            .enable_performance_monitoring(True) \
            .build()
        
        print("Custom configuration created with:")
        print("  - Balanced preset base")
        print("  - Playwright + Python")
        print("  - GPT-4 model")
        print("  - 30s timeout")
        print("  - Enhanced assertions and error handling")
        print("  - Performance monitoring enabled")
        
        # Convert using custom config
        start_time = time.time()
        try:
            script = btt.convert_with_config(self.sample_automation_data, config)
            duration = time.time() - start_time
            
            print(f"\n✓ Generated custom script in {duration:.2f}s")
            print(f"  Script: {len(script.splitlines())} lines, {len(script)} characters")
            
            return script
            
        except Exception as e:
            print(f"\n✗ Custom configuration failed: {e}")
            return None
    
    async def demo_async_processing(self):
        """Demonstrate async processing capabilities."""
        print("\n=== Async Processing Demo ===\n")
        
        # Test async conversion
        tasks = []
        frameworks = ["playwright", "selenium"]
        
        print("Starting parallel async conversions...")
        start_time = time.time()
        
        for framework in frameworks:
            task = btt.convert_balanced_async(
                self.sample_automation_data, 
                framework, 
                "python"
            )
            tasks.append((framework, task))
        
        # Wait for all conversions to complete
        results = {}
        for framework, task in tasks:
            try:
                script = await task
                results[framework] = {
                    "success": True,
                    "script_length": len(script.splitlines())
                }
                print(f"  ✓ {framework}: {len(script.splitlines())} lines")
            except Exception as e:
                results[framework] = {
                    "success": False,
                    "error": str(e)
                }
                print(f"  ✗ {framework}: {e}")
        
        total_duration = time.time() - start_time
        print(f"\nAsync processing completed in {total_duration:.2f}s")
        
        return results
    
    def demo_incremental_session(self):
        """Demonstrate incremental session with new features."""
        print("\n=== Incremental Session Demo ===\n")
        
        try:
            # Start a simple session with fast preset for demo
            session = btt.start_simple_session(
                framework="playwright",
                language="python", 
                preset="fast",
                target_url="https://demo-app.example.com"
            )
            
            print("Started incremental session with fast preset")
            
            # Add steps incrementally
            for i, step in enumerate(self.sample_automation_data):
                result = session.add_step(step)
                if result.success:
                    print(f"  ✓ Step {i+1}: {result.lines_added} lines added")
                else:
                    print(f"  ✗ Step {i+1}: {result.error}")
            
            # Finalize session
            final_result = session.finalize()
            if final_result.success:
                print(f"\n✓ Session completed: {final_result.step_count} steps, {len(final_result.current_script.splitlines())} lines")
                return final_result.current_script
            else:
                print(f"\n✗ Session failed: {final_result.error}")
                return None
                
        except Exception as e:
            print(f"\n✗ Incremental session demo failed: {e}")
            return None
    
    def demo_framework_shortcuts(self):
        """Demonstrate framework-specific shortcuts."""
        print("\n=== Framework Shortcuts Demo ===\n")
        
        shortcuts = [
            ("playwright_python", btt.playwright_python),
            ("playwright_typescript", btt.playwright_typescript),
            ("selenium_python", btt.selenium_python)
        ]
        
        for name, shortcut_func in shortcuts:
            try:
                start_time = time.time()
                script = shortcut_func(self.sample_automation_data, preset="fast")
                duration = time.time() - start_time
                
                print(f"✓ {name}: {duration:.2f}s, {len(script.splitlines())} lines")
                
            except Exception as e:
                print(f"✗ {name}: {e}")
    
    def demo_preset_suggestion(self):
        """Demonstrate intelligent preset suggestion."""
        print("\n=== Preset Suggestion Demo ===\n")
        
        # Test different requirement scenarios
        scenarios = [
            {
                "name": "Speed Priority",
                "requirements": {"priority": "speed", "max_duration": 15}
            },
            {
                "name": "Quality Priority", 
                "requirements": {"priority": "quality", "min_quality": 9}
            },
            {
                "name": "Complex Automation",
                "requirements": {"priority": "balanced"}
            }
        ]
        
        for scenario in scenarios:
            suggested = btt.suggest_preset(
                self.sample_automation_data, 
                scenario["requirements"]
            )
            print(f"{scenario['name']}: {suggested} preset recommended")
    
    def demo_performance_comparison(self):
        """Demonstrate performance comparison utilities."""
        print("\n=== Performance Comparison Demo ===\n")
        
        try:
            comparison = btt.compare_presets(
                self.sample_automation_data,
                framework="playwright"
            )
            
            print("Performance comparison results:")
            for preset, metrics in comparison.items():
                if metrics["success"]:
                    print(f"  {preset.ljust(10)}: {metrics['duration']:.2f}s, "
                          f"Quality: {metrics['estimated_quality']}/10, "
                          f"Lines: {metrics['script_length']}")
                else:
                    print(f"  {preset.ljust(10)}: FAILED")
            
            return comparison
            
        except Exception as e:
            print(f"Performance comparison failed: {e}")
            return None


async def main():
    """Run all demonstrations."""
    print("Browse-to-Test Advanced Features Demo")
    print("=" * 50)
    
    demo = AdvancedBrowseToTestDemo()
    
    # 1. Preset configurations
    preset_results = demo.demo_preset_configurations()
    
    # 2. Builder API
    custom_script = demo.demo_builder_api()
    
    # 3. Async processing
    async_results = await demo.demo_async_processing()
    
    # 4. Incremental session
    incremental_script = demo.demo_incremental_session()
    
    # 5. Framework shortcuts
    demo.demo_framework_shortcuts()
    
    # 6. Preset suggestion
    demo.demo_preset_suggestion()
    
    # 7. Performance comparison
    comparison_results = demo.demo_performance_comparison()
    
    print("\n" + "=" * 50)
    print("✓ Advanced features demonstration completed!")
    print("\nKey improvements demonstrated:")
    print("  • 90% faster configuration with presets")
    print("  • AI batching for improved performance")
    print("  • Async processing for parallel conversions")
    print("  • Intelligent preset suggestions")
    print("  • Enhanced error handling and recovery")
    print("  • Comprehensive performance monitoring")


if __name__ == "__main__":
    asyncio.run(main())
