"""
Input parser for browser automation data.

Handles parsing and normalizing various formats of browser automation data
into a standardized internal format for processing.
"""

import json
import logging
from typing import Any, Dict, List, Optional, Union
from dataclasses import dataclass
from pathlib import Path

from ..config import Config


logger = logging.getLogger(__name__)


@dataclass
class ParsedAction:
    """Represents a single parsed action from browser automation data."""
    
    action_type: str  # go_to_url, click_element, input_text, etc.
    parameters: Dict[str, Any]  # Action-specific parameters
    step_index: int  # Which step this action belongs to
    action_index: int  # Index within the step
    selector_info: Optional[Dict[str, Any]] = None  # Element selector information
    metadata: Optional[Dict[str, Any]] = None  # Additional metadata
    
    def __post_init__(self):
        """Ensure parameters is not None."""
        if self.parameters is None:
            self.parameters = {}
        # Don't automatically set metadata to {} - let it stay None if not provided
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'ParsedAction':
        """Create ParsedAction from dictionary."""
        return cls(
            action_type=data.get('action_type', ''),
            parameters=data.get('parameters', {}),
            step_index=data.get('step_index', 0),
            action_index=data.get('action_index', 0),
            selector_info=data.get('selector_info'),
            metadata=data.get('metadata')
        )
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert ParsedAction to dictionary."""
        return {
            'action_type': self.action_type,
            'parameters': self.parameters,
            'step_index': self.step_index,
            'action_index': self.action_index,
            'selector_info': self.selector_info,
            'metadata': self.metadata
        }


@dataclass
class ParsedStep:
    """Represents a single parsed step containing one or more actions."""
    
    step_index: int
    actions: List[ParsedAction]
    metadata: Optional[Dict[str, Any]] = None
    timing_info: Optional[Dict[str, Any]] = None
    
    def __post_init__(self):
        """Post-initialization processing for ParsedStep."""
        # Don't automatically set timing_info or metadata to {} - let them stay None if not provided
        pass
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'ParsedStep':
        """Create ParsedStep from dictionary."""
        actions_data = data.get('actions', [])
        actions = []
        for action_data in actions_data:
            if isinstance(action_data, dict):
                actions.append(ParsedAction.from_dict(action_data))
            else:
                actions.append(action_data)  # Already a ParsedAction
        
        return cls(
            step_index=data.get('step_index', 0),
            actions=actions,
            metadata=data.get('metadata'),
            timing_info=data.get('timing_info')
        )
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert ParsedStep to dictionary."""
        return {
            'step_index': self.step_index,
            'actions': [action.to_dict() if hasattr(action, 'to_dict') else action.__dict__ for action in self.actions],
            'metadata': self.metadata,
            'timing_info': self.timing_info
        }


@dataclass
class ParsedAutomationData:
    """Container for all parsed automation data."""
    
    steps: List[ParsedStep]
    total_actions: int = 0  # Will be calculated in __post_init__
    sensitive_data_keys: Optional[List[str]] = None
    metadata: Optional[Dict[str, Any]] = None
    
    def __post_init__(self):
        """Calculate total actions from steps if not provided."""
        if self.sensitive_data_keys is None:
            self.sensitive_data_keys = []
        
        # Recalculate total actions from steps
        self.total_actions = sum(len(step.actions) for step in self.steps)
    
    def to_dict(self) -> Dict[str, Any]:
        """Convert the parsed data to a dictionary format."""
        return {
            "steps": [step.to_dict() if hasattr(step, 'to_dict') else step.__dict__ for step in self.steps],
            "total_actions": self.total_actions,
            "sensitive_data_keys": self.sensitive_data_keys or [],
            "metadata": self.metadata or {}
        }


class InputParser:
    """Parser for browser automation data from various sources."""
    
    def __init__(self, config):
        """
        Initialize the input parser.
        
        Args:
            config: Configuration object containing parsing settings.
        """
        self.config = config
        # Check for strict_mode in different possible locations
        if hasattr(config, 'processing') and hasattr(config.processing, 'strict_mode'):
            self.strict_mode = config.processing.strict_mode
        else:
            self.strict_mode = getattr(config, 'strict_mode', False)
        self.logger = logging.getLogger(__name__)
    
    def parse(self, automation_data: Union[List[Dict], str, Path]) -> ParsedAutomationData:
        """
        Parse browser automation data from various input formats.
        
        Args:
            automation_data: Can be:
                - List of step dictionaries (direct format)
                - JSON string containing the data
                - Path to JSON file containing the data
                
        Returns:
            ParsedAutomationData containing normalized data
            
        Raises:
            ValueError: If data format is invalid and strict_mode is True
        """
        # Handle None input
        if automation_data is None:
            raise ValueError("Automation data cannot be None")
        
        # Handle different input types
        if isinstance(automation_data, (str, Path)):
            automation_data = self._load_from_file_or_string(automation_data)
        
        if not isinstance(automation_data, list):
            error_msg = f"Expected list of steps, got {type(automation_data)}"
            if self.strict_mode:
                raise ValueError(error_msg)
            self.logger.warning(error_msg)
            return ParsedAutomationData(steps=[], total_actions=0)
        
        # Parse each step
        parsed_steps = []
        for step_index, step_data in enumerate(automation_data):
            try:
                parsed_step = self._parse_step(step_data, step_index)
                parsed_steps.append(parsed_step)  # Include all steps, even empty ones
            except Exception as e:
                error_msg = f"Error parsing step {step_index}: {e}"
                if self.strict_mode:
                    raise ValueError(error_msg) from e
                self.logger.warning(error_msg)
                continue
        
        total_actions = sum(len(step.actions) for step in parsed_steps)
        
        self.logger.info(
            f"Parsed {len(parsed_steps)} steps with {total_actions} total actions"
        )
        
        return ParsedAutomationData(
            steps=parsed_steps,
            total_actions=total_actions,
            metadata={
                "original_step_count": len(automation_data),
                "parsed_step_count": len(parsed_steps),
                "parser_version": "1.0.0"
            }
        )
    
    def _load_from_file_or_string(self, data: Union[str, Path]) -> List[Dict]:
        """Load automation data from file path or JSON string."""
        # Handle Path objects
        if isinstance(data, Path):
            if not data.exists():
                raise FileNotFoundError(f"Automation data file not found: {data}")
            with open(data, 'r', encoding='utf-8') as f:
                return json.load(f)
        
        # Handle string inputs
        if isinstance(data, str):
            # Check if it looks like a file path - be more restrictive
            # Must not start with [ or { (JSON indicators) and should be a reasonable file path
            is_potential_file = (
                not data.strip().startswith(('[', '{'))
                and len(data) < 500  # File paths are usually not very long
                and (data.endswith('.json') or ('/' in data and 'http' not in data))
            )
            
            if is_potential_file:
                file_path = Path(data)
                if file_path.exists():
                    with open(file_path, 'r', encoding='utf-8') as f:
                        return json.load(f)
                else:
                    raise FileNotFoundError(f"Automation data file not found: {file_path}")
            
            # Try to parse as JSON string
            try:
                return json.loads(data)
            except json.JSONDecodeError:
                # Re-raise the original JSONDecodeError for tests that expect it
                raise
        
        raise ValueError(f"Unsupported data type: {type(data)}")
    
    def _parse_step(self, step_data: Dict[str, Any], step_index: int) -> ParsedStep:
        """Parse a single step from the automation data."""
        if not isinstance(step_data, dict):
            raise ValueError(f"Step {step_index} is not a dictionary")
        
        # Handle different data formats - be flexible with legacy formats
        # Check if this is a legacy format with direct action data
        if 'model_output' not in step_data:
            # Check if it has direct action data (legacy format)
            if 'action' in step_data:
                # Convert legacy format to model_output format
                step_data = {
                    'model_output': {'action': step_data['action']},
                    'state': step_data.get('state', {}),
                    'metadata': step_data.get('metadata', {})
                }
            elif any(key in step_data for key in ['go_to_url', 'click_element', 'input_text', 'scroll_down', 'scroll_up', 'wait', 'done']):
                # Single action format - wrap in action array
                action_data = {k: v for k, v in step_data.items() if k not in ['state', 'metadata']}
                step_data = {
                    'model_output': {'action': [action_data]},
                    'state': step_data.get('state', {}),
                    'metadata': step_data.get('metadata', {})
                }
            elif self.strict_mode:
                # Only error in strict mode if we can't convert the format
                raise ValueError(f"Step {step_index} is missing required 'model_output' field")
            else:
                # In non-strict mode, create empty step but log warning
                self.logger.warning(f"Step {step_index} is missing required 'model_output' field")
                return ParsedStep(step_index=step_index, actions=[], metadata={}, timing_info={})
        
        # Extract step metadata - use the metadata field directly if available
        step_metadata = None
        if 'metadata' in step_data and isinstance(step_data['metadata'], dict):
            step_metadata = step_data['metadata'].copy()
        
        # Also include any other fields that aren't model_output, state, or metadata
        other_fields = {key: value for key, value in step_data.items()
                        if key not in ['model_output', 'state', 'metadata']}
        
        if other_fields:
            if step_metadata is None:
                step_metadata = other_fields
            else:
                step_metadata.update(other_fields)
        
        # Extract timing information if available
        timing_info = {}
        if 'metadata' in step_data and isinstance(step_data['metadata'], dict):
            metadata = step_data['metadata']
            timing_fields = ['step_start_time', 'step_end_time', 'elapsed_time']
            timing_info = {
                field: metadata.get(field) 
                for field in timing_fields 
                if field in metadata
            }
        
        # Get model output containing actions
        model_output = step_data.get('model_output', {})
        if not isinstance(model_output, dict):
            if self.strict_mode:
                raise ValueError(f"Step {step_index} has invalid model_output type: {type(model_output)}")
            self.logger.warning(f"Step {step_index} has invalid model_output")
            return ParsedStep(step_index=step_index, actions=[], metadata=step_metadata, timing_info=timing_info)
        
        # Extract actions
        actions_data = model_output.get('action', [])
        if not isinstance(actions_data, list):
            self.logger.warning(f"Step {step_index} has invalid actions format")
            return ParsedStep(step_index=step_index, actions=[], metadata=step_metadata, timing_info=timing_info)
        
        # Parse each action
        parsed_actions = []
        state_data = step_data.get('state', {})
        interacted_elements = state_data.get('interacted_element', [])
        
        for action_index, action_data in enumerate(actions_data):
            try:
                parsed_action = self._parse_action(
                    action_data, 
                    step_index, 
                    action_index,
                    interacted_elements
                )
                if parsed_action:
                    parsed_actions.append(parsed_action)
            except Exception as e:
                error_msg = f"Error parsing action {action_index} in step {step_index}: {e}"
                if self.strict_mode:
                    raise ValueError(error_msg) from e
                self.logger.warning(error_msg)
                continue
        
        # If no valid actions were found, log this appropriately
        if not parsed_actions and actions_data:
            self.logger.info(f"Step {step_index} has {len(actions_data)} action(s) but no valid actions after parsing - likely contains empty action objects")
        
        return ParsedStep(
            step_index=step_index,
            actions=parsed_actions,
            metadata=step_metadata,
            timing_info=timing_info
        )
    
    def _parse_action(
        self, 
        action_data: Dict[str, Any], 
        step_index: int, 
        action_index: int,
        interacted_elements: List[Dict[str, Any]]
    ) -> Optional[ParsedAction]:
        """Parse a single action from the action data."""
        if not isinstance(action_data, dict):
            self.logger.warning(f"Invalid action type at step {step_index}, action {action_index}: expected dict, got {type(action_data)}")
            return None
        
        if not action_data:
            # Empty dict - this is common in real automation data and should be handled gracefully
            self.logger.debug(f"Empty action at step {step_index}, action {action_index} - skipping")
            return None
        
        # Extract action type (first key in the dictionary)
        action_type = next(iter(action_data.keys()))
        parameters = action_data[action_type]
        
        if not isinstance(parameters, dict):
            if parameters is None:
                parameters = {}
            else:
                self.logger.warning(
                    f"Action {action_type} has invalid parameters at step {step_index}, action {action_index}"
                )
                return None
        
        # Extract selector information if available
        selector_info = None
        if action_index < len(interacted_elements) and interacted_elements[action_index]:
            selector_info = self._extract_selector_info(interacted_elements[action_index])
        elif 'selector' in parameters and parameters['selector']:
            # Fallback: extract selector directly from action parameters
            selector_info = {'css_selector': parameters['selector']}
        
        # Create action metadata
        action_metadata = {
            'original_format': 'browser_automation',
            'requires_element': self._requires_element_interaction(action_type),
            'has_selector': selector_info is not None
        }
        
        return ParsedAction(
            action_type=action_type,
            parameters=parameters,
            step_index=step_index,
            action_index=action_index,
            selector_info=selector_info,
            metadata=action_metadata
        )
    
    def _extract_selector_info(self, element_data: Dict[str, Any]) -> Dict[str, Any]:
        """Extract and normalize selector information from element data."""
        selector_info = {}
        
        # Extract XPath
        if 'xpath' in element_data and element_data['xpath']:
            xpath = element_data['xpath']
            # Store the xpath as-is without normalization
            selector_info['xpath'] = xpath
        
        # Extract CSS selector
        if 'css_selector' in element_data and element_data['css_selector']:
            selector_info['css_selector'] = element_data['css_selector']
        
        # Extract other useful information
        if 'highlight_index' in element_data:
            selector_info['highlight_index'] = element_data['highlight_index']
        
        # Extract element attributes if available
        if 'attributes' in element_data:
            selector_info['attributes'] = element_data['attributes']
        
        # Extract text content if available
        if 'text_content' in element_data:
            selector_info['text_content'] = element_data['text_content']
        
        return selector_info
    
    def _requires_element_interaction(self, action_type: str) -> bool:
        """Determine if an action type requires element interaction."""
        element_actions = {
            'click_element', 'click_element_by_index', 'input_text', 
            'click_download_button', 'drag_drop', 'hover_element',
            # Also support simplified action names used in tests
            'click', 'fill', 'hover'
        }
        return action_type in element_actions
    
    def validate_parsed_data(self, parsed_data: ParsedAutomationData) -> List[str]:
        """
        Validate parsed automation data and return list of validation issues.
        
        Args:
            parsed_data: The parsed automation data to validate
            
        Returns:
            List of validation warning/error messages
        """
        issues = []
        
        if not parsed_data.steps:
            issues.append("No valid steps found in automation data")
            return issues
        
        if parsed_data.total_actions == 0:
            issues.append("No valid actions found in automation data")
        
        # Check each step
        for step in parsed_data.steps:
            step_prefix = f"Step {step.step_index}"
            
            if not step.actions:
                issues.append(f"{step_prefix}: No valid actions found")
                continue
            
            # Check for actions that require selectors but don't have them
            for action in step.actions:
                action_prefix = f"{step_prefix}, Action {action.action_index}"
                
                if action.metadata.get('requires_element') and not action.metadata.get('has_selector'):
                    issues.append(
                        f"{action_prefix}: Action '{action.action_type}' requires element "
                        "selector but none was found"
                    )
                
                # Check for empty parameters on actions that typically need them
                if action.action_type in ['go_to_url', 'input_text', 'search_google'] and not action.parameters:
                    issues.append(
                        f"{action_prefix}: Action '{action.action_type}' has no parameters"
                    )
        
        return issues
    
    def extract_sensitive_data_keys(self, parsed_data: ParsedAutomationData) -> List[str]:
        """
        Extract potential sensitive data keys from the parsed data.
        
        Looks for patterns like <secret>key</secret> in action parameters.
        
        Args:
            parsed_data: The parsed automation data
            
        Returns:
            List of unique sensitive data keys found
        """
        import re
        
        sensitive_keys = set()
        secret_pattern = re.compile(r'<secret>([^<]+)</secret>')
        
        for step in parsed_data.steps:
            for action in step.actions:
                # Check all string values in parameters
                for _param_name, param_value in action.parameters.items():
                    if isinstance(param_value, str):
                        matches = secret_pattern.findall(param_value)
                        sensitive_keys.update(matches)
        
        return sorted(sensitive_keys)
    
    def parse_single_step(self, step_data):
        """Parse a single step for incremental sessions."""
        # If it's already a ParsedStep, return it
        if hasattr(step_data, 'step_index') and hasattr(step_data, 'actions'):
            return step_data
        
        # If it's a dictionary, parse it
        if isinstance(step_data, dict):
            # Parse as a single step with index 0
            return self._parse_step(step_data, 0)
        
        # Check if it's a dictionary-like object with standard automation data keys
        try:
            # Check for required keys to identify the data structure
            if all(key in step_data for key in ['model_output', 'state', 'metadata']):
                return self._parse_step(dict(step_data), 0)
        except Exception:
            pass
        
        # If none of the above, raise a helpful error
        raise ValueError(f"Unable to parse step data of type {type(step_data)}. Expected a dictionary or a pre-parsed step.")

    def validate(self, parsed_data: ParsedAutomationData) -> List[str]:
        """
        Validate parsed automation data and return list of validation errors.
        
        Args:
            parsed_data: ParsedAutomationData to validate
            
        Returns:
            List of validation error messages (empty if no errors)
        """
        errors = []
        
        # Basic structure validation
        if not isinstance(parsed_data, ParsedAutomationData):
            errors.append(f"Expected ParsedAutomationData, got {type(parsed_data)}")
            return errors
        
        # Check if we have any steps
        if not parsed_data.steps:
            if self.strict_mode:
                errors.append("No steps found in automation data")
            else:
                # In non-strict mode, just log warning
                self.logger.warning("No steps found in automation data")
        
        # Validate each step
        for step_index, step in enumerate(parsed_data.steps):
            if not isinstance(step, ParsedStep):
                errors.append(f"Step {step_index} is not a ParsedStep instance")
                continue
            
            # Check if step has actions
            if not step.actions:
                if self.strict_mode:
                    errors.append(f"Step {step_index} has no actions")
                else:
                    self.logger.warning(f"Step {step_index} has no actions")
            
            # Validate each action
            for action_index, action in enumerate(step.actions):
                if not isinstance(action, ParsedAction):
                    errors.append(f"Step {step_index}, action {action_index} is not a ParsedAction instance")
                    continue
                
                # Check action type
                if not action.action_type:
                    errors.append(f"Step {step_index}, action {action_index} has empty action_type")
                
                # Check parameters
                if action.parameters is None:
                    if self.strict_mode:
                        errors.append(f"Step {step_index}, action {action_index} has None parameters")
                
                # Validate specific action types
                self._validate_action_type(action, step_index, action_index, errors)
        
        # Check total actions consistency
        expected_total = sum(len(step.actions) for step in parsed_data.steps)
        if parsed_data.total_actions != expected_total:
            errors.append(f"Total actions mismatch: reported {parsed_data.total_actions}, calculated {expected_total}")
        
        return errors
    
    def _validate_action_type(self, action: ParsedAction, step_index: int, action_index: int, errors: List[str]):
        """Validate specific action types and their required parameters."""
        action_type = action.action_type
        params = action.parameters or {}
        
        # Define required parameters for different action types
        required_params = {
            'go_to_url': ['url'],
            'click_element': [],  # selector comes from selector_info
            'input_text': ['text'],  # selector comes from selector_info
            'scroll_down': [],
            'scroll_up': [],
            'wait': [],  # time is optional
            'done': [],
        }
        
        if action_type in required_params:
            for required_param in required_params[action_type]:
                if required_param not in params or params[required_param] is None:
                    if self.strict_mode:
                        errors.append(f"Step {step_index}, action {action_index} ({action_type}) missing required parameter: {required_param}")
        
        # Validate selector information for actions that need it
        if action_type in ['click_element', 'input_text']:
            if not action.selector_info:
                if self.strict_mode:
                    errors.append(f"Step {step_index}, action {action_index} ({action_type}) missing selector information")