"""Utilities for compact representation of equal-cost path sets.

This module defines ``PathBundle``, a structure that represents one or more
equal-cost paths between two nodes using a predecessor map. It supports
concatenation, containment checks, sub-bundle extraction with cost
recalculation, and enumeration into concrete ``Path`` instances.
"""

from __future__ import annotations

from collections import deque
from heapq import heappop, heappush
from typing import Dict, Iterator, List, Optional, Set, Tuple

from ngraph.algorithms.base import Cost, EdgeSelect
from ngraph.algorithms.edge_select import edge_select_fabric
from ngraph.algorithms.paths import resolve_to_paths
from ngraph.graph.strict_multidigraph import (
    AttrDict,
    EdgeID,
    NodeID,
    StrictMultiDiGraph,
)
from ngraph.paths.path import Path


class PathBundle:
    """A collection of equal-cost paths between two nodes.

    This class encapsulates one or more parallel paths (all of the same cost)
    between `src_node` and `dst_node`. The predecessor map `pred` associates
    each node with the node(s) from which it can be reached, along with a list
    of edge IDs used in that step. The constructor performs a reverse traversal
    from `dst_node` to `src_node` to collect all edges, nodes, and store them
    in this bundle.

    The constructor assumes the predecessor relation forms a DAG between
    ``src_node`` and ``dst_node``. No cycle detection is performed. If cycles
    are present, traversal may not terminate.
    """

    def __init__(
        self,
        src_node: NodeID,
        dst_node: NodeID,
        pred: Dict[NodeID, Dict[NodeID, List[EdgeID]]],
        cost: Cost,
    ) -> None:
        """Initialize the PathBundle.

        Args:
            src_node: The source node for all paths in this bundle.
            dst_node: The destination node for all paths in this bundle.
            pred: A predecessor map of the form:
                  {
                      current_node: {
                          prev_node: [edge_id_1, edge_id_2, ...],
                          ...
                      },
                      ...
                  }
                  Typically generated by a shortest-path or multi-path algorithm.
            cost: The total path cost (e.g. distance, cost) of all paths in the bundle.
        """
        self.src_node: NodeID = src_node
        self.dst_node: NodeID = dst_node
        self.cost: Cost = cost
        # We'll rebuild `pred` to store only the relevant portion from dst_node to src_node.
        self.pred: Dict[NodeID, Dict[NodeID, List[EdgeID]]] = {src_node: {}}
        self.edges: Set[EdgeID] = set()
        self.edge_tuples: Set[Tuple[EdgeID, ...]] = set()
        self.nodes: Set[NodeID] = {src_node}

        visited: Set[NodeID] = set()
        queue: deque[NodeID] = deque([dst_node])
        visited.add(dst_node)

        while queue:
            node = queue.popleft()
            self.nodes.add(node)
            # Ensure key exists even if `node` has no predecessors.
            self.pred.setdefault(node, {})

            # Traverse all predecessors of `node`
            for prev_node, edges_list in pred[node].items():
                # Record these edges in our local `pred` structure
                self.pred[node][prev_node] = edges_list
                # Update the set of all edges seen in this bundle
                self.edges.update(edges_list)
                # Store the tuple form for quick equality checks on parallel edges
                self.edge_tuples.add(tuple(edges_list))

                # Enqueue the predecessor unless it's the original source.
                # No cycle check is performed, since we trust `pred` is a DAG.
                if prev_node != src_node and prev_node not in visited:
                    visited.add(prev_node)
                    queue.append(prev_node)

    def __lt__(self, other: PathBundle) -> bool:
        """Compare two PathBundles by cost (for sorting)."""
        return self.cost < other.cost

    def __eq__(self, other: object) -> bool:
        """Check equality of two PathBundles by (src, dst, cost, edges)."""
        if not isinstance(other, PathBundle):
            return False
        return (
            self.src_node == other.src_node
            and self.dst_node == other.dst_node
            and self.cost == other.cost
            and self.edges == other.edges
        )

    def __hash__(self) -> int:
        """Create a unique hash based on (src, dst, cost, frozenset of edges)."""
        return hash((self.src_node, self.dst_node, self.cost, frozenset(self.edges)))

    def __repr__(self) -> str:
        """String representation of this PathBundle."""
        return f"PathBundle({self.src_node}, {self.dst_node}, {self.pred}, {self.cost})"

    def add(self, other: PathBundle) -> PathBundle:
        """Concatenate this bundle with another bundle (end-to-start).

        This effectively merges the predecessor maps and combines costs.

        Args:
            other: Another PathBundle whose `src_node` must match this bundle's `dst_node`.

        Returns:
            A new PathBundle from `self.src_node` to `other.dst_node`.

        Raises:
            ValueError: If this bundle's `dst_node` does not match the other's `src_node`.
        """
        if self.dst_node != other.src_node:
            raise ValueError("PathBundle dst_node != other.src_node")

        # Make a combined predecessor map
        new_pred: Dict[NodeID, Dict[NodeID, List[EdgeID]]] = {}
        # Copy from self
        for dnode in self.pred:
            new_pred.setdefault(dnode, {})
            for snode, edges_list in self.pred[dnode].items():
                new_pred[dnode][snode] = list(edges_list)
        # Copy from other
        for dnode in other.pred:
            new_pred.setdefault(dnode, {})
            for snode, edges_list in other.pred[dnode].items():
                new_pred[dnode][snode] = list(edges_list)

        return PathBundle(
            self.src_node, other.dst_node, new_pred, self.cost + other.cost
        )

    @classmethod
    def from_path(
        cls,
        path: Path,
        resolve_edges: bool = False,
        graph: Optional[StrictMultiDiGraph] = None,
        edge_select: Optional[EdgeSelect] = None,
        cost_attr: str = "cost",
        capacity_attr: str = "capacity",
    ) -> PathBundle:
        """Construct a PathBundle from a single `Path` object.

        Args:
            path: A `Path` object which contains node-edge tuples, plus a `cost`.
            resolve_edges: If True, dynamically choose the minimal-cost edges
                between each node pair via the provided `edge_select`.
            graph: The graph used for edge resolution (required if `resolve_edges=True`).
            edge_select: The selection criterion for picking edges if `resolve_edges=True`.
            cost_attr: The attribute name on edges representing cost (e.g., 'cost').
            capacity_attr: The attribute name on edges representing capacity.

        Returns:
            A new PathBundle corresponding to the single path. If `resolve_edges`
            is True, the cost is recalculated; otherwise the original `path.cost` is used.

        Raises:
            ValueError: If `resolve_edges` is True but no `graph` is provided.
        """
        if resolve_edges:
            if not graph:
                raise ValueError(
                    "A StrictMultiDiGraph `graph` is required when resolve_edges=True."
                )
            if edge_select is None:
                raise ValueError(
                    "edge_select must be provided when resolve_edges=True."
                )
            edge_selector = edge_select_fabric(
                edge_select,
                cost_attr=cost_attr,
                capacity_attr=capacity_attr,
            )
        else:
            edge_selector = None

        src_node = path.path_tuple[0][0]
        dst_node = path.path_tuple[-1][0]
        pred_map: Dict[NodeID, Dict[NodeID, List[EdgeID]]] = {src_node: {}}
        total_cost: Cost = 0

        # Build the predecessor map from each hop
        for (a_node, a_edges), (z_node, _) in zip(
            path.path_tuple[:-1], path.path_tuple[1:], strict=True
        ):
            pred_map.setdefault(z_node, {})
            # If we're not resolving edges, just copy whatever the path has
            if not resolve_edges:
                pred_map[z_node][a_node] = list(a_edges)
            else:
                # Re-select edges from a_node to z_node
                if edge_selector is not None and graph is not None:
                    # Convert edges_dict to the expected Dict[EdgeID, AttrDict] format
                    # Since EdgeID is just Hashable, we can cast the keys directly
                    typed_edges_dict: Dict[EdgeID, AttrDict] = {
                        k: v for k, v in graph[a_node][z_node].items()
                    }
                    min_cost, edge_list = edge_selector(
                        graph, a_node, z_node, typed_edges_dict, None, None
                    )
                    pred_map[z_node][a_node] = edge_list
                    total_cost += min_cost

        if resolve_edges:
            return cls(src_node, dst_node, pred_map, total_cost)
        return cls(src_node, dst_node, pred_map, path.cost)

    def resolve_to_paths(self, split_parallel_edges: bool = False) -> Iterator[Path]:
        """Generate all concrete `Path` objects contained in this PathBundle.

        Args:
            split_parallel_edges: If False, any parallel edges are grouped together
                into a single path segment. If True, produce all permutations
                of parallel edges as distinct paths.

        Yields:
            A `Path` object for each distinct route from `src_node` to `dst_node`.
        """
        for path_tuple in resolve_to_paths(
            self.src_node,
            self.dst_node,
            self.pred,
            split_parallel_edges,
        ):
            yield Path(path_tuple, self.cost)

    def contains(self, other: PathBundle) -> bool:
        """Check if this bundle's edge set contains all edges of `other`.

        Args:
            other: Another PathBundle.

        Returns:
            True if `other`'s edges are a subset of this bundle's edges.
        """
        return self.edges.issuperset(other.edges)

    def is_subset_of(self, other: PathBundle) -> bool:
        """Check if this bundle's edge set is contained in `other`'s edge set.

        Args:
            other: Another PathBundle.

        Returns:
            True if all edges in this bundle are in `other`.
        """
        return self.edges.issubset(other.edges)

    def is_disjoint_from(self, other: PathBundle) -> bool:
        """Check if this bundle shares no edges with `other`.

        Args:
            other: Another PathBundle.

        Returns:
            True if there are no common edges between the two bundles.
        """
        return self.edges.isdisjoint(other.edges)

    def get_sub_path_bundle(
        self,
        new_dst_node: NodeID,
        graph: StrictMultiDiGraph,
        cost_attr: str = "cost",
    ) -> PathBundle:
        """Create a sub-bundle ending at `new_dst_node` with correct minimal cost.

        The returned bundle contains the predecessor subgraph that reaches from
        `self.src_node` to `new_dst_node` using only relations present in this
        bundle's `pred`. The `cost` of the returned bundle is recomputed as the
        minimal sum of per-hop costs along any valid path from `self.src_node`
        to `new_dst_node`, where each hop cost is the minimum of `cost_attr`
        across the parallel edges recorded for that hop.

        Args:
            new_dst_node: The new destination node, which must be present in `pred`.
            graph: The underlying graph to look up edge attributes.
            cost_attr: The edge attribute representing cost/metric.

        Returns:
            A new PathBundle from `self.src_node` to `new_dst_node` with an updated cost.

        Raises:
            ValueError: If `new_dst_node` is not found in this bundle's `pred` map.
        """
        if new_dst_node not in self.pred:
            raise ValueError(f"{new_dst_node} not in this PathBundle's pred")

        edges_dict = graph.get_edges()

        # 1) Build the restricted predecessor subgraph reachable from new_dst_node
        #    back to self.src_node. This preserves all allowed predecessors without
        #    collapsing to a single path.
        new_pred: Dict[NodeID, Dict[NodeID, List[EdgeID]]] = {self.src_node: {}}
        visited_build: Set[NodeID] = set([new_dst_node])
        queue_build: deque[NodeID] = deque([new_dst_node])
        while queue_build:
            node = queue_build.popleft()
            for prev_node, edges_list in self.pred[node].items():
                new_pred.setdefault(node, {})[prev_node] = edges_list
                if prev_node != self.src_node and prev_node not in visited_build:
                    visited_build.add(prev_node)
                    queue_build.append(prev_node)

        # 2) Compute minimal cost from self.src_node to new_dst_node over new_pred
        #    using Dijkstra on the reversed edges (from dst backwards to src).
        def hop_cost(u: NodeID, v: NodeID) -> float:
            # cost to go from u<-v (i.e., v -> u in forward direction)
            edges_list = new_pred[u][v]
            return float(min(edges_dict[eid][3][cost_attr] for eid in edges_list))

        # Trivial case: src == dst
        if new_dst_node == self.src_node:
            return PathBundle(self.src_node, new_dst_node, new_pred, 0.0)

        dist: Dict[NodeID, float] = {new_dst_node: 0.0}
        heap: List[Tuple[float, NodeID]] = [(0.0, new_dst_node)]
        best_cost: float = float("inf")

        while heap:
            cost_to_node, node = heappop(heap)
            if cost_to_node > dist.get(node, float("inf")):
                continue
            if node == self.src_node:
                best_cost = cost_to_node
                break
            # Relax predecessors of `node` (reverse traversal)
            for prev_node in new_pred.get(node, {}):
                c = cost_to_node + hop_cost(node, prev_node)
                if c < dist.get(prev_node, float("inf")):
                    dist[prev_node] = c
                    heappush(heap, (c, prev_node))

        # If src_node was not reached, this subgraph does not connect src->new_dst.
        # Treat as an error to avoid silent mis-reporting.
        if best_cost == float("inf"):
            raise ValueError(
                f"No path from '{self.src_node}' to '{new_dst_node}' within this PathBundle."
            )

        return PathBundle(self.src_node, new_dst_node, new_pred, float(best_cost))
