# AUTOGENERATED! DO NOT EDIT! File to edit: 02_manifests.ipynb (unless otherwise specified).

__all__ = ['id_generator', 'create_manifest', 'merge_secret_manifest_with_ctx_index']

# Cell

import itertools
import string


def id_generator(dataset_id):
    """Create anonymous filenames for a new Zooniverse dataset.

    Parameters
    ----------
    dataset_id: str
        Dataset identifier. It starts with 4 characters to identify the project: APFT for "Anonymous Planet Four Terrains", APFH for "Anonymous Planet Four HiRISE" (didn't want APFO for original for readability), APFR for Ridges. This shall be followed by a 2 digit number enumerating a data set, e.g. 6 or 61, 62 for six/1, six/2. In total it then looks like APFT61 for example.
    """
    for newid in itertools.product(string.digits + "abcdefg", repeat=8):
        yield dataset_id + "".join(newid) + ".png"

# Cell

from shutil import copy2 as copy

import pandas as pd
from planetarypy.pds.apps import get_index


def create_manifest(saveroot, product_ids, dataset_id):
    """Copy anonymized tiles to a submit folder and create manifests.

    Parameters
    ----------
    saveroot: str, pathlib.Path
        Path to dataset folder
    product_ids: list-like
        List of product_ids included in this dataset
    dataset_id: str
        Short string that identifies this dataset, following our Zooniverse anomymized ID signature, like APFT61, or APFR03, or APFH10 for HiRISE datasets.
    """
    # subfolder in the dataset folder where the to-be-submitted
    # files will be stored:
    to_submit_folder = saveroot / "to_submit"
    to_submit_folder.mkdir(exist_ok=True)

    png_ids = id_generator(dataset_id)

    manifest = (saveroot / "manifest.csv").open("w")

    secret_manifest_fname = f"secret_manifest_small_{saveroot.name}.csv"
    secret_manifest = (saveroot / secret_manifest_fname).open("w")
    secret_manifest.write("source,APF_id,product_id\n")

    for pid in sorted(product_ids):
        files = sorted(list((saveroot / pid).glob("*.png")))
        for f, png_id in zip(files, png_ids):
            savepath = to_submit_folder / png_id
            copy(f, savepath)
            secret_manifest.write(f"{f.name},")
            secret_manifest.write(f"{png_id},")
            secret_manifest.write(f"{pid}\n")
            manifest.write(f"{png_id}\n")

    manifest.close()
    secret_manifest.close()

    # create 3rd form of manifest
    secret = pd.read_csv(saveroot / secret_manifest_fname)
    prefix = "http://viewer.mars.asu.edu/viewer/ctx#P={}&T=2"
    secret["!Public_CTX_link"] = secret["product_id"].map(lambda x: prefix.format(x))
    new_manifest = secret.drop("source", axis=1)
    new_manifest.columns = ["!filename", "!CTX_filename", "!Public_CTX_link"]
    new_manifest_fname = f"new_manifest_{saveroot.name}.csv"
    new_manifest.to_csv(saveroot / new_manifest_fname, index=False)


def merge_secret_manifest_with_ctx_index(saveroot):
    small_mani_fname = saveroot / f"secret_manifest_small_{saveroot.name}.csv"
    mani_fname = saveroot / f"secret_manifest_with_meta_{saveroot.name}.csv"
    manifest = pd.read_csv(small_mani_fname)
    #     return manifest
    ctxindex = get_index("mro.ctx.indexes.edr")
    manifest = manifest.merge(ctxindex, left_on="product_id", right_on="PRODUCT_ID")
    manifest.to_csv(mani_fname, index=False)