"use strict";
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'add stacked property to graphs'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'Test widget',
            stacked: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'Test widget',
                    region: { Ref: 'AWS::Region' },
                    stacked: true,
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add metrics to graphs on either axis'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'label and color are respected in constructor'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test', label: 'MyMetric', color: '000000' })],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { label: 'MyMetric', color: '000000' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'singlevalue widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                },
            }]);
        test.done();
    },
    'alarm widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }).createAlarm(stack, 'Alarm', {
            evaluationPeriods: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.AlarmWidget({
            alarm,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    annotations: {
                        alarms: [{ 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add annotations to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            leftAnnotations: [{
                    value: 1000,
                    color: '667788',
                    fill: lib_1.Shading.BELOW,
                    label: 'this is the annotation',
                }],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    annotations: { horizontal: [{
                                yAxis: 'left',
                                value: 1000,
                                color: '667788',
                                fill: 'below',
                                label: 'this is the annotation',
                            }] },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'convert alarm to annotation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        const alarm = metric.createAlarm(stack, 'Alarm', {
            evaluationPeriods: 7,
            datapointsToAlarm: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.GraphWidget({
            right: [metric],
            rightAnnotations: [alarm.toAnnotation()],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'right' }],
                    ],
                    annotations: {
                        horizontal: [{
                                yAxis: 'right',
                                value: 1000,
                                label: 'Test >= 1000 for 2 datapoints within 35 minutes',
                            }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add yAxis to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
            leftYAxis: ({
                label: 'Left yAxis',
                max: 100,
            }),
            rightYAxis: ({
                label: 'Right yAxis',
                min: 10,
                showUnits: false,
            }),
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {
                        left: { label: 'Left yAxis', max: 100 },
                        right: { label: 'Right yAxis', min: 10, showUnits: false }
                    },
                },
            }]);
        test.done();
    },
    'can use imported alarm with graph'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = lib_1.Alarm.fromAlarmArn(stack, 'Alarm', 'arn:aws:cloudwatch:region:account-id:alarm:alarm-name');
        // WHEN
        new lib_1.AlarmWidget({
            title: 'My fancy graph',
            alarm,
        });
        // THEN: Compiles
        test.done();
    },
    'add setPeriodToTimeRange to singleValueWidget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
            setPeriodToTimeRange: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    setPeriodToTimeRange: true,
                },
            }]);
        test.done();
    },
    'allows overriding custom values of dashboard widgets'(test) {
        class HiddenMetric extends lib_1.Metric {
            toMetricConfig() {
                const ret = super.toMetricConfig();
                // @ts-ignore
                ret.renderingProperties.visible = false;
                return ret;
            }
        }
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [
                new HiddenMetric({ namespace: 'CDK', metricName: 'Test' }),
            ],
        });
        // test.ok(widget.toJson()[0].properties.metrics[0].visible === false);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { visible: false }]);
        test.done();
    },
    'GraphColor is correctly converted into the correct hexcode'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.GraphWidget({
            left: [metric.with({
                    color: lib_1.Color.BLUE,
                })],
            leftAnnotations: [
                { color: lib_1.Color.RED, value: 100 },
            ],
        });
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { color: '#1f77b4' }]);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.annotations.horizontal[0], { yAxis: 'left', value: 100, color: '#d62728' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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