"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn],
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                    },
                },
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    },
                },
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange,
                },
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
/**
 * Fill shading options that will be used with an annotation
 */
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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