# -*- coding: utf-8 -*-
#
# This file is part of the parce Python package.
#
# Copyright © 2019-2020 by Wilbert Berendsen <info@wilbertberendsen.nl>
#
# This module is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This module is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


"""
Updates the language stub files in source/lang and the listing in source/langs.inc
"""

EXAMPLES_DIRECTORY = "../tests/lang/"


import collections
import glob
import pprint
import io
import os
import sys

sys.path.insert(0, "..")

import parce.transform
import parce.registry
from parce.out.html import HtmlFormatter, escape


def get_all_modules():
    """Return a list of all the modules below parce.lang containing bundled language definitions."""
    return sorted(set(name.split('.')[-3]
        for name in parce.registry.registry
            if name.startswith('parce.lang.')))


def get_languages(name):
     """Yield the Language subclasses defined in the module parce.lang.``name``."""
     for fullname in parce.registry.registry:
         if fullname.split('.')[-3] == name:
             yield parce.registry.registry.lexicon(fullname).language


STUB = r"""
.. Generated by ../update_languages.py -- do not edit!

{title}

.. automodule:: parce.lang.{module}
   :members:
   :undoc-members:
   :show-inheritance:

   In this module:
   ---------------

{registry}

{examples}

"""

REGISTRY_STUB_HEADER = r"""
.. list-table::
   :header-rows: 1
   :widths: 10, 10, 40, 20, 20

   * - Language
     - Name (Aliases)
     - Description
     - Filename(s)
     - Mime Type(s)
"""

REGISTRY_STUB = r"""
   * - :class:`~parce.lang.{module}.{lang}`
     - {name}
     - {desc}
     - {fnames}
     - {mtypes}
"""

EXAMPLE_STUB = r"""
Root lexicon ``{root}`` and text:

.. admonition:: Text rendered using default theme

   .. raw:: html

{code}

Result tree:

.. code-block:: none
   :class: token-tree

{result}
"""

TRANSFORM_STUB = r"""
Transformed result (pretty-printed)::

{result}
"""

HTML_STUB = r"""<pre style="overflow: scroll; white-space: pre; {baseformat}">{html}</pre>"""

LANGS_INC_HEADER = r"""
.. Generated by ../update_languages.py -- do not edit!

.. list-table::
   :header-rows: 1
   :widths: 20 70

   * - Module
     - Languages

"""

EXAMPLE_THEME_STUB_HTML = r"""
<div id="all-themed-examples"
     style="padding: 10px; background: lightgray; border-radius: 4px;">
<script type="text/javascript">
/*
 * select the example corresponding to selected language and theme.
 */
function update_theme() {

    var examples = document.getElementById("all-examples").children;
    var lang = document.getElementById("language-selector").value;
    var theme = document.getElementById("theme-selector").value;
    var show = lang + "_" + theme;

    for (var i = 0; i < examples.length; i++) {
        box = examples[i]
        if (box.className == show) {
            box.style.display = "block";
        } else {
            box.style.display = "none";
        }
    }
}

function update_language() {

    var trees = document.getElementById("all-tokentrees").children;
    var lang = document.getElementById("language-selector").value;
    var show = "tokentree_" + lang;

    for (var i = 0; i < trees.length; i++) {
        box = trees[i]
        if (box.className == show) {
            box.style.display = "block";
        } else {
            box.style.display = "none";
        }
    }

    update_theme();
}

function initialize_themes() {
    update_language();
}
</script>
Language:
<select id="language-selector" name="language" onchange="update_language()">
@LANGUAGES@
</select>
Theme:
<select id="theme-selector" name="theme" onchange="update_theme()">
@THEMES@
</select>

<div id="all-examples" style="margin: 10px 0;">
@EXAMPLES@
</div>
<div id="all-tokentrees" style="margin: 10px 0 0;">
@TOKENTREES@
</div>
</div> <!--all-examples-->

<script type="text/javascript">
    initialize_themes();
</script>
</div> <!--all-themed-examples-->
"""


# mapping from the example files to their Language
all_examples = collections.defaultdict(list)


# default theme
default_theme = parce.theme_by_name('default')

# default formatter for the examples
default_formatter = HtmlFormatter(default_theme)


def title(text, char='-'):
    """Return text with a line of hyphens of the same length below it."""
    return text + '\n' + char * len(text)


def indent(text, indent=3):
    """Indent text."""
    return '\n'.join(' ' * indent + line for line in text.splitlines())


def make_html(doc, formatter=None):
    """Formats Document as HTML using formatter.

    If formatter is None, the default formatter is used.

    """
    f = default_formatter if formatter is None else formatter
    cursor = parce.Cursor(doc).select_all()
    return HTML_STUB.format(
                baseformat=f.baseformat() or "",
                html=f.html(cursor))


def make_stub(module):
    """Writes a documentation page for a language module."""
    # registry listing, if any
    registered = []
    langs = list(get_languages(module))
    if langs:
        registered.append(REGISTRY_STUB_HEADER)
        langs.sort(key=lambda l: l.__name__)
        for l in langs:
            rootname = parce.registry.registry.qualname(l.__name__)
            if rootname:
                r = parce.registry.registry[rootname]
                name = r.name
                if r.aliases:
                    name += ' ({})'.format(', '.join(r.aliases))
                desc = r.desc
                fnames = ', '.join('``{}``'.format(fname) for fname, weight in r.filenames)
                mtypes = ', '.join('``{}``'.format(mtype) for mtype, weight in r.mimetypes)
            else:
                name, desc, fnames, mtypes = '', '(not registered)', '', ''
            registered.append(REGISTRY_STUB.format(module=module, lang=l.__name__,
                name=name, desc=desc, fnames=fnames, mtypes=mtypes))

    # examples, if any
    examples = sum((all_examples.get(lang, []) for lang in langs), [])
    example = []
    if examples:

        example.append(title("Examples:" if len(examples) > 1 else "Example:"))

        for root_lexicon, text in examples:
            doc = parce.Document(root_lexicon, text, transformer=True)
            buf = io.StringIO()
            doc.get_root(True).dump(buf)
            html = make_html(doc)

            text = EXAMPLE_STUB.format(root=root_lexicon, language=module,
                        code=indent(html, 6), result=indent(buf.getvalue()))

            tf_result = doc.get_transform(True)
            if tf_result is not None:
                text += TRANSFORM_STUB.format(result=indent(pprint.pformat(tf_result)))

            example.append(text)

    text = STUB.format(
        title=module + '\n' + '=' * len(module),
        module=module,
        registry=indent(''.join(registered)),
        examples=''.join(example))

    with open("source/lang/{0}.rst".format(module), "w") as f:
        f.write(text)


def get_examples():
    """Return a list of all example documents in tests/lang/example.*."""
    pattern = os.path.join(EXAMPLES_DIRECTORY, "example*.*")
    return glob.glob(pattern)


def load_examples():
    """Load the examples in the global ``all_examples`` variable."""
    # make a mapping from the example files to their Language
    for filename in get_examples():
        text = open(filename).read()    # TODO encoding?
        root_lexicon = parce.find(filename=filename, contents=text)
        all_examples[root_lexicon.language].append((root_lexicon, text))


def write_langs_and_module_stubs():
    """Write the source/langs.inc file and all the module doc files"""
    with open("source/langs.inc", "w") as f:
        f.write(LANGS_INC_HEADER)
        for name in get_all_modules():
            langs = list(get_languages(name))
            if langs:
                def class_link(lang):
                    """Return a class link to the lang, with * if there is also a transform."""
                    txt = ":class:`~parce.lang.{0}.{1}`".format(name, lang.__name__)
                    if parce.transform.Transformer().find_transform(lang):
                        txt += ":class:`\* <parce.lang.{0}.{1}Transform>`".format(name, lang.__name__)
                    return txt
                clss = ", ".join(class_link(lang) for lang in langs)
                make_stub(name)
                f.write("   * - :mod:`~parce.lang.{0}`\n".format(name))
                f.write("     - {0}\n\n".format(clss))


def format_all_examples():
    """Format all examples, using all themes."""
    import parce.themes

    themes = list(sorted(parce.themes.get_all_themes()))
    # default at first and debug at end
    themes = ["default"] + [t for t in themes if t not in ("default", "debug")] + ["debug"]

    all_themes = [
        (name, HtmlFormatter(parce.theme_by_name(name)))
            for name in themes]

    languages = []
    examples_html = []
    tokentrees_html = []

    for lang in sorted(all_examples, key=lambda l: l.__name__):
        langname = lang.__name__
        lang_class = langname.lower()
        examples = all_examples[lang]
        for n, (root_lexicon, text) in enumerate(examples):
            if n > 0:
                langname += " ({})".format(n + 1)
                lang_class += format(n)
            languages.append((lang_class, langname))
            doc = parce.Document(root_lexicon, text)
            # themed boxes
            for name, f in all_themes:
                html = '<pre style="padding: 2px; height: 300px; overflow: auto; {3}" class="{0}_{1}">{2}</pre>\n'.format(
                    lang_class, name, f.html(parce.Cursor(doc, 0, None)), f.baseformat())
                examples_html.append(html)
            # tokentree box
            buf = io.StringIO()
            doc.get_root(True).dump(buf)
            html = '<pre style="padding: 2px; height: 300px; overflow: auto; background: #fcfefc; font-family: monospace;" class="tokentree_{0}">{1}</pre>\n'.format(
                lang_class, escape(buf.getvalue()))
            tokentrees_html.append(html)

    text = EXAMPLE_THEME_STUB_HTML\
        .replace("@THEMES@", '\n'.join(map('<option value="{0}">{0}</option>'.format, themes))) \
        .replace("@LANGUAGES@", '\n'.join('<option value="{0}">{1}</option>'.format(*l) for l in languages)) \
        .replace("@EXAMPLES@", '\n'.join(examples_html)) \
        .replace("@TOKENTREES@", '\n'.join(tokentrees_html))

    with open("source/examples.html", "w") as f:
        f.write(text)


def main():
    """Main function."""
    load_examples()
    write_langs_and_module_stubs()
    format_all_examples()


if __name__ == "__main__":
    main()

