#  _________________________________________________________________________
#
#  Kipet: Kinetic parameter estimation toolkit
#  Copyright (c) 2016 Eli Lilly.
#  _________________________________________________________________________

# Sample Problem 1 (From Sawall et.al.)
# Basic simulation of ODE system using multistep-integrator
#
#		\frac{dC_a}{dt} = -k*C_a	C_a(0) = 1
#		\frac{dC_b}{dt} = k*C_a		C_b(0) = 0


from kipet.library.TemplateBuilder import *
from kipet.library.CasadiSimulator import *
import matplotlib.pyplot as plt
import sys

if __name__ == "__main__":

    with_plots = True
    if len(sys.argv)==2:
        if int(sys.argv[1]):
            with_plots = False

    
    # create template model 
    builder = TemplateBuilder()    
    builder.add_mixture_component('A',1)
    builder.add_mixture_component('B',0)
    builder.add_algebraic_variable('ra')
    builder.add_parameter('k',0.01)
    
    
    # define explicit system of ODEs
    def rule_odes(m,t):
        exprs = dict()
        exprs['A'] = -m.Y[t,'ra']
        exprs['B'] = m.Y[t,'ra']
        return exprs

    builder.set_odes_rule(rule_odes)

    def rule_algebraics(m,t):
        algebraics = list()
        algebraics.append(m.Y[t,'ra']-m.P['k']*m.Z[t,'A'])
        return algebraics
    builder.set_algebraics_rule(rule_algebraics)

    # create an instance of a casadi model template
    casadi_model = builder.create_casadi_model(0.0,200.0)    

    # create instance of simulator
    sim = CasadiSimulator(casadi_model)
    # defines the discrete points wanted in the concentration profile
    sim.apply_discretization('integrator',nfe=200)
    # simulate
    results_casadi = sim.run_sim("idas")

    # display concentration results
    if with_plots:
        results_casadi.Z.plot.line(legend=True)
        plt.xlabel("time (s)")
        plt.ylabel("Concentration (mol/L)")
        plt.title("Concentration Profile")

        results_casadi.Y.plot.line(legend=True)
        plt.xlabel("time (s)")
        plt.ylabel("Algebraics")
        plt.title("Algebraics")
        
        plt.show()
        
