from types import ModuleType
from importlib import import_module

from .seed import seed_everything
from .debug import is_debug_enabled, debug, set_debug

import torch_geometric.data
import torch_geometric.loader
import torch_geometric.transforms
import torch_geometric.utils
import torch_geometric.profile


# https://github.com/tensorflow/tensorflow/blob/master/tensorflow/
# python/util/lazy_loader.py
class LazyLoader(ModuleType):
    def __init__(self, local_name, parent_module_globals, name):
        self._local_name = local_name
        self._parent_module_globals = parent_module_globals
        super(LazyLoader, self).__init__(name)

    def _load(self):
        module = import_module(self.__name__)
        self._parent_module_globals[self._local_name] = module
        self.__dict__.update(module.__dict__)
        return module

    def __getattr__(self, item):
        module = self._load()
        return getattr(module, item)

    def __dir__(self):
        module = self._load()
        return dir(module)


datasets = LazyLoader('datasets', globals(), 'torch_geometric.datasets')
nn = LazyLoader('nn', globals(), 'torch_geometric.nn')
graphgym = LazyLoader('graphgym', globals(), 'torch_geometric.graphgym')

__version__ = '2.0.0'

__all__ = [
    'seed_everything',
    'is_debug_enabled',
    'debug',
    'set_debug',
    'torch_geometric',
    '__version__',
]
