# -*- coding: utf-8 -*-
'''Chemical Engineering Design Library (ChEDL). Utilities for process modeling.
Copyright (C) 2016, 2017 Caleb Bell <Caleb.Andrew.Bell@gmail.com>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.'''

from numpy.testing import assert_allclose
from fluids.numerics import assert_close, assert_close1d
import pytest
import numpy as np
from fluids.design_climate import *
from fluids.design_climate import _latlongs, stations
import os

try:
    import geopy
    has_geopy = True
except:
    has_geopy = False

data_dir_override = os.path.join(os.path.dirname(__file__), 'gsod')


def test_heating_degree_days():
    assert_close(heating_degree_days(273, truncate=False), -18.483333333333292)
    assert 0 == heating_degree_days(273, truncate=True)
    assert 0 == heating_degree_days(273)
    assert_close(heating_degree_days(273, T_base = 250), 23)
    assert_close(heating_degree_days(279, T_base=300, truncate=False), -21)

def test_cooling_degree_days():

    assert_close( cooling_degree_days(250), 33.15)
    assert 0 == cooling_degree_days(300)
    assert_close(cooling_degree_days(300, truncate=False), -16.85)
    assert_close(cooling_degree_days(250, T_base=300), 50)


@pytest.mark.slow
def test_month_average_temperature():
    station = get_closest_station(38.8572, -77.0369)
    station_data = StationDataGSOD(station, data_dir_override=data_dir_override)
    Ts_calc = station_data.month_average_temperature(1990, 2000, include_yearly=False, minimum_days=23)
    Ts_expect = [276.1599380905833, 277.5375516246206, 281.1881231671554, 286.7367003367004, 291.8689638318671, 296.79545454545456, 299.51868686868687, 298.2097914630174, 294.4116161616162, 288.25883023786247, 282.3188552188553, 277.8282339524275]
    assert_close1d(Ts_calc, Ts_expect, rtol=1E-3)

    assert station_data.warmest_month(1990, 2000) == 6
    assert station_data.coldest_month(1990, 2000) == 0



def test_IntegratedSurfaceDatabaseStation():

    # Information confirmed elsewhere i.e. https://geographic.org/global_weather/not_specified_canada/calgary_intl_cs_713930_99999.html
    values = [713930.0, 99999.0, 'CALGARY INTL CS', 'CA', None, None, 51.1, -114.0, 1081.0, 20040921.0, 20150831.0]
    test_station = IntegratedSurfaceDatabaseStation(*values)
    for value, attr in zip(values, test_station.__slots__):
        assert value == getattr(test_station, attr)

def test_data():
    assert _latlongs.shape[0] >= 27591
    for station in stations:
        assert abs(station.LAT) <= 90
        assert abs(station.LON) <= 180


@pytest.mark.slow
def test_correct_WBAN():
    station = get_closest_station(31.9973, -102.0779)
    station_data = StationDataGSOD(station, data_dir_override=data_dir_override)
    assert station.WBAN == '03071'
    assert station_data.month_average_temperature(2010, 2011, include_yearly=False)



def test_get_closest_station():
    s = get_closest_station(51.02532675, -114.049868485806, 20150000)
    assert s.NAME == 'CALGARY INTL CS'

    with pytest.raises(Exception):
         get_closest_station(51.02532675, -114.049868485806, 90150000)


sample_data_random_station_1999 = '''STN--- WBAN   YEARMODA    TEMP       DEWP      SLP        STP       VISIB      WDSP     MXSPD   GUST    MAX     MIN   PRCP   SNDP   FRSHTT
712650 99999  19990101    12.3 24     3.1 24  1022.8 24  1013.0 24    8.4 24   15.0 24   22.9   29.9    23.7     4.1   0.00G 999.9  001000
712650 99999  19990102    10.1 24    -0.6 24  1034.1 24  1024.1 24    6.8 24   12.7 24   29.9   35.0    23.0*    5.0*  0.00G 999.9  001000
712650 99999  19990103    28.1 24    24.9 24  1007.5 24   997.9 24    3.6 24   24.2 24   36.9   43.9    35.6*   21.2*  0.71G 999.9  001000
712650 99999  19990104    17.8 24    10.8 24  1011.0 24  1001.3 24    8.8 24   26.1 24   30.9   42.0    36.7    13.3   0.04G 999.9  001000
712650 99999  19990105    16.3 22     9.3 22  1022.9 22  1013.1 22    8.0 22   15.4 22   22.9   38.1    20.1    13.3   0.00G 999.9  001000
712650 99999  19990106    19.3 24    14.4 24  1016.3 24  1006.6 24    6.2 24   19.7 24   28.0   34.0    28.4*   10.4*  0.00G 999.9  001000
712650 99999  19990107    17.4 23     8.6 23  1019.5 23  1009.8 23    8.9 23   20.9 23   23.9   35.0    29.1    10.9   0.02G 999.9  001000
712650 99999  19990108    15.5 24     9.8 24  1026.3 24  1016.5 24    7.0 24    9.3 24   20.0   26.0    21.2*   10.4*  0.00G 999.9  001000
712650 99999  19990109    20.8 24    16.2 24  1012.1 24  1002.4 24    3.8 24    9.3 24   18.1   23.9    24.8*   19.4*  0.04G 999.9  001000
712650 99999  19990111    12.2 24     4.7 24  1016.8 24  1007.0 24    7.7 24   14.4 24   27.0   38.1    20.7     7.5   0.02G 999.9  000000
712650 99999  19990112    20.8 24    15.8 24  1011.8 24  1002.1 24    4.6 24    7.8 24   15.9   19.0    24.8     7.5   0.00G 999.9  001000
712650 99999  19990113    10.3 24     2.7 24  1026.1 24  1016.3 24    3.9 24   11.7 24   19.0   23.9    22.5*   -0.4*  0.20G 999.9  001000
712650 99999  19990114    -0.7 24    -9.1 24  1034.7 24  1024.5 24    2.1 24   20.1 24   34.0   42.0    15.8*   -7.6*  0.00G 999.9  001000
712650 99999  19990115    13.4 22     9.2 22  1016.2 22  1006.4 22    3.2 22   13.5 22   20.0   42.0    19.4*    8.6*  0.55G 999.9  001000
712650 99999  19990116    25.4 24    21.7 24  1009.2 24   999.7 24    5.6 24   18.8 24   26.0   32.1    35.6*   12.2*  0.02G 999.9  001000
712650 99999  19990117    33.3 24    28.0 24  1017.6 24  1008.1 24    6.1 24    8.9 24   19.0   32.1    37.4*   29.8*  0.16G 999.9  000000
712650 99999  19990118    36.6 24    33.9 24  1006.8 24   997.5 24    6.8 24   13.7 24   18.1   21.0    41.4    27.7   0.26G 999.9  010000
712650 99999  19990119    34.2 24    28.5 24  1006.3 24   996.9 24    8.7 24   22.6 24   30.9   36.9    41.4    32.5   0.02G 999.9  000000
712650 99999  19990120    32.3 24    25.8 24  1017.6 24  1008.2 24    8.5 24   12.1 24   20.0   38.1    33.8*   30.2*  0.00G 999.9  000000
712650 99999  19990121    32.0 24    27.5 24  1018.7 24  1009.3 24    7.2 24    4.6 24   11.1   21.0    37.4*   26.6*  0.00G 999.9  000000
712650 99999  19990122    34.0 19    31.6 19  1020.7 19  1011.2 19    6.7 19   21.5 19   28.9   37.9    36.7    26.1   0.02G 999.9  010000
712650 99999  19990123    40.1 24    39.1 24  1013.3 24  1003.9 24    4.8 24    9.9 24   21.0   39.0    48.2*   36.3*  0.33G 999.9  010000
712650 99999  19990124    38.9 24    34.0 24  1009.7 23  1000.3 23    7.7 24   14.4 24   22.9   28.9    48.2*   33.8*  0.12G 999.9  011000
712650 99999  19990125    31.7 23    24.9 23  1021.4 23  1011.8 23    6.3 23    8.1 23   18.1   31.1    41.7    29.3   0.00G 999.9  001000
712650 99999  19990126    31.3 24    23.7 24  1028.3 24  1018.6 24    9.0 24   11.2 24   18.1   25.1    32.7    27.3   0.00I 999.9  000000
712650 99999  19990127    30.5 23    25.8 23  1021.1 23  1011.5 23    6.0 23   11.3 23   23.9   32.1    34.3    22.1   0.00G 999.9  010000
712650 99999  19990128    28.2 23    25.9 23  1015.2 23  1005.7 23    4.4 23   15.0 23   23.9   32.1    34.3    22.1   0.08G 999.9  011000
712650 99999  19990129    21.9 24    16.4 24  1028.6 24  1018.8 24    7.8 24    4.1 24    9.9   21.0    31.3    18.3   0.02G 999.9  001000
712650 99999  19990130    30.1 23    22.0 23  1033.4 23  1023.7 23    7.9 23    9.1 23   14.0   18.1    33.8*   26.6*  0.02G 999.9  000000
712650 99999  19990131    27.4 24    16.6 24  1042.4 24  1032.6 24    9.0 24    9.9 24   15.0   21.0    34.0    24.6   0.00G 999.9  000000
712650 99999  19990201    28.0 24    24.4 24  1030.5 24  1022.4  4    7.4 24    8.5 24   15.9   18.1    33.8*   23.0*  0.00G 999.9  000000
712650 99999  19990202    34.9 24    34.3 24  1012.5 22  1004.2  4    1.7 24    8.7 24   15.0   20.0    37.4*   33.6*  0.04G 999.9  010000
712650 99999  19990203    36.9 24    32.0 24  1009.6 24   998.9  4    9.0 24   12.5 24   16.9   25.1    40.3    33.4   0.14G 999.9  000000
712650 99999  19990204    37.2 24    32.1 24  1008.1 21   998.7  4    8.6 24    9.3 24   21.0   33.0    39.6*   33.8*  0.00G 999.9  010000
712650 99999  19990205    25.7 23    14.5 23  1027.2 23  9999.9  0    9.0 23    9.5 23   23.9   35.9    41.4    21.2   0.06G 999.9  000000
712650 99999  19990206    33.5 24    27.4 24  1014.9 21  1006.3  4    6.8 24    6.7 24   11.1   20.0    39.2*   27.9*  0.00G 999.9  001000
712650 99999  19990207    33.7 24    25.9 24  1015.5 22  1006.2  4    8.4 24   10.0 24   25.1   28.0    37.4*   30.2*  0.02G 999.9  000000
712650 99999  19990208    26.8 21    16.6 21  1016.7 21  9999.9  0    9.0 21    7.2 21   15.9   29.9    34.5    19.8   0.00G 999.9  000000
712650 99999  19990209    38.2 23    30.3 23  1012.0 23  1003.0  4    8.5 23   13.1 23   22.9   28.9    48.2*   32.0*  0.00G 999.9  000000
712650 99999  19990210    37.2 24    26.4 24  1020.4 24  1009.4  4    9.0 24    9.8 24   16.9   35.9    48.4    30.7   0.02G 999.9  000000
712650 99999  19990211    35.6 24    32.6 24  1020.1 23  1011.9  4    8.0 24    9.2 24   15.0   24.1    41.5    30.2   0.00G 999.9  010000
712650 99999  19990212    38.8 24    34.8 24  1006.6 24   997.3  4    7.5 24   13.7 24   25.1   40.0    50.0    31.3   0.02G 999.9  011000
712650 99999  19990213    25.2 24    15.8 24  1013.8 21  1002.8  4    8.1 24   18.3 24   22.9   45.1    32.0*   20.8*  0.18G 999.9  001000
712650 99999  19990214    23.0 24     7.9 24  1028.4 24  1018.2  4    9.0 24    7.6 24   15.0   32.1    32.0*   16.9*  0.00G 999.9  000000
712650 99999  19990215    34.5 21    26.4 21  1021.5 18  1013.0  4    8.9 21   11.6 21   18.1  999.9    41.0*   30.2*  0.00I 999.9  000000
712650 99999  19990216    36.6 24    32.6 24  1016.2 24  1007.4  4    7.6 24    7.6 24   14.0   20.0    42.3    28.9   0.00G 999.9  000000
712650 99999  19990217    37.4 24    33.3 24  1010.6 20  1001.2  4    5.8 24   12.8 24   22.0   28.0    42.3    31.6   0.06G 999.9  010000
712650 99999  19990218    33.2 24    24.0 24  1014.6 23  1004.7  4    8.3 24   10.1 24   15.9   24.1    41.2    29.8   0.02G 999.9  000000
712650 99999  19990219    24.7 24    11.2 24  1019.0 24  1009.2  4    9.0 24    8.5 24   12.0   19.0    34.2    19.0   0.00G 999.9  000000
712650 99999  19990220    23.6 24    11.2 24  1020.3 23  1010.4  4    9.0 24    8.8 24   13.0   16.9    30.2*   18.9*  0.00I 999.9  000000
712650 99999  19990221    21.0 24     2.7 24  1023.8 24  1013.6  4    9.0 24   12.6 24   15.9   26.0    30.6    15.4   0.00G 999.9  000000
712650 99999  19990222    13.1 23    -9.4 23  1031.8 23  1021.4  4    8.9 23    8.2 23   15.9   25.1    26.2     4.5   0.00G 999.9  000000
712650 99999  19990223    18.9 24     1.1 24  1032.5 24  1022.8  4    8.9 24   10.1 24   18.1   21.0    25.7     4.5   0.02G 999.9  000000
712650 99999  19990224    26.5 23    13.4 23  1028.9 22  1019.8  4    8.9 23   13.4 23   18.1   23.9    32.0*   21.2*  0.00G 999.9  000000
712650 99999  19990225    33.4 24    22.1 24  1021.3 24  1012.3  4    9.0 24   11.0 24   20.0  999.9    35.6*   32.0*  0.00I 999.9  000000
712650 99999  19990226    33.5 24    20.7 24  1021.9 24  1012.3  4    9.0 24    7.5 24   12.0   24.1    42.6    26.1   0.00G 999.9  000000
712650 99999  19990227    32.8 24    26.0 24  1017.3 24  1008.7  4    4.7 24    7.1 24   16.9  999.9    42.6    25.9   0.00I 999.9  000000
712650 99999  19990228    37.7 24    35.8 24   998.6 23   990.9  4    4.3 24   11.6 24   16.9   22.0    41.9    25.9   0.37G 999.9  010000
712650 99999  19990301    36.3 24    32.1 24   994.7 23   985.0  4    6.8 24   10.2 24   22.0   35.0    41.0*   32.0*  0.10G 999.9  010000
712650 99999  19990302    33.2 24    19.5 24  1005.5 23  9999.9  0    9.0 24   13.2 24   21.0   30.9    41.0*   24.8*  0.00G 999.9  001000
712650 99999  19990303    34.6 24    31.5 24  1004.4 24   996.3  4    5.6 24   11.8 24   21.0   29.9    42.3    25.0   0.02G 999.9  010000
712650 99999  19990304    28.0 24    18.7 24  1003.4 24   991.9  4    9.0 24   21.1 24   29.9   39.0    36.7    24.8   0.33G 999.9  001000
712650 99999  19990305    23.1 24    10.9 24  1026.0 24  1015.0  4    9.0 24    9.2 24   19.0   42.0    31.8    17.6   0.02G 999.9  000000
712650 99999  19990306    20.1 24    13.9 24  1023.4 21  1014.4  4    2.2 24   19.0 24   27.0   35.0    27.0    15.3   0.06G 999.9  001000
712650 99999  19990307    13.8 24    -2.9 24  1034.1 24  1022.7  4    8.7 24   13.2 24   22.0   36.9    25.3     6.6   0.04G 999.9  001000
712650 99999  19990308    18.6 24     3.4 24  1040.4 24  1030.7  4    9.0 24    6.2 24    9.9   28.9    26.6     6.6   0.00G 999.9  000000
712650 99999  19990309    24.0 24    10.8 24  1029.5 23  1021.2  4    9.0 24   12.3 24   25.1   29.9    27.5    10.9   0.00G 999.9  001000
712650 99999  19990310    24.4 24     5.7 24  1020.0 24  1010.5  4    9.0 24   10.9 24   15.9   32.1    28.4*   18.5*  0.06G 999.9  000000
712650 99999  19990311    28.9 24    14.6 24  1020.3 24  1010.8  4    9.0 24   11.0 24   19.0   28.0    33.8*   25.9*  0.02G 999.9  000000
712650 99999  19990312    28.2 24    13.8 24  1022.6 24  1012.6  4    9.0 24   13.5 24   19.0   28.9    34.0    22.8   0.00G 999.9  000000
712650 99999  19990313    29.5 24    12.7 24  1027.6 24  1017.9  4    9.0 24    7.9 24   14.0   28.9    35.4    22.8   0.00G 999.9  000000
712650 99999  19990314    32.4 24    17.6 24  1022.5 24  1013.8  4    9.0 24    9.8 24   15.0   22.0    35.4    23.5   0.00G 999.9  000000
712650 99999  19990315    34.1 24    15.5 24  1016.4 24  1007.3  4    9.0 24    7.5 24   15.9   21.0    42.8*   28.4*  0.00G 999.9  000000
712650 99999  19990316    37.0 24    22.3 24  1011.8 24  1003.3  4    9.0 24   11.1 24   26.0  999.9    42.8*   32.0*  0.00G 999.9  000000
712650 99999  19990317    39.8 24    33.0 24  1007.6 24   998.9  4    6.1 24    7.2 24   15.0   32.1    50.0*   34.9*  0.00G 999.9  000000
712650 99999  19990318    43.5 24    32.8 24  1004.3 24   994.1  4    8.9 24   17.5 24   34.0   41.0    52.5    33.6   0.00G 999.9  000000
712650 99999  19990319    35.3 24    22.5 24  1021.6 24  1010.9  4    9.0 24   12.8 24   18.1   41.0    47.5    29.3   0.02G 999.9  000000
712650 99999  19990320    32.9 24    23.4 24  1025.4 24  1016.4  4    9.0 24    6.8 24   13.0   22.0    43.3    27.0   0.00G 999.9  000000
712650 99999  19990321    38.0 24    31.7 24  1012.5 24  1004.6  4    8.9 24   11.0 24   18.1  999.9    42.8*   35.6* 99.99  999.9  010000
712650 99999  19990322    35.9 24    29.4 24  1005.4 23   996.1  4    7.8 24   15.6 24   27.0   34.0    43.9    32.9  99.99  999.9  011000
712650 99999  19990323    35.3 24    23.0 24  1015.7 24  1005.3  4    9.0 24   15.4 24   22.9   36.9    43.0    29.7   0.02G 999.9  000000
712650 99999  19990324    39.1 24    26.8 24  1014.2 24  1004.8  4    8.9 24    9.6 24   15.9   26.0    46.4*   34.9*  0.00G 999.9  000000
712650 99999  19990325    33.0 24    18.2 24  1022.4 22  9999.9  0    9.0 24    9.2 24   15.0   22.0    49.1    27.7   0.00G 999.9  001000
712650 99999  19990326    33.3 24    22.8 24  1026.8 24  1017.1  4    9.0 24    7.6 24   13.0   21.0    39.2*   26.6*  0.00G 999.9  001000
712650 99999  19990327    37.3 24    25.6 24  1026.6 24  1017.2  4    9.0 24    2.5 24    7.0  999.9    45.5    27.0   0.00I 999.9  000000
712650 99999  19990328    41.9 24    24.2 24  1021.7 24  1013.0  4    9.0 24    3.6 24    8.0  999.9    50.5    29.1   0.00I 999.9  000000
712650 99999  19990329    47.4 24    29.6 24  1016.6 24  1007.4  4    8.9 24   11.1 24   21.0   30.9    59.9    32.5   0.00I 999.9  000000
712650 99999  19990330    44.7 24    28.6 24  1022.6 24  1012.8  4    9.0 24   10.5 24   21.0   33.0    59.9    36.3   0.02G 999.9  000000
712650 99999  19990331    46.0 24    32.9 24  1018.4 24  1009.9  4    9.0 24    8.6 24   15.0   17.1    55.6    37.4   0.00G 999.9  000000
712650 99999  19990401    46.9 24    41.6 24  1013.6 23  9999.9  0    5.3 24    3.6 24   14.0  999.9    56.3    37.9   0.00I 999.9  000000
712650 99999  19990402    42.7 24    40.9 24  1017.8 22  1007.9  4    3.2 24    9.6 24   16.9   20.0    52.3    38.8   0.00I 999.9  000000
712650 99999  19990403    43.0 24    33.9 24  1016.1 24  1007.3  4    9.0 24    9.0 24   15.0   24.1    48.0    38.8   0.00G 999.9  000000
712650 99999  19990404    42.7 24    35.4 24  1016.1 23  1006.1  4    8.1 24   10.0 24   18.1  999.9    57.0    38.7  99.99  999.9  010000
712650 99999  19990405    38.8 23    23.5 23  1026.1 23  1016.1  4    8.9 23   13.3 23   19.0   27.0    57.0    33.8   0.06G 999.9  000000
712650 99999  19990406    42.6 24    35.7 24  1016.0 24  1009.1  4    8.9 24   14.5 24   21.0   27.0    50.0*   39.2*  0.00G 999.9  010000
712650 99999  19990407    48.8 24    36.6 24  1015.0 24  1004.7  4    9.0 24   16.1 24   25.1   36.9    57.0    38.8   0.12G 999.9  000000
712650 99999  19990408    51.4 23    37.9 23  1010.0 23  9999.9  0    8.4 23   10.9 23   22.9   34.0    64.4*   39.2*  0.00G 999.9  010000
712650 99999  19990409    44.9 24    31.2 24  1009.2 24  1000.5  4    9.0 24   14.4 24   20.0   33.0    64.6    39.0   0.06G 999.9  000000
712650 99999  19990410    42.3 24    23.7 24  1017.0 24  1006.3  4    9.0 24    8.6 24   14.0   27.0    48.7    36.3   0.00G 999.9  000000
712650 99999  19990411    38.0 24    30.0 24  1015.3 24  1007.0  4    8.1 24   20.3 24   35.0   42.0    41.2*   33.8*  0.00G 999.9  010000
712650 99999  19990412    43.0 23    25.8 23  1014.4 22  1003.8  4    8.9 23   12.4 23   19.0   42.9    52.2    32.9   0.12G 999.9  011000
712650 99999  19990413    45.6 24    18.7 24  1017.8 24  1008.8  4    9.0 24   12.2 24   16.9   25.1    56.5    36.0   0.02G 999.9  000000
712650 99999  19990414    50.2 23    22.0 23  1012.4 23  1003.5  4    8.9 23   10.5 23   16.9   28.9    64.4    36.0   0.00G 999.9  000000
712650 99999  19990415    47.2 24    28.2 24  1008.4 24  1000.0  4    9.0 24    6.5 24   12.0   29.9    65.7    38.1   0.00G 999.9  000000
712650 99999  19990416    43.0 24    32.4 24   998.6 24   990.0  4    8.6 24   15.3 24   25.1   30.9    51.1    38.8   0.00G 999.9  010000
712650 99999  19990417    43.2 24    38.8 24  1002.0 22  9999.9  0    8.5 24    6.9 24   12.0   33.0    51.8*   40.5*  0.14G 999.9  000000
712650 99999  19990418    45.0 24    36.6 24  1010.7 22  1000.7  4    9.0 24    7.1 24   15.9   19.0    51.8*   37.4*  0.00G 999.9  010000
712650 99999  19990419    44.2 24    39.3 24  1015.5 23  1006.1  4    8.4 24    6.4 24   13.0   22.0    51.6    37.9   0.02G 999.9  010000
712650 99999  19990420    45.7 23    32.9 23  1015.5 23  1006.0  4    8.9 23    6.8 22    9.9   15.0    55.0    40.1   0.00G 999.9  010000
712650 99999  19990421    45.8 24    35.8 24  1015.9 24  1006.9  4    8.7 24    5.0 24   11.1   17.1    55.0    38.1   0.02G 999.9  000000
712650 99999  19990422    46.9 24    39.8 24  1014.5 21  1004.6  4    7.4 24    7.8 24   14.0  999.9    55.8    38.1   0.00G 999.9  010000
712650 99999  19990423    45.1 24    35.6 24  1018.4 24  1008.6  4    7.1 24   10.7 24   18.1   22.0    50.0    42.4   0.75G 999.9  010000
712650 99999  19990424    43.6 24    17.3 24  1027.5 24  1017.2  4    9.0 24    9.6 24   15.0   27.0    53.6*   33.8*  0.06G 999.9  000000
712650 99999  19990425    45.4 24    24.3 24  1023.8 24  1015.5  4    9.0 24   11.1 24   21.0   26.0    62.6*   37.4*  0.02G 999.9  000000
712650 99999  19990426    53.5 24    32.5 24  1012.2 24  1003.3  4    9.0 24   11.6 24   18.1   26.0    65.1    36.7   0.00G 999.9  000000
712650 99999  19990427    46.4 24    28.6 24  1019.9 24  1009.4  4    9.0 24    8.9 24   13.0   29.9    65.1    41.0   0.00G 999.9  000000
712650 99999  19990428    48.8 24    33.4 24  1025.6 24  9999.9  0    9.0 24   12.3 24   21.0   26.0    51.8*   46.4*  0.00G 999.9  000000
712650 99999  19990429    51.7 24    30.7 24  1026.9 24  9999.9  0    9.0 24    7.3 24   12.0   18.1    55.9    45.5   0.00I 999.9  000000
712650 99999  19990430    54.4 24    28.9 24  1027.0 24  1017.7  4    9.0 24    5.5 24   11.1   18.1    59.4    46.6   0.00G 999.9  000000
712650 99999  19990501    55.9 24    30.6 24  1025.1 24  1016.0  4    9.0 24    4.2 24    8.0  999.9    62.6*   50.0*  0.00I 999.9  000000
712650 99999  19990502    57.4 24    37.8 24  1022.9 24  1013.8  4    9.0 24    3.6 24    8.0  999.9    68.7    45.7   0.00I 999.9  000000
712650 99999  19990503    59.9 24    39.3 24  1020.3 24  1011.3  4    9.0 24    6.9 24   11.1  999.9    68.9    45.7   0.00I 999.9  000000
712650 99999  19990504    59.5 24    40.8 24  1016.0 24  1007.0  4    9.0 24    4.2 24   11.1  999.9    67.6    50.0   0.00I 999.9  000000
712650 99999  19990505    59.7 23    49.5 23  1012.7 23  9999.9  0    9.0 23   11.4 23   19.0   25.1    67.6    50.0   0.00I 999.9  000000
712650 99999  19990506    59.0 24    52.7 24  1009.0 24  1000.4  4    8.2 24   13.0 24   20.0   26.0    65.5    48.2   0.00G 999.9  000000
712650 99999  19990507    57.8 24    53.0 24  1009.6 23  1000.0  4    7.7 24    6.9 24   12.0   27.0    64.2    48.2   0.00G 999.9  010000
712650 99999  19990508    55.2 24    53.4 24  1007.8 24   999.0  4    4.9 24    7.7 24   13.0   21.0    62.2    49.3  99.99  999.9  010000
712650 99999  19990509    55.0 24    47.3 24  1012.3 24  9999.9  0    7.5 24   10.3 24   19.0   22.0    62.1    49.3   0.20G 999.9  010000
712650 99999  19990510    49.6 24    36.8 24  1020.9 24  9999.9  0    9.0 24    6.3 24   13.0   24.1    61.3    40.5   0.06G 999.9  000000
712650 99999  19990511    52.2 24    37.1 24  1025.2 24  1015.6  4    9.0 24   10.0 24   18.1  999.9    57.9    40.5   0.00G 999.9  000000
712650 99999  19990512    51.5 24    32.4 24  1021.5 24  1012.9  4    9.0 24    5.6 24   15.0   26.0    57.6    42.4   0.00G 999.9  000000
712650 99999  19990513    50.1 22    29.9 22  1017.5 22  1008.1  4    8.9 22    9.6 22   16.9   20.0    57.6    42.4   0.00G 999.9  000000
712650 99999  19990514    54.0 22    36.7 22  1022.2 19  9999.9  0    9.0 22   13.6 22   20.0   25.1    57.2*   50.0*  0.00I 999.9  000000
712650 99999  19990515    57.2 24    41.9 24  1024.7 24  1015.3  4    9.0 24    7.2 24   11.1   26.0    60.8    49.5   0.00G 999.9  000000
712650 99999  19990516    57.9 24    45.2 24  1023.1 24  1014.0  4    9.0 24    8.6 24   12.0  999.9    62.8    48.6   0.00I 999.9  000000
712650 99999  19990517    57.7 24    48.1 24  1018.9 24  1010.2  4    9.0 24    8.9 24   14.0   17.1    64.6    48.6   0.00G 999.9  000000
712650 99999  19990518    61.1 24    53.6 24  1013.8 24  1005.2  4    8.8 24    7.6 24   14.0   19.0    66.2*   57.2*  0.00G 999.9  010000
712650 99999  19990519    58.8 24    51.6 24  1016.1 22  9999.9  0    7.2 24    8.9 24   14.0   22.0    68.0*   51.4*  0.26G 999.9  010000
712650 99999  19990520    55.2 24    44.1 24  1023.3 24  1013.8  4    9.0 24    6.2 24   11.1   25.1    68.2    45.1   0.18G 999.9  000000
712650 99999  19990521    57.5 24    50.0 24  1020.4 24  1011.6  4    8.2 24    4.0 24   11.1  999.9    67.1    45.1   0.00G 999.9  000000
712650 99999  19990522    60.2 24    50.6 24  1014.7 24  1005.9  4    8.9 24    5.5 24   12.0  999.9    67.1    50.0   0.00I 999.9  000000
712650 99999  19990523    62.4 22    52.7 22  1012.3 22  9999.9  0    8.9 22    5.6 22   11.1  999.9    69.8    55.8   0.00I 999.9  000000
712650 99999  19990524    57.2 24    53.4 24   999.1 22  9999.9  0    7.1 24    6.6 24   15.0   22.0    60.8*   53.6* 99.99  999.9  010000
712650 99999  19990525    50.4 22    44.3 22   999.2 22   989.7  4    8.8 22   18.1 22   22.9   29.9    55.0*   48.2*  0.41G 999.9  010000
712650 99999  19990526    51.7 24    46.7 24  1005.2 22   995.1  4    9.0 24   11.5 24   20.0   35.9    57.2*   47.7*  0.20G 999.9  010000
712650 99999  19990527    56.0 24    42.7 24  1013.1 24  1003.6  4    9.0 24    8.0 24   15.0   20.0    75.2*   42.8*  0.02G 999.9  000000
712650 99999  19990528    57.1 23    46.4 23  1014.9 23  1005.4  4    8.9 23    6.8 23   13.0   15.9    74.3    42.4   0.00G 999.9  000000
712650 99999  19990529    65.9 24    48.9 24  1018.7 23  1009.1  4    8.9 23    4.6 24    8.9  999.9    79.2    46.4   0.00I 999.9  000000
712650 99999  19990530    64.0 23    53.0 23  1021.1 23  1011.8  4    6.9 22    3.8 23   12.0  999.9    79.3    52.9   0.00I 999.9  000000
712650 99999  19990531    64.7 24    55.2 24  1017.1 24  1008.4  4    8.1 24    2.8 24    7.0  999.9    80.2    52.9  99.99  999.9  010000
712650 99999  19990601    61.3 24    59.3 24  1012.0 22  1003.2  4    2.9 24    5.5 24   12.0  999.9    73.8    57.6  99.99  999.9  010000
712650 99999  19990602    63.5 24    62.4 24  1010.4 15  9999.9  0    2.1 24    3.4 24    8.0  999.9    67.6    57.7  99.99  999.9  010000
712650 99999  19990603    61.4 24    55.6 24  1012.6 21  1001.5  4    7.4 24   11.3 24   16.9   21.0    67.8    57.9   0.00I 999.9  000000
712650 99999  19990604    59.4 23    46.2 23  1021.8 23  1012.1  4    8.9 23    6.4 23    9.9   25.1    65.5    52.5   0.02G 999.9  000000
712650 99999  19990605    58.6 24    45.9 24  1020.7 23  1011.6  4    9.0 24    8.0 24   13.0  999.9    66.6    52.5   0.00G 999.9  010000
712650 99999  19990606    63.2 24    56.3 24  1017.2 24  1008.4  4    7.2 24    6.2 24   15.0  999.9    72.0    52.5   0.00I 999.9  000000
712650 99999  19990607    66.6 24    62.2 24  1014.8 23  1005.9  4    4.4 24    4.7 24   22.9   41.0    77.7    52.5   0.00G 999.9  010000
712650 99999  19990608    72.4 24    59.7 24  1010.9 24  9999.9  0    8.8 24   11.1 24   25.1   41.0    84.2*   57.2*  0.33G 999.9  000000
712650 99999  19990609    68.0 23    58.2 23  1015.3 19  1006.0  4    8.9 23    7.0 23   16.9   31.1    84.2    57.0   0.02G 999.9  000000
712650 99999  19990610    65.0 23    59.5 23  1020.2 21  1010.6  4    8.8 23   13.9 23   18.1   21.0    72.1    60.8   0.00G 999.9  000000
712650 99999  19990611    65.9 24    61.1 24  1021.1 24  1011.8  4    6.3 24    9.9 24   15.9   24.1    71.6*   60.8*  0.00G 999.9  000000
712650 99999  19990612    68.4 24    62.2 24  1021.1 24  1012.0  4    6.6 24    7.6 24   13.0   18.1    73.6    60.3   0.00G 999.9  010000
712650 99999  19990613    69.0 24    60.3 24  1019.1 24  1010.4  4    9.0 24    5.8 24   11.1  999.9    74.5    60.6   0.00I 999.9  000000
712650 99999  19990614    68.4 24    62.2 24  1011.5 19  9999.9  0    7.9 24    9.6 24   22.9   27.0    73.6    61.2  99.99  999.9  010000
712650 99999  19990615    58.0 24    43.5 24  1018.2 24  1007.8  4    9.0 24   11.8 24   15.9   28.0    73.4    49.1   0.22G 999.9  000000
712650 99999  19990616    56.5 24    41.5 24  1022.4 24  1013.2  4    9.0 24    3.8 24    8.0   22.0    63.3    48.7   0.00G 999.9  000000
712650 99999  19990617    59.0 24    46.8 24  1020.8 24  1011.2  4    9.0 24    4.0 24    8.9  999.9    64.6    48.7   0.00I 999.9  000000
712650 99999  19990618    58.9 24    49.6 24  1024.7 24  1015.2  4    9.0 24    8.1 24   14.0   15.0    66.2*   50.0*  0.00I 999.9  000000
712650 99999  19990619    62.9 24    53.5 24  1027.8 24  1018.2  4    9.0 24    2.4 24    7.0  999.9    71.6*   53.6*  0.00I 999.9  000000
712650 99999  19990620    66.2 24    56.4 24  1026.6 24  1017.7  4    9.0 24    6.0 24    8.0  999.9    73.4*   62.6*  0.00I 999.9  000000
712650 99999  19990621    68.0 23    55.6 23  1025.0 23  1015.8  4    8.9 23    4.5 23    7.0  999.9    75.2*   60.8*  0.00I 999.9  000000
712650 99999  19990622    69.3 23    54.3 23  1022.8 23  9999.9  0    9.0 23    4.5 23    8.9  999.9    75.9    59.2   0.00I 999.9  000000
712650 99999  19990623    71.0 23    60.8 23  1018.1 23  1009.6  4    8.9 23    5.8 23    8.0  999.9    82.4*   66.2*  0.00I 999.9  000000
712650 99999  19990624    72.0 23    63.1 23  1013.3 22  1004.5  4    8.7 23    5.3 23    8.9  999.9    82.0    65.5  99.99  999.9  010000
712650 99999  19990625    72.3 24    67.7 24  1011.4 22  1002.2  4    6.9 24    6.6 24   15.9  999.9    78.3    66.9  99.99  999.9  010000
712650 99999  19990626    74.1 21    61.7 21  1013.6 21  9999.9  0    9.0 21    5.1 21    9.9  999.9    83.1*   66.2* 99.99  999.9  010000
712650 99999  19990627    73.3 24    65.5 24  1011.3 22  1002.7  4    8.3 24    6.4 24   13.0  999.9    84.2    66.4  99.99  999.9  010000
712650 99999  19990628    74.8 23    70.1 23  1005.2 22   996.7  4    5.4 23    5.7 23    9.9   17.1    80.6*   71.6*  0.06G 999.9  000000
712650 99999  19990629    72.4 23    63.5 23  1001.1 23   991.5  4    6.8 23   12.2 23   23.9   29.9    82.8    68.7   0.00G 999.9  010000
712650 99999  19990630    61.3 23    50.0 23  1014.4 23  1004.5  4    8.9 23    7.1 23   16.9   32.1    68.0*   51.8*  0.04G 999.9  000000
712650 99999  19990701    65.6 23    60.5 23  1012.3 23  1004.6  4    8.1 23    8.6 23   16.9  999.9    73.4*   60.8*  0.00G 999.9  010000
712650 99999  19990702    71.5 22    62.0 22  1010.5 22  1000.5  4    7.5 22   11.1 22   19.0   21.0    82.4*   62.6*  0.08G 999.9  010000
712650 99999  19990703    71.8 22    64.0 22  1017.3 22  1008.4  4    8.4 22    5.0 22   14.0   32.1    84.9    62.6   0.00G 999.9  010000
712650 99999  19990704    75.5 22    71.1 22  1014.8 20  9999.9  0    5.9 22    6.2 22   13.0   21.0    93.2*   69.8* 99.99  999.9  010000
712650 99999  19990705    80.2 24    71.9 24  1013.3 24  1004.6  4    8.7 24   10.0 24   15.9   20.0    93.4    67.8   0.00I 999.9  000000
712650 99999  19990706    77.9 22    68.1 22  1009.4 22  1000.4  4    6.7 22   12.8 22   21.0   33.0    92.7    66.6   0.00G 999.9  000000
712650 99999  19990707    71.8 23    53.8 23  1012.5 23  1003.4  4    9.0 21   12.5 23   21.0   28.9    89.8    53.4   0.00I 999.9  000000
712650 99999  19990708    67.4 19    51.2 19  1014.2 19  9999.9  0    9.0 19   10.8 19   16.9   21.0    82.8    53.4   0.00I 999.9  000000
712650 99999  19990709    63.4 24    55.3 24  1010.7 23  1002.7  4    9.0 24    8.8 24   20.0   25.1    75.9    55.8  99.99  999.9  010000
712650 99999  19990710    67.3 23    53.7 23  1009.5 23   998.7  4    8.9 23   12.6 23   15.9   26.0    77.2    55.8   0.00G 999.9  000000
712650 99999  19990711    63.2 22    49.5 22  1021.3 22  9999.9  0    9.0 22    7.7 22   14.0   20.0    74.3    55.2   0.00I 999.9  000000
712650 99999  19990712    62.8 23    54.6 23  1023.8 23  1014.5  4    8.9 23    5.0 23   11.1  999.9    73.9    54.0   0.00I 999.9  000000
712650 99999  19990713    67.1 22    55.8 22  1021.2 20  9999.9  0    9.0 22    6.3 22    9.9  999.9    75.2*   59.0*  0.00I 999.9  000000
712650 99999  19990714    69.0 23    61.4 23  1019.4 21  9999.9  0    8.6 23    6.1 23    8.9  999.9    75.2*   64.4*  0.00I 999.9  000000
712650 99999  19990715    69.9 20    60.6 20  1017.3 20  9999.9  0    6.6 20    5.7 20   15.0  999.9    78.8*   64.4*  0.00I 999.9  000000
712650 99999  19990716    72.2 23    64.0 23  1017.9 23  9999.9  0    2.5 15    6.4 23   15.9  999.9    80.6*   64.4*  0.00I 999.9  000000
712650 99999  19990717    72.6 22    65.4 22  1017.4 22  9999.9  0    4.0 21    6.9 22   22.9  999.9    84.2*   66.2* 99.99  999.9  010000
712650 99999  19990718    73.4 23    67.0 23  1018.4 23  9999.9  0    5.3 23    4.8 23   15.0   40.0    87.8*   66.2*  0.02G 999.9  000000
712650 99999  19990719    71.8 22    62.4 22  1016.7 21  9999.9  0    8.7 22    4.7 22    8.0  999.9    77.0*   68.0*  0.00G 999.9  010000
712650 99999  19990720    70.1 22    60.1 22  1019.2 22  9999.9  0    8.6 22    7.3 22   12.0   15.9    73.4*   64.4*  0.00I 999.9  000000
712650 99999  19990721    70.7 23    61.1 23  1019.3 23  9999.9  0    9.0 23    6.7 23   12.0  999.9    75.2*   66.2*  0.00I 999.9  000000
712650 99999  19990722    73.3 22    60.6 22  1015.1 22  9999.9  0    9.0 22    3.4 22    8.9  999.9    82.4*   66.2*  0.00I 999.9  000000
712650 99999  19990723    77.0 22    66.5 22  1013.8 22  9999.9  0    8.7 22    7.0 22   13.0  999.9    84.2*   69.8*  0.00I 999.9  000000
712650 99999  19990724    77.9 24    68.9 24  1008.2 24  9999.9  0    8.3 24    9.6 24   14.0  999.9    87.8*   71.6*  0.00I 999.9  000000
712650 99999  19990725    77.4 24    62.3 24  1007.7 23  9999.9  0    9.0 24    9.8 24   23.9   33.0    89.2    67.1  99.99  999.9  010000
712650 99999  19990726    77.2 24    64.2 24  1009.8 24  9999.9  0    9.0 24    9.6 24   16.9   22.0    86.0*   69.8*  0.00I 999.9  000000
712650 99999  19990727    80.6 24    60.3 24  1011.2 24  9999.9  0    9.0 24    9.6 24   21.0   28.0    91.4*   71.6*  0.00G 999.9  000000
712650 99999  19990728    73.2 21    54.5 21  1011.6 21  9999.9  0    9.0 21    5.8 21   14.0   20.0    82.4*   62.6*  0.00I 999.9  000000
712650 99999  19990729    75.7 21    66.0 21  1005.5 19  9999.9  0    9.0 21    6.7 21   11.1  999.9    87.6    68.9   0.00I 999.9  000000
712650 99999  19990730    78.0 24    68.5 24  1003.5 24   994.5  4    9.0 24    5.3 24   12.0  999.9    88.5    68.9   0.00I 999.9  000000
712650 99999  19990731    75.1 21    70.2 21  1005.6 18  9999.9  0    7.2 21    5.4 21   16.9   26.0    87.8    69.6  99.99  999.9  010000
712650 99999  19990801    75.4 24    61.8 24  1009.3 24  9999.9  0    8.6 24    7.9 24   14.0   15.9    82.4*   68.0*  0.00I 999.9  000000
712650 99999  19990802    69.9 24    56.6 24  1017.6 24  9999.9  0    9.0 24    7.1 24   13.0   16.9    78.8*   60.8*  0.00I 999.9  000000
712650 99999  19990803    69.7 23    54.5 23  1019.9 23  9999.9  0    9.0 23    8.1 23   15.0   19.0    79.2    60.8   0.00I 999.9  000000
712650 99999  19990804    66.9 23    63.0 23  1014.2 21  9999.9  0    7.7 23    5.4 23    8.9   21.0    79.2    60.8  99.99  999.9  010000
712650 99999  19990805    67.5 24    59.5 24  1010.2 24  9999.9  0    8.3 24    4.4 24   11.1  999.9    77.0*   60.8*  0.00I 999.9  000000
712650 99999  19990806    66.8 20    54.0 20  1011.6 20  9999.9  0    8.8 20    9.2 20   18.1   23.9    78.8*   59.0* 99.99  999.9  010000
712650 99999  19990807    64.4 23    55.7 23  1015.9 23  9999.9  0    9.0 23    5.5 23   11.1   15.9    71.6*   57.2*  0.00I 999.9  000000
712650 99999  19990808    68.9 23    62.3 23  1007.8 23  9999.9  0    6.6 23    9.5 23   22.0   29.9    78.8*   64.4* 99.99  999.9  010000
712650 99999  19990809    62.9 22    50.7 22  1013.9 22  9999.9  0    9.0 22    6.6 22   15.0   21.0    69.8*   55.4*  0.00I 999.9  000000
712650 99999  19990810    62.6 23    54.8 23  1010.4 23  9999.9  0    8.7 23    6.9 23   18.1  999.9    74.1    60.8  99.99  999.9  010000
712650 99999  19990811    66.0 23    63.5 23  1008.8 23  9999.9  0    4.0 23    3.8 23    9.9  999.9    71.6*   62.6*  0.00I 999.9  000000
712650 99999  19990812    68.8 21    62.3 21  1013.7 21  9999.9  0    7.6 21    4.3 21    9.9  999.9    78.8*   62.6*  0.00I 999.9  000000
712650 99999  19990813    70.8 21    67.4 21  1007.6 17  9999.9  0    4.9 21    6.0 21   11.1  999.9    75.2*   68.0* 99.99  999.9  010000
712650 99999  19990814    66.3 21    60.6 21  1008.7 21   998.7  4    8.5 21    9.2 21   16.9   22.0    75.9    61.7  99.99  999.9  010000
712650 99999  19990815    66.1 22    55.1 22  1019.5 22  1009.6  4    8.9 22    5.0 22   12.0  999.9    71.8*   60.8*  0.00I 999.9  000000
712650 99999  19990816    67.3 23    58.3 23  1022.2 22  1013.0  4    8.9 23    3.9 23    9.9  999.9    75.2*   62.6*  0.00I 999.9  000000
712650 99999  19990817    73.3 22    63.9 22  1013.4 21  1004.8  4    7.6 22    9.0 22   15.0   21.0    84.7*   66.2*  0.00I 999.9  000000
712650 99999  19990818    68.4 23    59.5 23  1014.0 23  1004.2  4    8.9 23    7.3 23   12.0   15.0    86.0    61.7   0.00I 999.9  000000
712650 99999  19990819    67.5 23    57.1 23  1016.5 23  1007.1  4    8.9 23    6.9 23   11.1  999.9    74.8    61.7   0.00I 999.9  000000
712650 99999  19990820    67.1 24    59.6 24  1016.8 24  1007.8  4    8.8 24    6.3 24   14.0   18.1    71.6    63.0  99.99  999.9  010000
712650 99999  19990821    67.8 23    62.3 23  1017.6 23  1008.2  4    8.9 23    9.0 23   14.0   18.1    72.1    63.0   0.24G 999.9  010000
712650 99999  19990822    68.3 23    60.0 23  1017.8 23  9999.9  0    8.9 23    4.9 23    8.9  999.9    74.1    61.2   0.00G 999.9  000000
712650 99999  19990823    70.8 23    61.3 23  1016.9 23  1007.9  4    8.6 23    3.8 23    7.0  999.9    79.0    61.2   0.00I 999.9  000000
712650 99999  19990824    72.5 23    66.3 23  1014.8 22  1005.9  4    8.4 23   10.3 23   15.0  999.9    79.9    62.6  99.99  999.9  010000
712650 99999  19990825    72.1 23    67.7 23  1014.4 22  1005.3  4    5.7 23    5.2 23   13.0   15.9    76.1    68.9  99.99  999.9  010000
712650 99999  19990826    72.4 23    67.7 23  1012.7 23  1003.8  4    8.6 23    9.3 23   15.0   18.1    75.7    68.9  99.99  999.9  010000
712650 99999  19990827    72.0 24    67.0 24  1011.6 24  1002.4  4    8.7 24    7.2 24   12.0  999.9    75.2*   68.0* 99.99  999.9  010000
712650 99999  19990828    76.7 24    66.3 24  1010.7 24  1001.9  4    6.7 24    8.5 24   16.9   25.1    89.6*   69.8* 99.99  999.9  010000
712650 99999  19990829    67.7 17    47.7 17  1015.5 15  1005.2  4    8.9 17   12.3 17   18.1   27.0    80.6*   62.6*  0.00I 999.9  000000
712650 99999  19990830    60.4 20    48.2 20  1026.3 15  9999.9  0    9.0 20    9.7 20   15.9   21.0    64.4*   53.6*  0.00I 999.9  000000
712650 99999  19990831    64.1 23    55.9 23  1026.3 23  1017.1  4    8.9 23    4.6 23    8.0  999.9    70.2    57.6   0.00I 999.9  000000
712650 99999  19990901    66.9 24    58.9 24  1023.0 22  1014.1  4    8.9 23    3.2 24    6.0  999.9    73.4    56.5  99.99  999.9  010000
712650 99999  19990902    69.7 24    59.6 24  1020.3 24  1011.4  4    8.5 24    2.9 24    8.0  999.9    82.4*   60.8*  0.00I 999.9  000000
712650 99999  19990903    72.2 23    60.2 23  1018.8 23  9999.9  0    8.5 23    3.2 23    8.0  999.9    82.4*   62.6*  0.00I 999.9  000000
712650 99999  19990904    73.4 23    62.7 23  1018.6 23  1009.5  4    7.9 23    3.4 23    9.9  999.9    81.9    62.1   0.00I 999.9  000000
712650 99999  19990905    74.8 23    67.1 23  1018.0 22  1009.2  4    8.0 23    8.7 23   14.0  999.9    80.2    66.4  99.99  999.9  010000
712650 99999  19990906    75.4 23    70.0 23  1012.5 21  9999.9  0    9.0 23    7.2 23   11.1  999.9    80.6*   71.6* 99.99  999.9  010000
712650 99999  19990907    72.0 24    68.1 24  1009.4 21  1000.4  4    7.4 24    7.1 24   14.0   15.9    77.0*   68.0* 99.99  999.9  010000
712650 99999  19990908    72.8 24    66.7 24  1009.0 23  1000.2  4    5.8 24    6.0 24   13.0   21.0    78.8*   68.0*  0.10G 999.9  010000
712650 99999  19990909    72.0 23    67.7 23  1007.3 21  9999.9  0    6.2 23    9.2 23   15.9  999.9    78.6    68.0   0.00G 999.9  010000
712650 99999  19990910    66.2 22    54.2 22  1008.6 20  9999.9  0    9.0 22    8.5 22   15.0   19.0    73.4*   60.8*  0.00I 999.9  000000
712650 99999  19990911    66.6 24    51.5 24  1012.2 24  1002.3  4    9.0 24   14.2 24   19.0   22.0    72.0    59.7   0.00G 999.9  010000
712650 99999  19990912    63.4 21    55.6 21  1019.0 21  1010.0  4    8.9 21    6.4 21   14.0   26.0    73.8    53.8   0.00G 999.9  000000
712650 99999  19990913    68.8 21    65.2 21  1015.0 15  9999.9  0    7.8 21   11.2 21   20.0  999.9    72.0    66.9  99.99  999.9  010000
712650 99999  19990914    66.1 23    59.1 23  1016.0 23  9999.9  0    7.0 23    9.0 23   15.9  999.9    72.0    57.9   0.00H 999.9  010000
712650 99999  19990915    61.6 24    52.0 24  1019.2 24  1009.7  4    9.0 24    5.4 24   13.0   35.0    71.8    53.4   0.00G 999.9  000000
712650 99999  19990916    63.4 24    51.7 24  1017.0 24  1008.2  4    9.0 24   10.9 24   19.0   21.0    71.1    53.4   0.00G 999.9  000000
712650 99999  19990917    61.2 22    46.6 22  1018.0 22  9999.9  0    9.0 22   11.3 22   15.9   24.1    69.8*   53.6*  0.00G 999.9  000000
712650 99999  19990918    60.8 24    50.2 24  1020.3 24  1011.3  4    9.0 24    5.8 24    9.9   15.0    70.5    51.4   0.00I 999.9  000000
712650 99999  19990919    62.9 22    57.8 22  1017.4 22  1008.3  4    8.6 22    5.0 22   11.1  999.9    69.8*   55.4*  0.00I 999.9  000000
712650 99999  19990920    64.0 23    58.7 23  1010.8 22  9999.9  0    8.2 23    7.9 23   14.0   21.0    70.2    55.0  99.99  999.9  010000
712650 99999  19990921    52.0 24    39.8 24  1017.7 24  1007.9  4    9.0 24    9.6 24   15.0   23.9    68.2    45.5   0.00I 999.9  000000
712650 99999  19990922    51.5 23    39.4 23  1015.8 23  1007.4  4    8.9 23    7.3 23   11.1  999.9    64.4*   41.0*  0.00I 999.9  000000
712650 99999  19990923    60.6 24    49.7 24  1008.0 23   999.3  4    9.0 24   10.5 24   21.0   23.9    69.8*   53.6* 99.99  999.9  010000
712650 99999  19990924    61.8 24    53.3 24  1007.1 24   997.5  4    8.9 24    7.7 24   15.9   27.0    69.6    53.4   0.00G 999.9  010000
712650 99999  19990925    56.6 24    46.5 24  1016.2 24  1006.1  4    9.0 24    6.2 24    8.9   19.0    67.6    51.3   0.08G 999.9  000000
712650 99999  19990926    59.3 24    52.2 24  1020.4 24  1011.2  4    9.0 24    9.7 24   20.0   21.0    64.4*   53.6*  0.00G 999.9  000000
712650 99999  19990927    63.9 20    60.6 20  1020.2 20  1010.8  4    6.1 20    7.8 20   14.0   24.1    70.2    55.6   0.00G 999.9  000000
712650 99999  19990928    66.7  4    64.6  4  1020.2  4  1011.0  4  999.9  0    6.8  4   11.1  999.9    70.9    60.3   0.00D 999.9  000000
712650 99999  19990930    57.2  4    51.0  4  1004.7  4   995.6  4    7.3  4   13.7  4   20.0   36.9    69.6    51.3   1.87G 999.9  000000
712650 99999  19991001    58.5  8    42.9  8  1012.8  8  1001.3  4    8.8  8   12.9  8   18.1   32.1    62.6    51.3   0.06G   3.1  000000
712650 99999  19991002    52.7  4    45.8  4  1017.7  4  1008.4  4    8.7  4    4.0  4    7.0   24.1    63.3    46.0   0.00G   3.5  000000
712650 99999  19991004    46.6  9    36.3  9  1024.1  9  1015.1  4    8.7  9    8.4  9   12.0   22.0    54.5    38.7   0.28G   2.8  000000
712650 99999  19991005    47.0 24    38.9 24  1022.1 24  9999.9  0    9.0 24    5.6 24   11.1   25.1    55.4*   40.6*  0.10G   0.4  000000
712650 99999  19991006    49.7 23    37.8 23  1018.7 21  1008.2  4    8.9 23   14.0 23   18.1   25.1    55.8    38.3   0.00G   0.4  010000
712650 99999  19991007    39.0 22    24.8 22  1030.5 22  1020.4  4    8.9 22    7.3 22    9.9  999.9    53.2    34.3   0.00D   1.2  000000
712650 99999  19991008    51.5 21    45.0 21  1022.3 19  1014.6  4    8.7 21    6.6 21   15.0  999.9    59.0*   42.4*  0.02D   2.0  010000
712650 99999  19991009    55.9 19    54.4 19  1018.5 13  9999.9  0    3.5 19    9.2 19   14.0  999.9    61.3    45.0   0.22C   2.0  010000
712650 99999  19991010    56.6 22    56.0 22  1016.1 15  1007.4  4    2.7 22    2.4 22    6.0  999.9    62.6*   51.8*  0.00D   2.0  000000
712650 99999  19991011    57.4 21    45.9 21  1019.3 21  1009.2  4    8.3 21    9.5 21   15.9   23.9    64.4*   49.5*  0.02D 999.9  010000
712650 99999  19991012    49.0 20    40.5 20  1025.9 18  1017.2  4    8.6 20    5.3 20   11.1  999.9    55.6*   42.8*  0.00D 999.9  000000
712650 99999  19991013    55.6 22    52.8 22  1009.6 22  9999.9  0    7.0 22    6.8 22   15.0  999.9    62.1    41.9   0.00H   0.4  010000
712650 99999  19991014    44.2 22    35.8 22  1015.6 20  1004.4  4    8.3 22   13.0 22   22.9   35.0    52.0*   37.4*  1.59D 999.9  010000
712650 99999  19991015    49.7 24    40.7 24  1020.9 24  1011.8  4    9.0 24    6.1 24   12.0   15.0    57.2*   44.6*  0.02D 999.9  000000
712650 99999  19991016    55.7 23    51.6 23  1015.1 23  1006.6  4    8.8 23    6.6 23   15.0  999.9    62.6    42.6   0.00D   2.0  000000
712650 99999  19991017    55.5 24    51.0 24  1014.3 22  1005.0  4    8.3 24    7.4 24   15.9   21.0    64.2    49.1   0.00D   2.4  010000
712650 99999  19991018    47.2 24    39.5 24  1017.8 23  9999.9  0    9.0 24    7.2 24   12.0   15.9    51.8*   42.8*  0.00C 999.9  000000
712650 99999  19991019    46.1 23    37.5 23  1024.8 23  1015.5  4    8.9 23    8.6 23   14.0   20.0    51.3    39.0   0.00G   2.0  000000
712650 99999  19991020    50.3 23    41.1 23  1020.2 22  1011.2  4    8.9 23    9.7 23   21.0   28.9    55.9    40.8   0.00G 999.9  000000
712650 99999  19991021    47.7 24    37.1 24  1018.0 24  1009.5  4    9.0 24    9.0 24   18.1  999.9    56.1    39.4   0.00D   2.0  000000
712650 99999  19991022    52.5 24    44.9 24   999.1 22   990.9  4    9.0 24   15.0 24   22.9   28.0    55.8    39.4   0.00G   2.0  010000
712650 99999  19991023    45.6 24    39.0 24   999.4 24   989.7  4    9.0 24   11.8 24   18.1   29.9    55.8    42.1   0.08G   2.0  000000
712650 99999  19991024    44.0 24    34.6 24  1012.1 24  1001.3  4    9.0 24   12.1 24   18.1   27.0    49.6    39.4   0.00G 999.9  000000
712650 99999  19991025    45.4 23    36.9 23  1017.6 23  1008.3  4    8.9 23   13.0 23   18.1   29.9    53.8    39.4   0.00G   2.4  000000
712650 99999  19991026    51.7 24    39.2 24  1011.6 23  1002.1  4    9.0 24   13.4 24   20.0   28.0    59.2    40.8   0.02G   3.1  000000
712650 99999  19991027    44.4 23    34.1 23  1023.9 23  1013.3  4    8.9 23    8.3 23   14.0   16.9    49.5*   41.0*  0.00G   2.0  000000
712650 99999  19991028    46.6 23    41.1 23  1024.4 23  1016.1  4    8.9 23    6.1 23   11.1  999.9    55.6    40.3   0.00D   0.4  000000
712650 99999  19991029    48.7 23    44.7 23  1024.5 23  1014.2  4    7.4 23    7.0 23   13.0  999.9    57.2    40.3   0.00D 999.9  000000
712650 99999  19991030    51.2 21    47.9 21  1024.6 21  1015.5  4    7.7 21    7.3 21   12.0  999.9    57.6*   48.7*  0.00D 999.9  000000
712650 99999  19991031    55.7 24    48.5 24  1020.6 21  1011.2  4    5.5 24    7.6 24   15.0   25.1    64.4*   48.2*  0.04D 999.9  010000
712650 99999  19991101    47.1 24    43.6 24  1020.6 24  1012.0  4    7.8 24    4.5 24   12.0  999.9    65.8    39.6   0.00D 999.9  000000
712650 99999  19991102    49.3 24    47.8 24  1005.6 22   998.0  4    4.3 24   11.0 24   25.1   35.9    55.8    39.6   0.31D 999.9  010000
712650 99999  19991103    37.1 24    34.0 24   994.6 20   985.3  4    7.5 24   22.1 24   30.9   42.9    40.1*   35.1*  1.46G 999.9  010000
712650 99999  19991104    40.7 24    31.3 24  1015.3 22  1004.0  4    9.0 24   18.0 24   26.0   33.0    46.4*   36.9*  0.06G 999.9  010000
712650 99999  19991105    47.0 23    41.7 23  1018.8 23  1010.1  4    8.9 23   12.2 23   25.1   30.9    55.4*   40.5*  0.00G 999.9  000000
712650 99999  19991106    46.3 22    34.5 22  1021.9 22  9999.9  0    9.0 22   11.4 22   16.9   31.1    56.3    39.7   0.00G 999.9  000000
712650 99999  19991107    39.3 23    27.0 23  1027.0 23  1017.1  4    8.9 23    7.7 23   15.0   19.0    49.6    33.4   0.00D 999.9  000000
712650 99999  19991108    40.3 22    30.1 22  1024.2 22  1015.6  4    8.9 22    5.3 22   11.1  999.9    46.4*   35.6*  0.00D 999.9  000000
712650 99999  19991109    47.0 21    43.1 21  1013.6 21  9999.9  0    7.6 21    6.8 21   16.9   22.0    55.4*   41.0*  0.00C 999.9  000000
712650 99999  19991110    50.6 24    47.7 24  1009.4 23   999.7  4    6.0 24    7.7 24   15.9   19.0    56.3    39.9   0.02D 999.9  010000
712650 99999  19991111    35.2 23    23.5 23  1027.1 23  1015.7  4    8.9 23   12.0 23   18.1   25.1    57.6    30.0   0.14G 999.9  000000
712650 99999  19991112    39.0 22    31.0 22  1024.7 22  1017.2  4    8.7 22    6.2 22   13.0  999.9    44.6*   34.5*  0.00G 999.9  000000
712650 99999  19991113    44.4 22    41.5 22  1017.8 22  1009.0  4    7.5 22    7.3 22   11.1  999.9    48.2*   42.8*  0.02D 999.9  000000
712650 99999  19991114    47.2 24    38.3 24  1005.8 23   996.5  4    5.9 24   13.9 24   22.9   35.0    55.4*   41.0*  0.00D 999.9  000000
712650 99999  19991115    38.7 23    26.2 23  1013.1 21  9999.9  0    8.8 23   12.6 23   19.0   38.1    42.8*   35.6*  0.00G 999.9  000000
712650 99999  19991116    30.4 22    18.3 22  1011.8 22  1001.9  4    8.9 22   13.1 22   22.9   29.9    39.6    26.1   0.00D 999.9  000000
712650 99999  19991117    31.0 22    20.2 22  1019.4 22  1009.5  4    8.9 22    8.7 22   15.0   15.9    39.2*   25.2*  0.00D 999.9  000000
712650 99999  19991118    39.5 22    29.2 22  1020.8 20  9999.9  0    9.0 22    8.5 22   12.0  999.9    44.6*   35.6*  0.00H 999.9  010000
712650 99999  19991119    46.4 21    38.2 21  1018.7 21  1010.2  4    8.2 21    6.0 21   14.0  999.9    51.8*   41.0*  0.00D 999.9  000000
712650 99999  19991120    46.6 24    42.6 24  1015.2 22  1005.4  4    6.2 24    7.0 24   13.0  999.9    53.2    40.8   0.18D 999.9  010000
712650 99999  19991121    44.1 21    43.3 21  1020.5 19  9999.9  0    4.4 21    4.4 21   12.0  999.9    50.9    42.3   0.00B 999.9  000000
712650 99999  19991122    47.7 24    46.4 24  1019.8 21  1010.3  4    3.8 24    7.1 24   13.0  999.9    51.1    43.0   0.02D 999.9  010000
712650 99999  19991123    49.4 24    48.3 24  1023.0 23  1013.5  4    2.4 24    5.8 24   16.9  999.9    55.2    43.2   0.02D 999.9  000000
712650 99999  19991124    48.5 24    44.4 24  1015.8 24  1006.1  4    6.3 24   13.8 24   23.9   29.9    55.2    43.2   0.00D 999.9  010000
712650 99999  19991125    42.3 24    35.0 24  1024.9 24  1015.1  4    9.0 24    6.7 24   19.0   32.1    52.7    38.1   0.00G 999.9  000000
712650 99999  19991126    42.2 24    41.3 24  1013.8 19  1005.4  4    6.1 24   12.2 24   21.0   27.0    45.1    38.1   0.00G 999.9  010000
712650 99999  19991127    43.6 23    39.5 23  1010.4 22  1000.2  4    7.8 23   11.1 23   16.9   21.0    48.2*   41.0*  0.26D 999.9  010000
712650 99999  19991128    41.1 22    30.1 22  1020.2 22  1009.3  4    8.9 22   14.4 22   21.0   28.9    48.7    37.2   0.00D 999.9  000000
712650 99999  19991129    35.4 23    26.6 23  1026.5 21  1016.4  4    8.9 23   11.2 23   16.9   23.9    38.7*   32.0*  0.00D 999.9  001000
712650 99999  19991130    27.9 23    15.6 23  1037.0 23  1026.5  4    8.9 23    7.5 23   12.0   27.0    32.0*   23.0*  0.00G 999.9  000000
712650 99999  19991201    28.0 23    16.0 23  1036.0 23  1026.9  4    8.9 23    5.2 23    8.0  999.9    33.8*   23.0*  0.00D 999.9  001000
712650 99999  19991202    37.9 21    30.9 21  1025.1 21  1016.1  4    8.9 21   10.2 21   16.9  999.9    41.2    22.8   0.00D 999.9  000000
712650 99999  19991203    43.8 23    39.5 23  1017.6 19  1008.4  4    8.9 23    4.6 23    8.9  999.9    48.9    40.5   0.02D 999.9  010000
712650 99999  19991204    46.2 24    44.7 24  1016.4 21  9999.9  0    6.3 24    7.1 24   15.9  999.9    55.4*   42.8*  0.02B 999.9  010000
712650 99999  19991205    47.0 24    46.7 24  1013.0 21  1005.5  4    3.4 24    5.4 24   13.0  999.9    49.6    44.4   0.08C 999.9  010000
712650 99999  19991206    42.9 24    40.1 24  1010.4 17  9999.9  0    7.7 24    8.1 24   12.0   18.1    48.2*   37.4*  0.36C 999.9  010000
712650 99999  19991207    36.3 24    27.5 24  1022.9 24  1012.6  4    8.9 23    9.2 24   15.9   18.1    37.4*   33.8*  0.00D 999.9  000000
712650 99999  19991208    37.4 23    32.7 23  1026.1 23  1016.5  4    7.9 22    5.1 23   14.0  999.9    43.2    30.7   0.00D 999.9  000000
712650 99999  19991209    37.9 24    36.9 24  1025.2 20  1016.4  4    2.9 24    3.0 24    8.9  999.9    43.5    31.8   0.00D 999.9  000000
712650 99999  19991210    42.7 23    38.7 23  1010.8 21  1002.3  4    5.9 23    9.0 23   22.9   35.9    45.5    31.8   0.14D 999.9  010000
712650 99999  19991211    34.2 21    22.8 21  1020.9 21  1010.4  4    8.9 21   15.5 21   23.9   35.9    47.1    31.6   0.00D 999.9  000000
712650 99999  19991212    31.6 22    25.5 22  1022.2 20  1012.7  4    8.9 22    5.8 22   15.0  999.9    37.4*   27.3*  0.00G 999.9  000000
712650 99999  19991213    36.5 24    32.2 24  1018.2 23  9999.9  0    7.5 24    6.2 24   15.0   15.9    41.0*   33.8*  0.02C 999.9  010000
712650 99999  19991214    35.9 22    30.6 22  1018.9 22  1010.1  4    8.5 22   22.7 22   35.0   40.0    41.2    34.0   0.02D 999.9  010000
712650 99999  19991215    40.8 23    38.2 23  1010.8 18  1002.5  4    7.1 23    9.7 23   26.0   32.1    46.4    34.0   0.18D 999.9  010000
712650 99999  19991216    39.8 24    32.1 24  1006.2 22  9999.9  0    8.9 24   20.9 24   28.9   36.9    46.6    37.2   0.06D 999.9  011000
712650 99999  19991217    31.6 23    16.3 23  1016.6 23  1006.0  4    8.9 23   16.9 23   22.9   35.9    41.4    26.6   0.00G 999.9  000000
712650 99999  19991218    27.5 22    17.5 22  1026.6 22  9999.9  0    9.0 22    5.3 22   12.0  999.9    33.6    23.5   0.00C 999.9  000000
712650 99999  19991219    31.0 23    21.3 23  1028.5 23  1019.0  4    8.9 23   10.4 23   22.0   27.0    35.6*   25.0*  0.00D 999.9  000000
712650 99999  19991220    40.4 23    35.3 23  1016.4 22  1008.5  4    8.6 23   10.9 23   23.9   30.9    44.6*   36.7*  0.02D 999.9  010000
712650 99999  19991221    28.2 18    14.5 18  1019.3 17  9999.9  0    9.0 18   19.2 18   29.9   36.9    45.1    24.3   0.02C 999.9  000000
712650 99999  19991222    21.5 23    10.1 23  1024.4 23  9999.9  0    9.0 23   14.9 23   20.0   31.1    28.2    18.7   0.00G 999.9  000000
712650 99999  19991223    22.1 22    10.3 22  1022.4 21  9999.9  0    8.6 22   15.8 22   22.0   26.0    25.0    18.7   0.00G 999.9  001000
712650 99999  19991224    18.5 24     7.7 24  1023.5 24  1013.4  4    9.0 24    8.4 24   15.9   27.0    24.8*   14.0*  0.00G 999.9  000000
712650 99999  19991225    20.8 24    11.4 24  1025.5 24  1016.7  4    9.0 24   11.2 24   30.9   36.9    30.2*   15.8*  0.00G 999.9  001000
712650 99999  19991226    31.1 23    19.8 23  1002.6 20  9999.9  0    8.6 23   26.2 23   36.9   43.9    33.8*   26.6*  0.00G 999.9  001000
712650 99999  19991227    21.9 22    12.4 22  1008.7 22   998.3  4    8.3 22    9.6 22   16.9  999.9    26.6*   17.6*  0.00D 999.9  001000
712650 99999  19991228    18.9 24    11.0 24  1005.5 24   997.5  4    8.3 24   13.9 24   26.0   30.9    27.0*   10.4*  0.00G 999.9  001000
712650 99999  19991229    23.1 24    14.9 24  1004.7 23   994.4  4    8.1 24   13.1 24   29.9   35.9    32.0*   14.0*  0.00G 999.9  001000
712650 99999  19991230    34.5 24    24.7 24  1003.9 24   992.8  4    8.7 24   17.9 24   35.0   45.1    42.3    12.6   0.00G 999.9  001000
712650 99999  19991231    22.9 24    13.1 24  1018.2 24  1008.2  4    9.0 24    6.2 24   11.1   29.9    42.3    16.3   0.02G 999.9  000000
'''


@pytest.mark.slow
def test_get_station_year_text():
    downloaded_data = get_station_year_text(712650, 99999, 1999)
    try:
        downloaded_data = downloaded_data.decode('utf-8')
    except:
        pass
    assert downloaded_data == sample_data_random_station_1999

    with pytest.raises(Exception):
        get_station_year_text(712650, 99999, 19999999999)


@pytest.mark.slow
@pytest.mark.online
@pytest.mark.skipif(not has_geopy, reason='geopy is required')
def test_geocode():
    latlon = geocode('Fredericton, NB')
    assert_close(latlon, (45.966425, -66.645813), rtol=5e-4)