#!/usr/bin/env python
# ****************************************************************************
# * Software: FPDF for python                                                *
# * License:  LGPL v3.0+                                                     *
# *                                                                          *
# * Original Author (PHP):  Olivier PLATHEY 2004-12-31                       *
# * Ported to Python 2.4 by Max (maxpat78@yahoo.it) on 2006-05               *
# * Maintainer:  Mariano Reingart (reingart@gmail.com) et al since 2008 est. *
# * Maintainer:  David Alexander (daveankin@gmail.com) et al since 2017 est. *
# ****************************************************************************
"""fpdf module (in fpdf package housing FPDF class)

This module contains FPDF class inspiring this library.
The version number is updated here (above and below in variable).
"""

import errno
import hashlib
import io
import logging
import math
import os
import pickle
import re
import sys
import warnings
import zlib
from collections import defaultdict
from collections.abc import Sequence
from contextlib import contextmanager
from datetime import datetime
from enum import IntEnum
from functools import wraps
from pathlib import Path
from typing import Callable, NamedTuple, Optional, Union

from PIL import Image

from .errors import FPDFException, FPDFPageFormatException
from .fonts import fpdf_charwidths
from .image_parsing import get_img_info, load_resource
from .outline import serialize_outline, OutlineSection
from .recorder import FPDFRecorder
from .structure_tree import MarkedContent, StructureTreeBuilder
from .ttfonts import TTFontFile
from .util import (
    enclose_in_parens,
    escape_parens,
    substr,
)
from .deprecation import WarnOnDeprecatedModuleAttributes
from .syntax import (
    create_dictionary_string as pdf_d,
    create_list_string as pdf_l,
    create_stream as pdf_stream,
    iobj_ref as pdf_ref,
    InternalLink,
)

LOGGER = logging.getLogger(__name__)
HERE = Path(__file__).resolve().parent

# Global variables
FPDF_VERSION = "2.3.4"
FPDF_FONT_DIR = HERE / "font"
SYSTEM_TTFONTS = None

PAGE_FORMATS = {
    "a3": (841.89, 1190.55),
    "a4": (595.28, 841.89),
    "a5": (420.94, 595.28),
    "letter": (612, 792),
    "legal": (612, 1008),
}
LAYOUT_NAMES = {
    "single": "/SinglePage",
    "continuous": "/OneColumn",
    "two": "/TwoColumnLeft",
}
ZOOM_CONFIGS = {  # cf. section 8.2.1 "Destinations" of the 2006 PDF spec 1.7:
    "default": ("/Fit",),  # TODO FIXME
    "fullpage": ("/Fit",),
    "fullwidth": ("/FitH", "null"),
    "real": ("/XYZ", "null", "null", "1"),
}


class DocumentState(IntEnum):
    UNINITIALIZED = 0
    READY = 1  # page not started yet
    GENERATING_PAGE = 2
    CLOSED = 3  # EOF printed


class Annotation(NamedTuple):
    type: str
    x: int
    y: int
    width: int
    height: int
    contents: str = None
    link: Union[str, int] = None
    alt_text: Optional[str] = None


class TitleStyle(NamedTuple):
    font_family: Optional[str] = None
    font_style: Optional[str] = None
    font_size_pt: Optional[int] = None
    color: Union[int, tuple] = None  # grey scale or (red, green, blue)
    underline: bool = False
    t_margin: Optional[int] = None
    l_margin: Optional[int] = None
    b_margin: Optional[int] = None


class ToCPlaceholder(NamedTuple):
    render_function: Callable
    start_page: int
    y: int
    pages: int = 1


# Disabling this check due to the "format" parameter below:
# pylint: disable=redefined-builtin
def get_page_format(format, k=None):
    """Return page width and height size in points.

    Throws FPDFPageFormatException

    `format` can be either a 2-tuple or one of 'a3', 'a4', 'a5', 'letter', or
    'legal'.

    If format is a tuple, then the return value is the tuple's values
    given in the units specified on this document in the constructor,
    multiplied by the corresponding scale factor `k`, taken from instance
    variable `self.k`.

    If format is a string, the (width, height) tuple returned is in points.
    For a width and height of 8.5 * 11, 72 dpi is assumed, so the value
    returned is (8.5 * 72, 11 * 72), or (612, 792). Additional formats can be
    added by adding fields to the `fpdf.fpdf.PAGE_FORMATS` dictionary with a
    case insensitive key (the name of the new format) and 2-tuple value of
    (width, height) in dots per inch with a 72 dpi resolution.
    """
    if isinstance(format, str):
        format = format.lower()
        if format in PAGE_FORMATS:
            return PAGE_FORMATS[format]
        raise FPDFPageFormatException(format, unknown=True)

    if k is None:
        raise FPDFPageFormatException(format, one=True)

    try:
        return format[0] * k, format[1] * k
    except Exception as e:
        args = f"{format}, {k}"
        raise FPDFPageFormatException(f"Arguments must be numbers: {args}") from e


def load_cache(filename: Path):
    """Return unpickled object, or None if cache unavailable"""
    if not filename:
        return None
    try:
        return pickle.loads(filename.read_bytes())
    # File missing, unsupported pickle, etc
    except (OSError, ValueError):
        return None


def check_page(fn):
    """Decorator to protect drawing methods"""

    @wraps(fn)
    def wrapper(self, *args, **kwargs):
        if not self.page and not kwargs.get("split_only"):
            raise FPDFException("No page open, you need to call add_page() first")
        return fn(self, *args, **kwargs)

    return wrapper


class FPDF:
    """PDF Generation class"""

    def __init__(
        self, orientation="portrait", unit="mm", format="A4", font_cache_dir=True
    ):
        """
        Args:
            orientation (str): "portrait" ("P") or "landscape" ("L").
                Default to "portrait".
            unit (str): "pt", "mm", "cm" or "in". Default to "mm".
            format (str): "a3", "a4", "a5", "letter" or "legal".
                Default to "a4".
            font_cache_dir (Path or str): directory where pickle files
                for TTF font files are kept.
                The default is `True`, meaning the current folder.
        """
        # Initialization of properties
        self.offsets = {}  # array of object offsets
        self.page = 0  # current page number
        self.n = 2  # current object number
        self.buffer = bytearray()  # buffer holding in-memory PDF
        # Associative array from page number to dicts containing pages and metadata:
        self.pages = {}
        self.state = DocumentState.UNINITIALIZED  # current document state
        self.fonts = {}  # array of used fonts
        self.font_files = {}  # array of font files
        self.diffs = {}  # array of encoding differences
        self.images = {}  # array of used images
        self.annots = defaultdict(list)  # map page numbers to arrays of Annotations
        self.links = {}  # array of InternalLink
        self.in_footer = 0  # flag set when processing footer
        self.lasth = 0  # height of last cell printed
        self.current_font = {}  # current font
        self.font_family = ""  # current font family
        self.font_style = ""  # current font style
        self.font_size_pt = 12  # current font size in points
        self.font_stretching = 100  # current font stretching
        self.str_alias_nb_pages = "{nb}"
        self.underline = 0  # underlining flag
        self.draw_color = "0 G"
        self.fill_color = "0 g"
        self.text_color = "0 g"
        self.ws = 0  # word spacing
        self.angle = 0  # used by deprecated method: rotate()
        self.font_cache_dir = font_cache_dir
        self.xmp_metadata = None
        # Only set if XMP metadata is added to the document:
        self._xmp_metadata_obj_id = None
        self.struct_builder = StructureTreeBuilder()
        self._struct_parents_id_per_page = {}  # {page_object_id -> StructParent(s) ID}
        # Only set if a Structure Tree is added to the document:
        self._struct_tree_root_obj_id = None
        self._outlines_obj_id = None
        self._toc_placeholder = None  # ToCPlaceholder
        self._outline = []  # list of OutlineSection
        self.section_title_styles = {}  # level -> TitleStyle

        # Standard fonts
        self.core_fonts = {
            "courier": "Courier",
            "courierB": "Courier-Bold",
            "courierI": "Courier-Oblique",
            "courierBI": "Courier-BoldOblique",
            "helvetica": "Helvetica",
            "helveticaB": "Helvetica-Bold",
            "helveticaI": "Helvetica-Oblique",
            "helveticaBI": "Helvetica-BoldOblique",
            "times": "Times-Roman",
            "timesB": "Times-Bold",
            "timesI": "Times-Italic",
            "timesBI": "Times-BoldItalic",
            "symbol": "Symbol",
            "zapfdingbats": "ZapfDingbats",
        }
        self.core_fonts_encoding = "latin-1"
        # Replace these fonts with these core fonts
        self.font_aliases = {
            "arial": "helvetica",
            "couriernew": "courier",
            "timesnewroman": "times",
        }
        # Scale factor
        if unit == "pt":
            self.k = 1
        elif unit == "mm":
            self.k = 72 / 25.4
        elif unit == "cm":
            self.k = 72 / 2.54
        elif unit == "in":
            self.k = 72.0
        else:
            raise FPDFException(f"Incorrect unit: {unit}")

        # Page format
        self.dw_pt, self.dh_pt = get_page_format(format, self.k)

        # Page orientation
        orientation = orientation.lower()
        if orientation in ("p", "portrait"):
            self.def_orientation = "P"
            self.w_pt = self.dw_pt
            self.h_pt = self.dh_pt
        elif orientation in ("l", "landscape"):
            self.def_orientation = "L"
            self.w_pt = self.dh_pt
            self.h_pt = self.dw_pt
        else:
            raise FPDFException(f"Incorrect orientation: {orientation}")
        self.cur_orientation = self.def_orientation
        self.w = self.w_pt / self.k
        self.h = self.h_pt / self.k
        self.font_size = self.font_size_pt / self.k

        # Page spacing
        # Page margins (1 cm)
        margin = (7200 / 254) / self.k
        self.x, self.y, self.l_margin, self.t_margin = 0, 0, 0, 0
        self.set_margins(margin, margin)
        self.x, self.y = self.l_margin, self.t_margin
        self.c_margin = margin / 10.0  # Interior cell margin (1 mm)
        self.line_width = 0.567 / self.k  # line width (0.2 mm)
        # sets self.auto_page_break, self.b_margin & self.page_break_trigger:
        self.set_auto_page_break(True, 2 * margin)
        self.set_display_mode("fullwidth")  # Full width display mode
        self.compress = True  # Enable compression by default
        self.pdf_version = "1.3"  # Set default PDF version No.

    @property
    def unifontsubset(self):
        return self.current_font.get("type") == "TTF"

    @property
    def epw(self):
        """
        Effective page width: the page width minus its horizontal margins.
        """
        return self.w - self.l_margin - self.r_margin

    @property
    def eph(self):
        """
        Effective page height: the page height minus its vertical margins.
        """
        return self.h - self.t_margin - self.b_margin

    def set_margin(self, margin):
        """
        Sets the document right, left, top & bottom margins to the same value.

        Args:
            margin (int): margin in the unit specified to FPDF constructor
        """
        self.set_margins(margin, margin)
        self.set_auto_page_break(self.auto_page_break, margin)

    def set_margins(self, left, top, right=-1):
        """
        Sets the document left, top & optionaly right margins to the same value.
        By default, they equal 1 cm.
        Also sets the current FPDF.y on the page to this minimum vertical position.

        Args:
            left (int): left margin in the unit specified to FPDF constructor
            top (int): top margin in the unit specified to FPDF constructor
            right (int): optional right margin in the unit specified to FPDF constructor
        """
        self.set_left_margin(left)
        if self.y < top or self.y == self.t_margin:
            self.y = top
        self.t_margin = top
        if right == -1:
            right = left
        self.r_margin = right

    def set_left_margin(self, margin):
        """
        Sets the document left margin.
        Also sets the current FPDF.x on the page to this minimum horizontal position.

        Args:
            margin (int): margin in the unit specified to FPDF constructor
        """
        if self.x < margin or self.x == self.l_margin:
            self.x = margin
        self.l_margin = margin

    def set_top_margin(self, margin):
        """
        Sets the document top margin.

        Args:
            margin (int): margin in the unit specified to FPDF constructor
        """
        self.t_margin = margin

    def set_right_margin(self, margin):
        """
        Sets the document right margin.

        Args:
            margin (int): margin in the unit specified to FPDF constructor
        """
        self.r_margin = margin

    def set_auto_page_break(self, auto, margin=0):
        """
        Set auto page break mode and triggering bottom margin.
        By default, the mode is on and the bottom margin is 2 cm.

        Args:
            auto (bool): enable or disable this mode
            margin (int): optional bottom margin (distance from the bottom of the page)
                in the unit specified to FPDF constructor
        """
        self.auto_page_break = auto
        self.b_margin = margin
        self.page_break_trigger = self.h - margin

    def set_display_mode(self, zoom, layout="continuous"):
        """
        Set display mode in viewer

        Args:
            zoom: either 'fullpage', 'fullwidth', 'real', 'default',
                or a number, interpreted as a percentage.
            layout (str): either "single", "continuous" or "two".
        """
        if zoom in ZOOM_CONFIGS or not isinstance(zoom, str):
            self.zoom_mode = zoom
        else:
            raise FPDFException(f"Incorrect zoom display mode: {zoom}")

        if layout in LAYOUT_NAMES:
            self.layout_mode = layout
        else:
            raise FPDFException(f"Incorrect layout display mode: {layout}")

    def set_compression(self, compress):
        """Set page compression"""
        self.compress = compress

    def set_title(self, title):
        """Title of document"""
        self.title = title

    def set_lang(self, lang):
        """
        A language identifier specifying the natural language for all text in the document
        except where overridden by language specifications for structure elements or marked content.
        A language identifier can either be the empty text string, to indicate that the language is unknown,
        or a Language-Tag as defined in RFC 3066, "Tags for the Identification of Languages".
        """
        self.lang = lang

    def set_subject(self, subject):
        """Subject of document"""
        self.subject = subject

    def set_author(self, author):
        """Author of document"""
        self.author = author

    def set_keywords(self, keywords):
        """Keywords of document"""
        self.keywords = keywords

    def set_creator(self, creator):
        """Creator of document"""
        self.creator = creator

    def set_producer(self, producer):
        """Producer of document"""
        self.producer = producer

    def set_creation_date(self, date=None):
        """Sets Creation of Date time, or current time if None given."""
        self.creation_date = datetime.now() if date is None else date

    def set_xmp_metadata(self, xmp_metadata):
        if "<?xpacket" in xmp_metadata[:50]:
            raise ValueError(
                "fpdf2 already performs XMP metadata wrapping in a <?xpacket> tag"
            )
        self.xmp_metadata = xmp_metadata

    def set_doc_option(self, opt, value):
        """Set document option"""
        if opt == "core_fonts_encoding":
            self.core_fonts_encoding = value
        else:
            raise FPDFException(f'Unknown document option "{opt}"')

    def alias_nb_pages(self, alias):
        """Define an alias for total number of pages"""
        self.str_alias_nb_pages = alias

    def open(self):
        """Begin document"""
        self.state = DocumentState.READY

    def close(self):
        """Terminate document"""
        if self.state == DocumentState.CLOSED:
            return
        if self.page == 0:
            self.add_page()

        # Page footer
        self.in_footer = 1
        self.footer()
        self.in_footer = 0

        self._endpage()  # close page
        self._enddoc()  # close document

    def add_page(self, orientation="", format="", same=False):
        """Start a new page, if same page format will be same as previous"""
        if self.state == DocumentState.CLOSED:
            raise FPDFException(
                "A page cannot be added on a closed document, after calling output()"
            )
        if self.state == DocumentState.UNINITIALIZED:
            self.open()
        family = self.font_family
        style = f"{self.font_style}U" if self.underline else self.font_style
        size = self.font_size_pt
        lw = self.line_width
        dc = self.draw_color
        fc = self.fill_color
        tc = self.text_color
        stretching = self.font_stretching
        if self.page > 0:
            # Page footer
            self.in_footer = 1
            self.footer()
            self.in_footer = 0
            # close page
            self._endpage()

        # Start new page
        self._beginpage(orientation, format, same)
        self._out("2 J")  # Set line cap style to square
        self.line_width = lw  # Set line width
        self._out(f"{lw * self.k:.2f} w")

        # Set font
        if family:
            self.set_font(family, style, size)

        # Set colors
        self.draw_color = dc
        if dc != "0 G":
            self._out(dc)
        self.fill_color = fc
        if fc != "0 g":
            self._out(fc)
        self.text_color = tc

        # BEGIN Page header
        self.header()

        if self.line_width != lw:  # Restore line width
            self.line_width = lw
            self._out(f"{lw * self.k:.2f} w")

        if family:
            self.set_font(family, style, size)  # Restore font

        if self.draw_color != dc:  # Restore colors
            self.draw_color = dc
            self._out(dc)
        if self.fill_color != fc:
            self.fill_color = fc
            self._out(fc)
        self.text_color = tc

        if stretching != 100:  # Restore stretching
            self.set_stretching(stretching)
        # END Page header

    def header(self):
        """Header to be implemented in your own inherited class"""

    def footer(self):
        """Footer to be implemented in your own inherited class"""

    def page_no(self):
        """Get current page number"""
        return self.page

    def set_draw_color(self, r, g=-1, b=-1):
        """Set color for all stroking operations"""
        if (r == 0 and g == 0 and b == 0) or g == -1:
            self.draw_color = f"{r / 255:.3f} G"
        else:
            self.draw_color = f"{r / 255:.3f} {g / 255:.3f} {b / 255:.3f} RG"
        if self.page > 0:
            self._out(self.draw_color)

    def set_fill_color(self, r, g=-1, b=-1):
        """Set color for all filling operations"""
        if (r == 0 and g == 0 and b == 0) or g == -1:
            self.fill_color = f"{r / 255:.3f} g"
        else:
            self.fill_color = f"{r / 255:.3f} {g / 255:.3f} {b / 255:.3f} rg"
        if self.page > 0:
            self._out(self.fill_color)

    def set_text_color(self, r, g=-1, b=-1):
        """Set color for text"""
        if (r == 0 and g == 0 and b == 0) or g == -1:
            self.text_color = f"{r / 255:.3f} g"
        else:
            self.text_color = f"{r / 255:.3f} {g / 255:.3f} {b / 255:.3f} rg"

    def get_string_width(self, s, normalized=False):
        """Get width of a string in the current font"""
        # normalized is parameter for internal use
        s = s if normalized else self.normalize_text(s)
        cw = self.current_font["cw"]
        w = 0
        if self.unifontsubset:
            for char in s:
                char = ord(char)
                if len(cw) > char:
                    w += cw[char]
                elif self.current_font["desc"]["MissingWidth"]:
                    w += self.current_font["desc"]["MissingWidth"]
                else:
                    w += 500
        else:
            w += sum(cw.get(char, 0) for char in s)
        if self.font_stretching != 100:
            w = w * self.font_stretching / 100
        return w * self.font_size / 1000

    def set_line_width(self, width):
        """Set line width"""
        self.line_width = width
        if self.page > 0:
            self._out(f"{width * self.k:.2f} w")

    @check_page
    def line(self, x1, y1, x2, y2):
        """
        Draw a line between two points.

        Args:
            x1 (int): Abscissa of first point
            y1 (int): Ordinate of first point
            x2 (int): Abscissa of second point
            y2 (int): Ordinate of second point
        """
        self._out(
            f"{x1 * self.k:.2f} {(self.h - y1) * self.k:.2f} m {x2 * self.k:.2f} "
            f"{(self.h - y2) * self.k:.2f} l S"
        )

    @check_page
    def polyline(self, point_list, fill=False, polygon=False):
        """
        Draws lines between two or more points.

        Args:
            point_list (list of tuples): List of Abscissa and Ordinate of
                                        segments that should be drawn
            fill (bool): If true then polyline should be filled
            polygon (bool): If true, close path before stroking
        """
        operator = "m"
        for point in point_list:
            self._out(
                f"{point[0] * self.k:.2f} {(self.h - point[1]) * self.k:.2f} {operator}"
            )
            operator = "l"
        if polygon:
            self._out(" h ")
        if fill:
            self._out(" B ")
        else:
            self._out(" S ")

    @check_page
    def polygon(self, point_list, fill=False):
        """
        Outputs a polygon defined by three or more points.

        Args:
            point_list (list of tuples): List of Abscissa and Ordinate of
                                        polygon that should be drawn
            fill (bool): If true polygon will be filled
        """
        self.polyline(point_list, fill=fill, polygon=True)

    def _set_dash(self, dash_length=None, space_length=None):
        dash = ""
        if dash_length and space_length:
            dash = f"{dash_length * self.k:.3f} {space_length * self.k:.3f}"
        self._out(f"[{dash}] 0 d")

    @check_page
    def dashed_line(self, x1, y1, x2, y2, dash_length=1, space_length=1):
        """
        Draw a dashed line between two points.

        Args:
            x1 (int): Abscissa of first point
            y1 (int): Ordinate of first point
            x2 (int): Abscissa of second point
            y2 (int): Ordinate of second point
            dash_length (int): Length of the dash
            space_length (int): Length of the space between 2 dashes
        """
        self._set_dash(dash_length, space_length)
        self.line(x1, y1, x2, y2)
        self._set_dash()

    @check_page
    def rect(self, x, y, w, h, style=None):
        """
        Outputs a rectangle.
        It can be drawn (border only), filled (with no border) or both.

        Args:
            x (int): Abscissa of upper-left bounging box.
            y (int): Ordinate of upper-left bounging box.
            w (int): Width.
            h (int): Height.
            style (int): Style of rendering. Possible values are:
                * `D` or empty string: draw border. This is the default value.
                * `F`: fill
                * `DF` or `FD`: draw and fill
        """
        style_to_operators = {"F": "f", "FD": "B", "DF": "B"}
        op = style_to_operators.get(style, "S")
        self._out(
            f"{x * self.k:.2f} {(self.h - y) * self.k:.2f} {w * self.k:.2f} "
            f"{-h * self.k:.2f} re {op}"
        )

    @check_page
    def ellipse(self, x, y, w, h, style=None):
        """
        Outputs an ellipse.
        It can be drawn (border only), filled (with no border) or both.

        Args:
            x (int): Abscissa of upper-left bounging box.
            y (int): Ordinate of upper-left bounging box.
            w (int): Width.
            h (int): Height.
            style (int): Style of rendering. Possible values are:
                * `D` or empty string: draw border. This is the default value.
                * `F`: fill
                * `DF` or `FD`: draw and fill
        """
        style_to_operators = {"F": "f", "FD": "B", "DF": "B"}
        op = style_to_operators.get(style, "S")

        cx = x + w / 2
        cy = y + h / 2
        rx = w / 2
        ry = h / 2

        lx = 4 / 3 * (math.sqrt(2) - 1) * rx
        ly = 4 / 3 * (math.sqrt(2) - 1) * ry

        self._out(
            (
                f"{(cx + rx) * self.k:.2f} {(self.h - cy) * self.k:.2f} m "
                f"{(cx + rx) * self.k:.2f} {(self.h - cy + ly) * self.k:.2f} "
                f"{(cx + lx) * self.k:.2f} {(self.h - cy + ry) * self.k:.2f} "
                f"{cx * self.k:.2f} {(self.h - cy + ry) * self.k:.2f} c"
            )
        )
        self._out(
            (
                f"{(cx - lx) * self.k:.2f} {(self.h - cy + ry) * self.k:.2f} "
                f"{(cx - rx) * self.k:.2f} {(self.h - cy + ly) * self.k:.2f} "
                f"{(cx - rx) * self.k:.2f} {(self.h - cy) * self.k:.2f} c"
            )
        )
        self._out(
            (
                f"{(cx - rx) * self.k:.2f} {(self.h - cy - ly) * self.k:.2f} "
                f"{(cx - lx) * self.k:.2f} {(self.h - cy - ry) * self.k:.2f} "
                f"{cx * self.k:.2f} {(self.h - cy - ry) * self.k:.2f} c"
            )
        )
        self._out(
            (
                f"{(cx + lx) * self.k:.2f} {(self.h - cy - ry) * self.k:.2f} "
                f"{(cx + rx) * self.k:.2f} {(self.h - cy - ly) * self.k:.2f} "
                f"{(cx + rx) * self.k:.2f} {(self.h - cy) * self.k:.2f} c {op}"
            )
        )

    def add_font(self, family, style="", fname=None, uni=False):
        """Add a TrueType or Type1 font"""
        if not fname:
            fname = family.replace(" ", "") + f"{style.lower()}.pkl"
        style = "".join(sorted(style.upper()))
        if any(letter not in "BI" for letter in style):
            raise ValueError(
                f"Unknown style provided (only B & I letters are allowed): {style}"
            )
        fontkey = f"{family.lower()}{style}"

        # Check if font already added or one of the core fonts
        if fontkey in self.fonts or fontkey in self.core_fonts:
            warnings.warn(f"Core font or font already added '{fontkey}': doing nothing")
            return
        if uni:
            for parent in (".", FPDF_FONT_DIR, SYSTEM_TTFONTS):
                if not parent:
                    continue
                if (Path(parent) / fname).exists():
                    ttffilename = Path(parent) / fname
                    break
            else:
                raise FileNotFoundError(f"TTF Font file not found: {fname}")

            cache_dir = (
                Path() if self.font_cache_dir is True else Path(self.font_cache_dir)
            )
            unifilename = cache_dir / f"{ttffilename.stem}.pkl"

            # include numbers in the subset! (if alias present)
            sbarr = list(range(57 if self.str_alias_nb_pages else 32))

            font_dict = load_cache(unifilename)
            if font_dict is None:
                ttf = TTFontFile()
                ttf.getMetrics(ttffilename)
                desc = {
                    "Ascent": round(ttf.ascent),
                    "Descent": round(ttf.descent),
                    "CapHeight": round(ttf.capHeight),
                    "Flags": ttf.flags,
                    "FontBBox": (
                        f"[{ttf.bbox[0]:.0f} {ttf.bbox[1]:.0f}"
                        f" {ttf.bbox[2]:.0f} {ttf.bbox[3]:.0f}]"
                    ),
                    "ItalicAngle": int(ttf.italicAngle),
                    "StemV": round(ttf.stemV),
                    "MissingWidth": round(ttf.defaultWidth),
                }

                # Generate metrics .pkl file
                font_dict = {
                    "type": "TTF",
                    "name": re.sub("[ ()]", "", ttf.fullName),
                    "desc": desc,
                    "up": round(ttf.underlinePosition),
                    "ut": round(ttf.underlineThickness),
                    "ttffile": ttffilename,
                    "fontkey": fontkey,
                    "subset": sbarr,
                    "unifilename": unifilename,
                    "originalsize": os.stat(ttffilename).st_size,
                    "cw": ttf.charWidths,
                }

                if unifilename:
                    try:
                        unifilename.write_bytes(pickle.dumps(font_dict))
                    except OSError as e:
                        if e.errno != errno.EACCES:
                            raise  # Not a permission error.

            self.fonts[fontkey] = {
                "i": len(self.fonts) + 1,
                "type": font_dict["type"],
                "name": font_dict["name"],
                "desc": font_dict["desc"],
                "up": font_dict["up"],
                "ut": font_dict["ut"],
                "cw": font_dict["cw"],
                "ttffile": font_dict["ttffile"],
                "fontkey": fontkey,
                "subset": sbarr,
                "unifilename": unifilename,
            }
            self.font_files[fontkey] = {
                "length1": font_dict["originalsize"],
                "type": "TTF",
                "ttffile": ttffilename,
            }
            self.font_files[fname] = {"type": "TTF"}
        else:
            if fname.endswith(".ttf"):
                warnings.warn(
                    "When providing a TTF font file you must pass uni=True to FPDF.set_font"
                )
            font_dict = pickle.loads(Path(fname).read_bytes())
            self.fonts[fontkey] = {"i": len(self.fonts) + 1}
            self.fonts[fontkey].update(font_dict)
            diff = font_dict.get("diff")
            if diff:
                # Search existing encodings
                nb = len(self.diffs)
                for i in range(1, nb + 1):
                    if self.diffs[i] == diff:
                        d = i
                        break
                else:
                    d = nb + 1
                    self.diffs[d] = diff
                self.fonts[fontkey]["diff"] = d
            filename = font_dict.get("filename")
            if filename:
                if font_dict["type"] == "TrueType":
                    originalsize = font_dict["originalsize"]
                    self.font_files[filename] = {"length1": originalsize}
                else:
                    self.font_files[filename] = {
                        "length1": font_dict["size1"],
                        "length2": font_dict["size2"],
                    }

    def set_font(self, family=None, style="", size=0):
        """
        Sets the font used to print character strings.
        It is mandatory to call this method at least once before printing text.

        Default encoding is not specified, but all text writing methods accept only
        unicode for external fonts and one byte encoding for standard.

        Standard fonts use `Latin-1` encoding by default, but Windows
        encoding `cp1252` (Western Europe) can be used with
        [set_doc_option](set_doc_option.md) ("core_fonts_encoding", encoding).

        The font specified is retained from page to page.
        The method can be called before the first page is created.

        Args:
            family (str): name of a font added with `FPDF.add_font`,
                or name of one of the 14 standard "PostScript" fonts:
                Courier (fixed-width), Helvetica (sans serif), Times (serif),
                Symbol (symbolic) or ZapfDingbats (symbolic)
                If an empty string is provided, the current family is retained.
            style (str): empty string (by default) or a combination
                of one or several letters among B (bold), I (italic) and U (underline).
                Bold and italic styles do not apply to Symbol and ZapfDingbats fonts.
            size (int): in points. The default value is the current size.
        """
        if not family:
            family = self.font_family

        family = family.lower()
        style = "".join(sorted(style.upper()))
        if any(letter not in "BIU" for letter in style):
            raise ValueError(
                f"Unknown style provided (only B/I/U letters are allowed): {style}"
            )
        if "U" in style:
            self.underline = 1
            style = style.replace("U", "")
        else:
            self.underline = 0

        if family in self.font_aliases and family + style not in self.fonts:
            warnings.warn(
                f"Substituting font {family} by core font "
                f"{self.font_aliases[family]}"
            )
            family = self.font_aliases[family]
        elif family in ("symbol", "zapfdingbats") and style:
            warnings.warn(
                f"Built-in font {family} only has a single 'style' and can't be bold "
                f"or italic"
            )
            style = ""

        if size == 0:
            size = self.font_size_pt

        # Test if font is already selected
        if (
            self.font_family == family
            and self.font_style == style
            and self.font_size_pt == size
        ):
            return

        # Test if used for the first time
        fontkey = family + style
        if fontkey not in self.fonts:
            if fontkey not in self.core_fonts:
                raise FPDFException(
                    f"Undefined font: {fontkey} - "
                    f"Use built-in fonts or FPDF.add_font() beforehand"
                )
            # If it's one of the core fonts, add it to self.fonts
            self.fonts[fontkey] = {
                "i": len(self.fonts) + 1,
                "type": "core",
                "name": self.core_fonts[fontkey],
                "up": -100,
                "ut": 50,
                "cw": fpdf_charwidths[fontkey],
                "fontkey": fontkey,
            }

        # Select it
        self.font_family = family
        self.font_style = style
        self.font_size_pt = size
        self.font_size = size / self.k
        self.current_font = self.fonts[fontkey]
        if self.page > 0:
            self._out(f"BT /F{self.current_font['i']} {self.font_size_pt:.2f} Tf ET")

    def set_font_size(self, size):
        """Set font size in points"""
        if self.font_size_pt == size:
            return
        self.font_size_pt = size
        self.font_size = size / self.k
        if self.page > 0:
            self._out(f"BT /F{self.current_font['i']} {self.font_size_pt:.2f} Tf ET")

    def set_stretching(self, factor):
        """Set from stretch factor percents (default: 100.0)"""
        if self.font_stretching == factor:
            return
        self.font_stretching = factor
        if self.page > 0:
            self._out(f"BT {self.font_stretching:.2f} Tz ET")

    def add_link(self):
        """Create a new internal link"""
        n = len(self.links) + 1
        self.links[n] = InternalLink()
        return n

    def set_link(self, link, y=0, page=-1):
        """Set destination of internal link"""
        if y == -1:
            y = self.y
        if page == -1:
            page_object_id = self._current_page_object_id()
        else:
            page_object_id = 2 * page + 1
        self.links[link] = InternalLink(page_object_id, y)

    def link(self, x, y, w, h, link, alt_text=None):
        """
        Puts a link annotation on a rectangular area of the page.
        Text or image links are generally put via [cell](#fpdf.FPDF.cell),
        [write](#fpdf.FPDF.write) or [image](#fpdf.FPDF.image),
        but this method can be useful for instance to define a clickable area inside an image.

        Args:
            x (int): horizontal position (from the left) to the left side of the link rectangle
            y (int): vertical position (from the top) to the bottom side of the link rectangle
            w (int): width of the link rectangle
            h (int): width of the link rectangle
            link: either an URL or a integer returned by `add_link`, defining an internal link to a page
            alt_text (str): optional textual description of the link, for accessibility purposes
        """
        self.annots[self.page].append(
            Annotation(
                "Link",
                x * self.k,
                self.h_pt - y * self.k,
                w * self.k,
                h * self.k,
                link=link,
                alt_text=alt_text,
            )
        )

    def text_annotation(self, x, y, w, h, text):
        """
        Puts a text annotation on a rectangular area of the page.

        Args:
            x (int): horizontal position (from the left) to the left side of the link rectangle
            y (int): vertical position (from the top) to the bottom side of the link rectangle
            w (int): width of the link rectangle
            h (int): width of the link rectangle
            text (str): text to display
        """
        self.annots[self.page].append(
            Annotation(
                "Text",
                x * self.k,
                self.h_pt - y * self.k,
                w * self.k,
                h * self.k,
                contents=text,
            )
        )

    @check_page
    def text(self, x, y, txt=""):
        """Output a string"""
        if not self.font_family:
            raise FPDFException("No font set, you need to call set_font() beforehand")
        txt = self.normalize_text(txt)
        if self.unifontsubset:
            txt2 = escape_parens(txt).encode("UTF-16BE").decode("latin-1")
            for char in txt:
                self.current_font["subset"].append(ord(char))
        else:
            txt2 = escape_parens(txt)
        s = f"BT {x * self.k:.2f} {(self.h - y) * self.k:.2f} Td ({txt2}) Tj ET"
        if self.underline and txt != "":
            s += " " + self._dounderline(x, y, txt)
        if self.fill_color != self.text_color:
            s = f"q {self.text_color} {s} Q"
        self._out(s)

    @check_page
    def rotate(self, angle, x=None, y=None):
        """
        .. deprecated:: 2.1.0
          Use `rotation` instead.
        """
        warnings.warn(
            "rotate() can produces malformed PDFs and is deprecated. "
            "Use the rotation() context manager instead.",
            PendingDeprecationWarning,
        )
        if x is None:
            x = self.x
        if y is None:
            y = self.y

        if self.angle != 0:
            self._out("Q")
        self.angle = angle
        if angle != 0:
            angle *= math.pi / 180
            c = math.cos(angle)
            s = math.sin(angle)
            cx = x * self.k
            cy = (self.h - y) * self.k
            s = (
                f"q {c:.5F} {s:.5F} {-s:.5F} {c:.5F} {cx:.2F} {cy:.2F} cm "
                f"1 0 0 1 {-cx:.2F} {-cy:.2F} cm"
            )
            self._out(s)

    @check_page
    @contextmanager
    def rotation(self, angle, x=None, y=None):
        """
        This method allows to perform a rotation around a given center.

        The rotation affects all elements which are printed inside the indented context
        (with the exception of clickable areas).

        Args:
            angle (float): angle in degrees
            x (float): abscissa of the center of the rotation
            y (float): ordinate of the center of the rotation

        Notes
        -----

        Only the rendering is altered. The `get_x()` and `get_y()` methods are not
        affected, nor the automatic page break mechanism.
        """
        if x is None:
            x = self.x
        if y is None:
            y = self.y
        angle *= math.pi / 180
        c, s = math.cos(angle), math.sin(angle)
        cx, cy = x * self.k, (self.h - y) * self.k
        s = (
            f"q {c:.5F} {s:.5F} {-s:.5F} {c:.5F} {cx:.2F} {cy:.2F} cm "
            f"1 0 0 1 {-cx:.2F} {-cy:.2F} cm\n"
        )
        self._out(s)
        yield
        self._out("Q\n")

    @property
    def accept_page_break(self):
        """Accept automatic page break or not"""
        return self.auto_page_break

    @check_page
    def cell(self, w, h=0, txt="", border=0, ln=0, align="", fill=False, link=""):
        """
        Prints a cell (rectangular area) with optional borders, background color and
        character string. The upper-left corner of the cell corresponds to the current
        position. The text can be aligned or centered. After the call, the current
        position moves to the right or to the next line. It is possible to put a link
        on the text.

        If automatic page breaking is enabled and the cell goes beyond the limit, a
        page break is performed before outputting.

        Args:
            w (int): Cell width. If 0, the cell extends up to the right margin.
            h (int): Cell height. Default value: 0.
            txt (str): String to print. Default value: empty string.
            border: Indicates if borders must be drawn around the cell.
                The value can be either a number (`0`: no border ; `1`: frame)
                or a string containing some or all of the following characters
                (in any order):
                `L`: left ; `T`: top ; `R`: right ; `B`: bottom. Default value: 0.
            ln (int): Indicates where the current position should go after the call.
                Possible values are: `0`: to the right ; `1`: to the beginning of the
                next line ; `2`: below. Putting 1 is equivalent to putting 0 and calling
                `ln` just after. Default value: 0.
            align (str): Allows to center or align the text. Possible values are:
                `L` or empty string: left align (default value) ; `C`: center ;
                `R`: right align
            fill (bool): Indicates if the cell background must be painted (`True`)
                or transparent (`False`). Default value: False.
            link (str): optional link to add on the cell, internal
                (identifier returned by `add_link`) or external URL.

        Returns: a boolean indicating if page break was triggered
        """
        if not self.font_family:
            raise FPDFException("No font set, you need to call set_font() beforehand")
        if isinstance(border, int) and border not in (0, 1):
            warnings.warn(
                'Integer values for "border" parameter other than 1 are currently '
                "ignored"
            )
            border = 1
        page_break_triggered = self._perform_page_break_if_need_be(h)
        if w == 0:
            w = self.w - self.r_margin - self.x
        s = ""

        k = self.k
        if fill:
            op = "B" if border == 1 else "f"
            s = (
                f"{self.x * k:.2f} {(self.h - self.y) * k:.2f} "
                f"{w * k:.2f} {-h * k:.2f} re {op} "
            )
        elif border == 1:
            s = (
                f"{self.x * k:.2f} {(self.h - self.y) * k:.2f} "
                f"{w * k:.2f} {-h * k:.2f} re S "
            )

        if isinstance(border, str):
            x = self.x
            y = self.y
            if "L" in border:
                s += (
                    f"{x * k:.2f} {(self.h - y) * k:.2f} m "
                    f"{x * k:.2f} {(self.h - (y + h)) * k:.2f} l S "
                )
            if "T" in border:
                s += (
                    f"{x * k:.2f} {(self.h - y) * k:.2f} m "
                    f"{(x + w) * k:.2f} {(self.h - y) * k:.2f} l S "
                )
            if "R" in border:
                s += (
                    f"{(x + w) * k:.2f} {(self.h - y) * k:.2f} m "
                    f"{(x + w) * k:.2f} {(self.h - (y + h)) * k:.2f} l S "
                )
            if "B" in border:
                s += (
                    f"{x * k:.2f} {(self.h - (y + h)) * k:.2f} m "
                    f"{(x + w) * k:.2f} {(self.h - (y + h)) * k:.2f} l S "
                )

        txt = self.normalize_text(txt)
        if txt != "":
            if align == "R":
                dx = w - self.c_margin - self.get_string_width(txt, True)
            elif align == "C":
                dx = (w - self.get_string_width(txt, True)) / 2
            else:
                dx = self.c_margin
            if self.fill_color != self.text_color:
                s += f"q {self.text_color} "

            # If multibyte, Tw has no effect - do word spacing using an
            # adjustment before each space
            if self.ws and self.unifontsubset:
                for char in txt:
                    self.current_font["subset"].append(ord(char))
                space = escape_parens(" ".encode("UTF-16BE").decode("latin-1"))

                s += (
                    f"BT 0 Tw {(self.x + dx) * k:.2F} "
                    f"{(self.h - self.y - 0.5 * h - 0.3 * self.font_size) * k:.2F} "
                    "Td ["
                )

                words = txt.split(" ")
                for i, word in enumerate(words):
                    word = escape_parens(word.encode("UTF-16BE").decode("latin-1"))
                    s += f"({word}) "
                    is_last_word = (i + 1) == len(words)
                    if not is_last_word:
                        adj = -(self.ws * self.k) * 1000 / self.font_size_pt
                        s += f"{adj}({space}) "
                s += "] TJ"
                s += " ET"
            else:
                if self.unifontsubset:
                    txt2 = escape_parens(txt.encode("UTF-16BE").decode("latin-1"))
                    for char in txt:
                        self.current_font["subset"].append(ord(char))
                else:
                    txt2 = escape_parens(txt)

                s += (
                    f"BT {(self.x + dx) * k:.2f} "
                    f"{(self.h - self.y - 0.5 * h - 0.3 * self.font_size) * k:.2f} "
                    f"Td ({txt2}) Tj ET"
                )

            if self.underline:
                s += " " + self._dounderline(
                    self.x + dx, self.y + (0.5 * h) + (0.3 * self.font_size), txt
                )
            if self.fill_color != self.text_color:
                s += " Q"
            if link:
                self.link(
                    self.x + dx,
                    self.y + (0.5 * h) - (0.5 * self.font_size),
                    self.get_string_width(txt, True),
                    self.font_size,
                    link,
                )
        if s:
            self._out(s)
        self.lasth = h

        if ln > 0:
            self.y += h  # Go to next line
            if ln == 1:
                self.x = self.l_margin
        else:
            self.x += w

        return page_break_triggered

    def _perform_page_break_if_need_be(self, h):
        if (
            self.y + h > self.page_break_trigger
            and not self.in_footer
            and self.accept_page_break
        ):
            LOGGER.debug(
                "Page break on page %d at y=%d for element of height %d",
                self.page,
                self.y,
                h,
            )
            x, ws = self.x, self.ws
            if ws > 0:
                self.ws = 0
                self._out("0 Tw")
            self.add_page(same=True)
            self.x = x  # restore x but not y after drawing header
            if ws > 0:
                self.ws = ws
                self._out(f"{ws * self.k:.3f} Tw")
            return True
        return False

    @check_page
    def multi_cell(
        self,
        w,
        h,
        txt="",
        border=0,
        align="J",
        fill=False,
        split_only=False,
        link="",
        ln=0,
        max_line_height=None,
    ):
        """
        This method allows printing text with line breaks. They can be automatic (as
        soon as the text reaches the right border of the cell) or explicit (via the
        `\n` character). As many cells as necessary are stacked, one below the other.
        Text can be aligned, centered or justified. The cell block can be framed and
        the background painted.

        Args:
            w (int): cells width. If 0, they extend up to the right margin of the page.
            h (int): cells height.
            txt (str): strign to print.
            border: Indicates if borders must be drawn around the cell.
                The value can be either a number (`0`: no border ; `1`: frame)
                or a string containing some or all of the following characters
                (in any order):
                `L`: left ; `T`: top ; `R`: right ; `B`: bottom. Default value: 0.
            align (str): Allows to center or align the text. Possible values are:
                `J`: justify (default value); `L` or empty string: left align ;
                `C`: center ; `R`: right align
            fill (bool): Indicates if the cell background must be painted (`True`)
                or transparent (`False`). Default value: False.
            split_only (bool): if `True`, does not output anything, only perform
                word-wrapping and return the resulting multi-lines array of strings.
            link (str): optional link to add on the cell, internal
                (identifier returned by `add_link`) or external URL.
            ln (int): Indicates where the current position should go after the call.
                Possible values are: `0`: to the bottom right ; `1`: to the beginning
                of the next line ; `2`: below with the same horizontal offset ;
                `3`: to the right with the same vertical offset. Default value: 0.
            max_line_height (int): optional maximum height of each sub-cell generated

        Using `ln=3` and `maximum height=pdf.font_size` is useful to build tables
        with multiline text in cells.

        Returns: a boolean indicating if page break was triggered.
        """
        page_break_triggered = False
        if split_only:
            _out, _add_page = self._out, self.add_page
            self._out = lambda *args, **kwargs: None
            self.add_page = lambda *args, **kwargs: None

        # Store this information for manipulating position.
        location = (self.get_x(), self.get_y())

        # If width is 0, set width to available width between margins
        if w == 0:
            w = self.w - self.r_margin - self.x
        wmax = (w - 2 * self.c_margin) * 1000 / self.font_size

        # Calculate text length
        txt = self.normalize_text(txt)
        s = txt.replace("\r", "")
        normalized_string_length = len(s)
        if normalized_string_length > 0 and s[-1] == "\n":
            normalized_string_length -= 1

        b = 0
        if border:
            if border == 1:
                border = "LTRB"
                b = "LRT"
                b2 = "LR"
            else:
                b2 = ""
                if "L" in border:
                    b2 += "L"
                if "R" in border:
                    b2 += "R"
                b = b2 + "T" if "T" in border else b2
        character_widths = self.current_font["cw"]
        text_cells = []
        sep = -1
        i = 0
        j = 0
        l = 0
        ns = 0
        nl = 1
        prev_x, prev_y = self.x, self.y
        while i < normalized_string_length:
            # Get next character
            c = s[i]

            # Explicit line break
            if c == "\n":
                if self.ws > 0:
                    self.ws = 0
                    self._out("0 Tw")

                if max_line_height and h > max_line_height:
                    height = max_line_height
                    h -= height
                else:
                    height = h
                new_page = self.cell(
                    w,
                    h=height,
                    txt=substr(s, j, i - j),
                    border=b,
                    ln=2,
                    align=align,
                    fill=fill,
                    link=link,
                )
                page_break_triggered = page_break_triggered or new_page
                text_cells.append(substr(s, j, i - j))

                i += 1
                sep = -1
                j = i
                l = 0
                ns = 0
                nl += 1
                if border and nl == 2:
                    b = b2
                continue

            if c == " ":
                sep = i
                ls = l
                ns += 1
            if self.unifontsubset:
                l += self.get_string_width(c, True) / self.font_size * 1000
            else:
                l += character_widths.get(c, 0)

            # Automatic line break
            if l > wmax:
                if sep == -1:
                    if i == j:
                        i += 1
                    if self.ws > 0:
                        self.ws = 0
                        self._out("0 Tw")

                    if max_line_height and h > max_line_height:
                        height = max_line_height
                        h -= height
                    else:
                        height = h
                    new_page = self.cell(
                        w,
                        h=height,
                        txt=substr(s, j, i - j),
                        border=b,
                        ln=2,
                        align=align,
                        fill=fill,
                        link=link,
                    )
                    page_break_triggered = page_break_triggered or new_page
                    text_cells.append(substr(s, j, i - j))

                else:
                    if align == "J":
                        self.ws = (
                            (wmax - ls) / 1000 * self.font_size / (ns - 1)
                            if ns > 1
                            else 0
                        )
                        self._out(f"{self.ws * self.k:.3f} Tw")

                    if max_line_height and h > max_line_height:
                        height = max_line_height
                        h -= height
                    else:
                        height = h
                    new_page = self.cell(
                        w,
                        h=height,
                        txt=substr(s, j, sep - j),
                        border=b,
                        ln=2,
                        align=align,
                        fill=fill,
                        link=link,
                    )
                    page_break_triggered = page_break_triggered or new_page
                    text_cells.append(substr(s, j, sep - j))

                    i = sep + 1
                sep = -1
                j = i
                l = 0
                ns = 0
                nl += 1
                if border and nl == 2:
                    b = b2
            else:
                i += 1

        # Last chunk
        if self.ws > 0:
            self.ws = 0
            self._out("0 Tw")
        if border and "B" in border:
            b += "B"

        new_page = self.cell(
            w,
            h=h,
            txt=substr(s, j, i - j),
            border=b,
            ln=0 if ln == 3 else ln,
            align=align,
            fill=fill,
            link=link,
        )
        if new_page:
            # When a page jump is performed and ln=3,
            # we stick to that new vertical offset.
            # cf. test_multi_cell_table_with_automatic_page_break
            prev_y = self.y
        page_break_triggered = page_break_triggered or new_page
        text_cells.append(substr(s, j, i - j))

        new_x, new_y = {
            0: (self.x, self.y + h),
            1: (self.l_margin, self.y),
            2: (prev_x, self.y),
            3: (self.x, prev_y),
        }[ln]
        self.set_xy(new_x, new_y)

        if split_only:
            # restore writing functions
            self._out, self.add_page = _out, _add_page
            self.set_xy(*location)  # restore location
            return text_cells

        return page_break_triggered

    @check_page
    def write(self, h, txt="", link=""):
        """Output text in flowing mode"""
        if not self.font_family:
            raise FPDFException("No font set, you need to call set_font() beforehand")
        txt = self.normalize_text(txt)
        cw = self.current_font["cw"]
        w = self.w - self.r_margin - self.x
        wmax = (w - 2 * self.c_margin) * 1000 / self.font_size
        s = txt.replace("\r", "")
        nb = len(s)
        sep = -1
        i = 0
        j = 0
        l = 0
        nl = 1
        while i < nb:
            # Get next character
            c = s[i]
            if c == "\n":
                # Explicit line break
                self.cell(w, h, substr(s, j, i - j), 0, 2, "", False, link)
                i += 1
                sep = -1
                j = i
                l = 0
                if nl == 1:
                    self.x = self.l_margin
                    w = self.w - self.r_margin - self.x
                    wmax = (w - 2 * self.c_margin) * 1000 / self.font_size
                nl += 1
                continue
            if c == " ":
                sep = i
            if self.unifontsubset:
                l += self.get_string_width(c, True) / self.font_size * 1000
            else:
                l += cw.get(c, 0)
            if l > wmax:
                # Automatic line break
                if sep == -1:
                    if self.x > self.l_margin:
                        # Move to next line
                        self.x = self.l_margin
                        self.y += h
                        w = self.w - self.r_margin - self.x
                        wmax = (w - 2 * self.c_margin) * 1000 / self.font_size
                        i += 1
                        nl += 1
                        continue
                    if i == j:
                        i += 1
                    self.cell(w, h, substr(s, j, i - j), 0, 2, "", False, link)
                else:
                    self.cell(w, h, substr(s, j, sep - j), 0, 2, "", False, link)
                    i = sep + 1
                sep = -1
                j = i
                l = 0
                if nl == 1:
                    self.x = self.l_margin
                    w = self.w - self.r_margin - self.x
                    wmax = (w - 2 * self.c_margin) * 1000 / self.font_size
                nl += 1
            else:
                i += 1
        # Last chunk
        if i != j:
            self.cell(l / 1000 * self.font_size, h, substr(s, j), 0, 0, "", False, link)

    @check_page
    def image(
        self,
        name,
        x=None,
        y=None,
        w=0,
        h=0,
        type="",
        link="",
        title=None,
        alt_text=None,
    ):
        """
        Put an image on the page.

        The size of the image on the page can be specified in different ways:
        * explicit width and height (expressed in user units)
        * one explicit dimension, the other being calculated automatically
          in order to keep the original proportions
        * no explicit dimension, in which case the image is put at 72 dpi.

        **Remarks**:
        * if an image is used several times, only one copy is embedded in the file.
        * when using an animated GIF, only the first frame is used.

        Args:
            name: either a string representing a file path to an image, an URL to an image,
                an io.BytesIO, or a instance of `PIL.Image.Image`
            x (int): optional horizontal position where to put the image on the page.
                If not specified or equal to None, the current abscissa is used.
            y (int): optional vertical position where to put the image on the page.
                If not specified or equal to None, the current ordinate is used.
                After the call, the current ordinate is moved to the bottom of the image
            w (int): optional width of the image. If not specified or equal to zero,
                it is automatically calculated from the image size.
                Pass `pdf.epw` to scale horizontally to the full page width.
            h (int): optional height of the image. If not specified or equal to zero,
                it is automatically calculated from the image size.
                Pass `pdf.eph` to scale horizontally to the full page height.
            type (str): [**DEPRECATED**] unused, will be removed in a later version.
            link (str): optional link to add on the image, internal
                (identifier returned by `add_link`) or external URL.
            title (str): optional. Currently, never seem rendered by PDF readers.
            alt_text (str): optional alternative text describing the image,
                for accessibility purposes. Displayed by some PDF readers on hover.
        """
        if type:
            warnings.warn(
                '"type" is unused and will soon be deprecated',
                PendingDeprecationWarning,
            )
        if isinstance(name, str):
            img = None
        elif isinstance(name, Image.Image):
            name, img = hashlib.md5(name.tobytes()).hexdigest(), name
        elif isinstance(name, io.BytesIO):
            name, img = hashlib.md5(name.getvalue()).hexdigest(), name
        else:
            name, img = str(name), name
        if name not in self.images:
            info = get_img_info(img or load_resource(name))
            info["i"] = len(self.images) + 1
            self.images[name] = info
        else:
            info = self.images[name]

        # Automatic width and height calculation if needed
        if w == 0 and h == 0:
            # Put image at 72 dpi
            w = info["w"] / self.k
            h = info["h"] / self.k
        elif w == 0:
            w = h * info["w"] / info["h"]
        elif h == 0:
            h = w * info["h"] / info["w"]

        # Flowing mode
        if y is None:
            self._perform_page_break_if_need_be(h)
            y = self.y
            self.y += h

        if x is None:
            x = self.x

        stream_content = (
            f"q {w * self.k:.2f} 0 0 {h * self.k:.2f} {x * self.k:.2f} "
            f"{(self.h - y - h) * self.k:.2f} cm /I{info['i']} Do Q"
        )
        if title or alt_text:
            with self._marked_sequence(title=title, alt_text=alt_text):
                self._out(stream_content)
        else:
            self._out(stream_content)
        if link:
            self.link(x, y, w, h, link)

        return info

    @contextmanager
    def _marked_sequence(self, **kwargs):
        page_object_id = self._current_page_object_id()
        mcid = self.struct_builder.next_mcid_for_page(page_object_id)
        marked_content = self._add_marked_content(
            page_object_id, struct_type="/Figure", mcid=mcid, **kwargs
        )
        self._out(f"/P <</MCID {mcid}>> BDC")
        yield marked_content
        self._out("EMC")

    def _add_marked_content(self, page_object_id, **kwargs):
        struct_parents_id = self._struct_parents_id_per_page.get(page_object_id)
        if struct_parents_id is None:
            struct_parents_id = len(self._struct_parents_id_per_page)
            self._struct_parents_id_per_page[page_object_id] = struct_parents_id
        marked_content = MarkedContent(page_object_id, struct_parents_id, **kwargs)
        self.struct_builder.add_marked_content(marked_content)
        return marked_content

    def _current_page_object_id(self):
        # Predictable given that _putpages is invoked first in _enddoc:
        return 2 * self.page + 1

    @check_page
    def ln(self, h=None):
        """
        Line Feed.
        The current abscissa goes back to the left margin and the ordinate increases by
        the amount passed as parameter.

        Args:
            h (int): The height of the break.
                By default, the value equals the height of the last printed cell.
        """
        self.x = self.l_margin
        self.y += self.lasth if h is None else h

    def get_x(self):
        """Get x position"""
        return self.x

    def set_x(self, x):
        """Set x position"""
        self.x = x if x >= 0 else self.w + x

    def get_y(self):
        """Get y position"""
        return self.y

    def set_y(self, y):
        """Set y position and reset x"""
        self.x = self.l_margin
        self.y = y if y >= 0 else self.h + y

    def set_xy(self, x, y):
        """Set x and y positions"""
        self.set_y(y)
        self.set_x(x)

    def output(self, name="", dest=""):
        """
        Output PDF to some destination.
        The method first calls [close](close.md) if necessary to terminate the document.

        By default the bytearray buffer is returned.
        If a `name` is given, the PDF is written to a new file.

        Args:
            name (str): optional File object or file path where to save the PDF under
            dest (str): [**DEPRECATED**] unused, will be removed in a later version
        """
        if dest:
            warnings.warn(
                '"dest" is unused and will soon be deprecated',
                PendingDeprecationWarning,
            )
        # Finish document if necessary:
        if self.state < DocumentState.CLOSED:
            self.close()
        if name:
            if isinstance(name, os.PathLike):
                name.write_bytes(self.buffer)
            elif isinstance(name, str):
                Path(name).write_bytes(self.buffer)
            else:
                name.write(self.buffer)
            return None
        return self.buffer

    def normalize_text(self, txt):
        """Check that text input is in the correct format/encoding"""
        # - for TTF unicode fonts: unicode object (utf8 encoding)
        # - for built-in fonts: string instances (encoding: latin-1, cp1252)
        if not self.unifontsubset and self.core_fonts_encoding:
            return txt.encode(self.core_fonts_encoding).decode("latin-1")
        return txt

    def _putpages(self):
        nb = self.page  # total number of pages
        if self.str_alias_nb_pages:
            self._substitute_page_number()
        if self._toc_placeholder:
            self._insert_table_of_contents()
        if self.def_orientation == "P":
            dw_pt = self.dw_pt
            dh_pt = self.dh_pt
        else:
            dw_pt = self.dh_pt
            dh_pt = self.dw_pt
        filter = "/Filter /FlateDecode " if self.compress else ""
        for n in range(1, nb + 1):
            # Page
            self._newobj()
            self._out("<</Type /Page")
            self._out("/Parent 1 0 R")
            w_pt = self.pages[n]["w_pt"]
            h_pt = self.pages[n]["h_pt"]
            if w_pt != dw_pt or h_pt != dh_pt:
                self._out(f"/MediaBox [0 0 {w_pt:.2f} {h_pt:.2f}]")
            self._out("/Resources 2 0 R")

            page_annots = self.annots[n]
            if page_annots:  # Annotations, e.g. links:
                annots = ""
                for annot in page_annots:
                    # first four things in 'link' list are coordinates?
                    rect = (
                        f"{annot.x:.2f} {annot.y:.2f} "
                        f"{annot.x + annot.width:.2f} {annot.y - annot.height:.2f}"
                    )

                    # start the annotation entry
                    annots += (
                        f"<</Type /Annot /Subtype /{annot.type} "
                        f"/Rect [{rect}] /Border [0 0 0] "
                        # Flag "Print" (bit position 3) specifies to print
                        # the annotation when the page is printed.
                        # cf. https://docs.verapdf.org/validation/pdfa-part1/#rule-653-2
                        f"/F 4"
                    )

                    if annot.contents:
                        annots += f"/Contents {enclose_in_parens(annot.contents)}"

                    if annot.alt_text is not None:
                        # Note: the spec indicates that a /StructParent could be added **inside* this /Annot,
                        # but tests with Adobe Acrobat Reader reveal that the page /StructParents inserted below
                        # is enough to link the marked content in the hierarchy tree with this annotation link.
                        self._add_marked_content(
                            self.n, struct_type="/Link", alt_text=annot.alt_text
                        )

                    if annot.link:
                        if isinstance(annot.link, str):
                            annots += (
                                f"/A <</S /URI /URI {enclose_in_parens(annot.link)}>>"
                            )
                        else:  # Dest type ending of annotation entry
                            assert annot.link in self.links, (
                                f"Page {n} has a link with an invalid index: "
                                f"{annot.link} (doc #links={len(self.links)})"
                            )
                            link = self.links[annot.link]
                            annots += f"/Dest {link.dest(self)}"
                    annots += ">>"
                # End links list
                self._out(f"/Annots [{annots}]")
            if self.pdf_version > "1.3":
                self._out("/Group <</Type /Group /S /Transparency" "/CS /DeviceRGB>>")
            spid = self._struct_parents_id_per_page.get(self.n)
            if spid is not None:
                self._out(f"/StructParents {spid}")
            self._out(f"/Contents {self.n + 1} 0 R>>")
            self._out("endobj")

            # Page content
            content = self.pages[n]["content"]
            p = zlib.compress(content) if self.compress else content
            self._newobj()
            self._out(f"<<{filter}/Length {len(p)}>>")
            self._out(pdf_stream(p))
            self._out("endobj")
        # Pages root
        self.offsets[1] = len(self.buffer)
        self._out("1 0 obj")
        self._out("<</Type /Pages")
        self._out("/Kids [" + " ".join(f"{3 + 2 * i} 0 R" for i in range(nb)) + "]")
        self._out(f"/Count {nb}")
        self._out(f"/MediaBox [0 0 {dw_pt:.2f} {dh_pt:.2f}]")
        self._out(">>")
        self._out("endobj")

    def _substitute_page_number(self):
        nb = self.page  # total number of pages
        substituted = False
        # Replace number of pages in fonts using subsets (unicode)
        alias = self.str_alias_nb_pages.encode("UTF-16BE")
        encoded_nb = str(nb).encode("UTF-16BE")
        for n in range(1, nb + 1):
            new_content = self.pages[n]["content"].replace(alias, encoded_nb)
            substituted |= self.pages[n]["content"] != new_content
            self.pages[n]["content"] = new_content
        # Now repeat for no pages in non-subset fonts
        alias = self.str_alias_nb_pages.encode("latin-1")
        encoded_nb = str(nb).encode("latin-1")
        for n in range(1, nb + 1):
            new_content = self.pages[n]["content"].replace(alias, encoded_nb)
            substituted |= self.pages[n]["content"] != new_content
            self.pages[n]["content"] = new_content
        if substituted:
            LOGGER.info(
                "Substitution of '%s' was performed in the document",
                self.str_alias_nb_pages,
            )

    def _insert_table_of_contents(self):
        prev_state = self.state
        tocp = self._toc_placeholder
        self.page = tocp.start_page
        # Doc has been closed but we want to write to self.pages[self.page] instead of self.buffer:
        self.state = DocumentState.GENERATING_PAGE
        self.y = tocp.y
        tocp.render_function(self, self._outline)
        expected_final_page = tocp.start_page + tocp.pages - 1
        if self.page != expected_final_page:
            too = "many" if self.page > expected_final_page else "few"
            error_msg = f"The rendering function passed to FPDF.insert_toc_placeholder triggered too {too} page breaks: "
            error_msg += f"ToC ended on page {self.page} while it was expected to span exactly {tocp.pages} pages"
            raise FPDFException(error_msg)
        self.state = prev_state

    def _putfonts(self):
        nf = self.n
        for diff in self.diffs:
            # Encodings
            self._newobj()
            self._out(
                "<</Type /Encoding /BaseEncoding /WinAnsiEncoding "
                + "/Differences ["
                + self.diffs[diff]
                + "]>>"
            )
            self._out("endobj")

        for name, info in self.font_files.items():
            if "type" in info and info["type"] != "TTF":
                # Font file embedding
                self._newobj()
                self.font_files[name]["n"] = self.n
                font = (FPDF_FONT_DIR / name).read_bytes()
                compressed = substr(name, -2) == ".z"
                if not compressed and "length2" in info:
                    header = ord(font[0]) == 128
                    if header:
                        # Strip first binary header
                        font = substr(font, 6)
                    if header and ord(font[info["length1"]]) == 128:
                        # Strip second binary header
                        font = substr(font, 0, info["length1"]) + substr(
                            font, info["length1"] + 6
                        )

                self._out(f"<</Length {len(font)}")
                if compressed:
                    self._out("/Filter /FlateDecode")
                self._out(f"/Length1 {info['length1']}")
                if "length2" in info:
                    self._out(f"/Length2 {info['length2']} /Length3 0")
                self._out(">>")
                self._out(pdf_stream(font))
                self._out("endobj")

        # Font objects
        flist = [(x[1]["i"], x[0], x[1]) for x in self.fonts.items()]
        flist.sort()
        for _, font_name, font in flist:
            self.fonts[font_name]["n"] = self.n + 1
            my_type = font["type"]
            name = font["name"]
            # Standard font
            if my_type == "core":
                self._newobj()
                self._out("<</Type /Font")
                self._out(f"/BaseFont /{name}")
                self._out("/Subtype /Type1")
                if name not in ("Symbol", "ZapfDingbats"):
                    self._out("/Encoding /WinAnsiEncoding")
                self._out(">>")
                self._out("endobj")

            # Additional Type1 or TrueType font
            elif my_type in ("Type1", "TrueType"):
                self._newobj()
                self._out("<</Type /Font")
                self._out(f"/BaseFont /{name}")
                self._out(f"/Subtype /{my_type}")
                self._out("/FirstChar 32 /LastChar 255")
                self._out(f"/Widths {self.n + 1} 0 R")
                self._out(f"/FontDescriptor {self.n + 2} 0 R")
                if font["enc"]:
                    if "diff" in font:
                        self._out(f"/Encoding {nf + font['diff']} 0 R")
                    else:
                        self._out("/Encoding /WinAnsiEncoding")
                self._out(">>")
                self._out("endobj")

                # Widths
                self._newobj()
                cw = font["cw"]
                self._out(
                    "[" + " ".join(cw.get(chr(i), 0) for i in range(32, 256)) + "]"
                )
                self._out("endobj")

                # Descriptor
                self._newobj()
                s = f"<</Type /FontDescriptor /FontName /{name}"
                for k in (
                    "Ascent",
                    "Descent",
                    "CapHeight",
                    "Flags",
                    "FontBBox",
                    "ItalicAngle",
                    "StemV",
                    "MissingWidth",
                ):
                    s += f" /{k} {font['desc'][k]}"

                filename = font["file"]
                if filename:
                    s += " /FontFile"
                    if my_type != "Type1":
                        s += "2"
                    s += f" {self.font_files[filename]['n']} 0 R"
                self._out(f"{s}>>")
                self._out("endobj")
            elif my_type == "TTF":
                self.fonts[font_name]["n"] = self.n + 1
                ttf = TTFontFile()
                fontname = f"MPDFAA+{font['name']}"
                subset = font["subset"]
                del subset[0]
                ttfontstream = ttf.makeSubset(font["ttffile"], subset)
                ttfontsize = len(ttfontstream)
                fontstream = zlib.compress(ttfontstream)
                codeToGlyph = ttf.codeToGlyph
                # del codeToGlyph[0]

                # Type0 Font
                # A composite font - a font composed of other fonts,
                # organized hierarchically
                self._newobj()
                self._out("<</Type /Font")
                self._out("/Subtype /Type0")
                self._out(f"/BaseFont /{fontname}")
                self._out("/Encoding /Identity-H")
                self._out(f"/DescendantFonts [{self.n + 1} 0 R]")
                self._out(f"/ToUnicode {self.n + 2} 0 R")
                self._out(">>")
                self._out("endobj")

                # CIDFontType2
                # A CIDFont whose glyph descriptions are based on
                # TrueType font technology
                self._newobj()
                self._out("<</Type /Font")
                self._out("/Subtype /CIDFontType2")
                self._out(f"/BaseFont /{fontname}")
                self._out(f"/CIDSystemInfo {self.n + 2} 0 R")
                self._out(f"/FontDescriptor {self.n + 3} 0 R")
                if font["desc"].get("MissingWidth"):
                    self._out(f"/DW {font['desc']['MissingWidth']}")
                self._putTTfontwidths(font, ttf.maxUni)
                self._out(f"/CIDToGIDMap {self.n + 4} 0 R")
                self._out(">>")
                self._out("endobj")

                # ToUnicode
                self._newobj()
                toUni = (
                    "/CIDInit /ProcSet findresource begin\n"
                    "12 dict begin\n"
                    "begincmap\n"
                    "/CIDSystemInfo\n"
                    "<</Registry (Adobe)\n"
                    "/Ordering (UCS)\n"
                    "/Supplement 0\n"
                    ">> def\n"
                    "/CMapName /Adobe-Identity-UCS def\n"
                    "/CMapType 2 def\n"
                    "1 begincodespacerange\n"
                    "<0000> <FFFF>\n"
                    "endcodespacerange\n"
                    "1 beginbfrange\n"
                    "<0000> <FFFF> <0000>\n"
                    "endbfrange\n"
                    "endcmap\n"
                    "CMapName currentdict /CMap defineresource pop\n"
                    "end\n"
                    "end"
                )
                self._out(f"<</Length {len(toUni)}>>")
                self._out(pdf_stream(toUni))
                self._out("endobj")

                # CIDSystemInfo dictionary
                self._newobj()
                self._out("<</Registry (Adobe)")
                self._out("/Ordering (UCS)")
                self._out("/Supplement 0")
                self._out(">>")
                self._out("endobj")

                # Font descriptor
                self._newobj()
                self._out("<</Type /FontDescriptor")
                self._out("/FontName /" + fontname)
                for kd in (
                    "Ascent",
                    "Descent",
                    "CapHeight",
                    "Flags",
                    "FontBBox",
                    "ItalicAngle",
                    "StemV",
                    "MissingWidth",
                ):
                    v = font["desc"][kd]
                    if kd == "Flags":
                        v = v | 4
                        v = v & ~32  # SYMBOLIC font flag
                    self._out(f" /{kd} {v}")
                self._out(f"/FontFile2 {self.n + 2} 0 R")
                self._out(">>")
                self._out("endobj")

                # Embed CIDToGIDMap
                # A specification of the mapping from CIDs to glyph indices
                cidtogidmap = ["\x00"] * 256 * 256 * 2
                for cc, glyph in codeToGlyph.items():
                    cidtogidmap[cc * 2] = chr(glyph >> 8)
                    cidtogidmap[cc * 2 + 1] = chr(glyph & 0xFF)
                cidtogidmap = "".join(cidtogidmap)
                # manage binary data as latin1 until PEP461-like function is implemented
                cidtogidmap = zlib.compress(cidtogidmap.encode("latin1"))
                self._newobj()
                self._out(f"<</Length {len(cidtogidmap)}")
                self._out("/Filter /FlateDecode")
                self._out(">>")
                self._out(pdf_stream(cidtogidmap))
                self._out("endobj")

                # Font file
                self._newobj()
                self._out(f"<</Length {len(fontstream)}")
                self._out("/Filter /FlateDecode")
                self._out(f"/Length1 {ttfontsize}")
                self._out(">>")
                self._out(pdf_stream(fontstream))
                self._out("endobj")
                del ttf
            else:
                # Allow for additional types
                mtd = f"_put{my_type.lower()}"
                # check if self has a attr mtd which is callable (method)
                if not callable(getattr(self, mtd, None)):
                    raise FPDFException(f"Unsupported font type: {my_type}")
                # pylint: disable=no-member
                self.mtd(font)

    def _putTTfontwidths(self, font, maxUni):
        cw127fname = Path(font["unifilename"]).with_suffix(".cw127.pkl")
        font_dict = load_cache(cw127fname)
        if font_dict:
            rangeid = font_dict["rangeid"]
            range_ = font_dict["range"]
            prevcid = font_dict["prevcid"]
            prevwidth = font_dict["prevwidth"]
            interval = font_dict["interval"]
            range_interval = font_dict["range_interval"]
            startcid = 128
        else:
            rangeid = 0
            range_ = {}
            range_interval = {}
            prevcid = -2
            prevwidth = -1
            interval = False
            startcid = 1
        cwlen = maxUni + 1

        # for each character
        subset = set(font["subset"])
        for cid in range(startcid, cwlen):
            if cid == 128 and font_dict:
                try:
                    with cw127fname.open("wb") as fh:
                        pickle.dump(font_dict, fh)
                except OSError as e:
                    if e.errno != errno.EACCES:
                        raise  # Not a permission error.

            if cid > 255 and (cid not in subset or cid >= len(font["cw"])):
                continue
            width = font["cw"][cid]
            if width == 0:
                continue
            if width == 65535:
                width = 0

            if "dw" not in font or (font["dw"] and width != font["dw"]):
                if cid == (prevcid + 1):
                    if width == prevwidth:
                        if width == range_[rangeid][0]:
                            range_.setdefault(rangeid, []).append(width)
                        else:
                            range_[rangeid].pop()
                            # new range
                            rangeid = prevcid
                            range_[rangeid] = [prevwidth, width]
                        interval = True
                        range_interval[rangeid] = True
                    else:
                        if interval:
                            # new range
                            rangeid = cid
                            range_[rangeid] = [width]
                        else:
                            range_[rangeid].append(width)
                        interval = False
                else:
                    rangeid = cid
                    range_[rangeid] = [width]
                    interval = False
                prevcid = cid
                prevwidth = width
        prevk = -1
        nextk = -1
        prevint = False

        ri = range_interval
        for k, ws in sorted(range_.items()):
            cws = len(ws)
            if k == nextk and not prevint and (k not in ri or cws < 3):
                if k in ri:
                    del ri[k]
                range_[prevk] = range_[prevk] + range_[k]
                del range_[k]
            else:
                prevk = k
            nextk = k + cws
            if k in ri:
                prevint = cws > 3
                del ri[k]
                nextk -= 1
            else:
                prevint = False
        w = []
        for k, ws in sorted(range_.items()):
            if len(set(ws)) == 1:
                w.append(f" {k} {k + len(ws) - 1} {ws[0]}")
            else:
                w.append(f" {k} [ {' '.join(str(int(h)) for h in ws)} ]\n")
        self._out(f"/W [{''.join(w)}]")

    def _putimages(self):
        for info in sorted(self.images.values(), key=lambda info: info["i"]):
            self._putimage(info)
            del info["data"]
            if "smask" in info:
                del info["smask"]

    def _putimage(self, info):
        if "data" not in info:
            return
        self._newobj()
        info["n"] = self.n
        self._out("<</Type /XObject")
        self._out("/Subtype /Image")
        self._out(f"/Width {info['w']}")
        self._out(f"/Height {info['h']}")

        if info["cs"] == "Indexed":
            self._out(
                f"/ColorSpace [/Indexed /DeviceRGB "
                f"{len(info['pal']) // 3 - 1} {self.n + 1} 0 R]"
            )
        else:
            self._out(f"/ColorSpace /{info['cs']}")
            if info["cs"] == "DeviceCMYK":
                self._out("/Decode [1 0 1 0 1 0 1 0]")

        self._out(f"/BitsPerComponent {info['bpc']}")

        if "f" in info:
            self._out(f"/Filter /{info['f']}")
        if "dp" in info:
            self._out(f"/DecodeParms <<{info['dp']}>>")

        if "trns" in info and isinstance(info["trns"], list):
            trns = " ".join(f"{x} {x}" for x in info["trns"])
            self._out(f"/Mask [{trns}]")

        if "smask" in info:
            self._out(f"/SMask {self.n + 1} 0 R")

        self._out(f"/Length {len(info['data'])}>>")
        self._out(pdf_stream(info["data"]))
        self._out("endobj")

        # Soft mask
        if "smask" in info:
            dp = f"/Predictor 15 /Colors 1 /BitsPerComponent 8 /Columns {info['w']}"
            smask = {
                "w": info["w"],
                "h": info["h"],
                "cs": "DeviceGray",
                "bpc": 8,
                "f": info["f"],
                "dp": dp,
                "data": info["smask"],
            }
            self._putimage(smask)

        # Palette
        if info["cs"] == "Indexed":
            self._newobj()
            filter, pal = (
                ("/Filter /FlateDecode ", zlib.compress(info["pal"]))
                if self.compress
                else ("", info["pal"])
            )
            self._out(f"<<{filter}/Length {len(pal)}>>")
            self._out(pdf_stream(pal))
            self._out("endobj")

    def _putxobjectdict(self):
        i = [(x["i"], x["n"]) for x in self.images.values()]
        i.sort()
        for idx, n in i:
            self._out(f"/I{idx} {n} 0 R")

    def _putresourcedict(self):
        # From section 10.1, "Procedure Sets", of PDF 1.7 spec:
        # > Beginning with PDF 1.4, this feature is considered obsolete.
        # > For compatibility with existing consumer applications,
        # > PDF producer applications should continue to specify procedure sets
        # > (preferably, all of those listed in Table 10.1).
        self._out("/ProcSet [/PDF /Text /ImageB /ImageC /ImageI]")
        self._out("/Font <<")
        f = [(x["i"], x["n"]) for x in self.fonts.values()]
        f.sort()
        for idx, n in f:
            self._out(f"/F{idx} {pdf_ref(n)}")
        self._out(">>")
        self._out("/XObject <<")
        self._putxobjectdict()
        self._out(">>")

    def _putresources(self):
        with self._trace_size("resources.fonts"):
            self._putfonts()
        with self._trace_size("resources.images"):
            self._putimages()

        # Resource dictionary
        with self._trace_size("resources.dict"):
            self.offsets[2] = len(self.buffer)
            self._out("2 0 obj")
            self._out("<<")
            self._putresourcedict()
            self._out(">>")
            self._out("endobj")

    def _put_structure_tree(self):
        "Builds a Structure Hierarchy, including image alternate descriptions"
        # This property is later used by _putcatalog to insert a reference to the StructTreeRoot:
        self._struct_tree_root_obj_id = self.n + 1
        self.struct_builder.serialize(
            first_object_id=self._struct_tree_root_obj_id, fpdf=self
        )

    def _put_document_outline(self):
        # This property is later used by _putcatalog to insert a reference to the Outlines:
        self._outlines_obj_id = self.n + 1
        serialize_outline(
            self._outline, first_object_id=self._outlines_obj_id, fpdf=self
        )

    def _put_xmp_metadata(self):
        xpacket = f'<?xpacket begin="ï»¿" id="W5M0MpCehiHzreSzNTczkc9d"?>\n{self.xmp_metadata}\n<?xpacket end="w"?>\n'
        self._newobj()
        self._out(f"<</Type /Metadata /Subtype /XML /Length {len(xpacket)}>>")
        self._out(pdf_stream(xpacket))
        self._out("endobj")
        self._xmp_metadata_obj_id = self.n

    def _putinfo(self):
        info_d = {
            "/Title": enclose_in_parens(getattr(self, "title", None)),
            "/Subject": enclose_in_parens(getattr(self, "subject", None)),
            "/Author": enclose_in_parens(getattr(self, "author", None)),
            "/Keywords": enclose_in_parens(getattr(self, "keywords", None)),
            "/Creator": enclose_in_parens(getattr(self, "creator", None)),
            "/Producer": enclose_in_parens(getattr(self, "producer", None)),
        }

        if hasattr(self, "creation_date"):
            try:
                creation_date = self.creation_date
                date_string = f"{creation_date:%Y%m%d%H%M%S}"
            except Exception as error:
                raise FPDFException(
                    f"Could not format date: {creation_date}"
                ) from error
        else:
            date_string = f"{datetime.now():%Y%m%d%H%M%S}"
        info_d["/CreationDate"] = enclose_in_parens(f"D:{date_string}")

        self._out(pdf_d(info_d, open_dict="", close_dict="", has_empty_fields=True))

    def _putcatalog(self):
        catalog_d = {
            "/Type": "/Catalog",
            # Pages is always the 1st object of the document, cf. the end of _putpages:
            "/Pages": pdf_ref(1),
        }
        lang = enclose_in_parens(getattr(self, "lang", None))
        if lang:
            catalog_d["/Lang"] = lang

        if self.zoom_mode in ZOOM_CONFIGS:
            zoom_config = [
                pdf_ref(3),  # reference to object ID of the 1st page
                *ZOOM_CONFIGS[self.zoom_mode],
            ]
        else:  # zoom_mode is a number, not one of the allowed strings:
            zoom_config = ["/XYZ", "null", "null", str(self.zoom_mode / 100)]
        catalog_d["/OpenAction"] = pdf_l(zoom_config)

        if self.layout_mode in LAYOUT_NAMES:
            catalog_d["/PageLayout"] = LAYOUT_NAMES[self.layout_mode]
        if self._xmp_metadata_obj_id:
            catalog_d["/Metadata"] = pdf_ref(self._xmp_metadata_obj_id)
        if self._struct_tree_root_obj_id:
            catalog_d["/MarkInfo"] = pdf_d({"/Marked": "true"})
            catalog_d["/StructTreeRoot"] = pdf_ref(self._struct_tree_root_obj_id)
        if self._outlines_obj_id:
            catalog_d["/Outlines"] = pdf_ref(self._outlines_obj_id)

        self._out(pdf_d(catalog_d, open_dict="", close_dict=""))

    def _putheader(self):
        self._out(f"%PDF-{self.pdf_version}")

    def _puttrailer(self):
        self._out(f"/Size {self.n + 1}")
        self._out(f"/Root {pdf_ref(self.n)}")  # Catalog object index
        self._out(f"/Info {pdf_ref(self.n - 1)}")  # Info object index

    def _enddoc(self):
        LOGGER.debug("Final doc sections size summary:")
        with self._trace_size("header"):
            self._putheader()
        with self._trace_size("pages"):
            self._putpages()
        self._putresources()  # trace_size is performed inside
        if not self.struct_builder.empty():
            with self._trace_size("structure_tree"):
                self._put_structure_tree()
        if self._outline:
            with self._trace_size("document_outline"):
                self._put_document_outline()
        if self.xmp_metadata:
            self._put_xmp_metadata()
        # Info
        with self._trace_size("info"):
            self._newobj()
            self._out("<<")
            self._putinfo()
            self._out(">>")
            self._out("endobj")
        # Catalog
        with self._trace_size("catalog"):
            self._newobj()
            self._out("<<")
            self._putcatalog()
            self._out(">>")
            self._out("endobj")
        # Cross-ref
        with self._trace_size("xref"):
            o = len(self.buffer)
            self._out("xref")
            self._out(f"0 {self.n + 1}")
            self._out("0000000000 65535 f ")
            for i in range(1, self.n + 1):
                self._out(f"{self.offsets[i]:010} 00000 n ")
        # Trailer
        with self._trace_size("trailer"):
            self._out("trailer")
            self._out("<<")
            self._puttrailer()
            self._out(">>")
            self._out("startxref")
            self._out(o)
        self._out("%%EOF")
        self.state = DocumentState.CLOSED

    def _beginpage(self, orientation, format, same):
        self.page += 1
        self.pages[self.page] = {"content": bytearray()}
        self.state = DocumentState.GENERATING_PAGE
        self.x = self.l_margin
        self.y = self.t_margin
        self.font_family = ""
        self.font_stretching = 100
        if not same:
            # Page format
            if format:
                # Change page format
                fw_pt, fh_pt = get_page_format(format, self.k)
            else:
                # Set to default format
                fw_pt = self.dw_pt
                fh_pt = self.dh_pt
            # Page orientation
            orientation = (
                orientation[0].upper() if orientation else self.def_orientation
            )
            if orientation == "P":
                self.w_pt = fw_pt
                self.h_pt = fh_pt
            else:
                self.w_pt = fh_pt
                self.h_pt = fw_pt
            self.w = self.w_pt / self.k
            self.h = self.h_pt / self.k
            self.page_break_trigger = self.h - self.b_margin
            self.cur_orientation = orientation
        self.pages[self.page]["w_pt"] = self.w_pt
        self.pages[self.page]["h_pt"] = self.h_pt

    def _endpage(self):
        # End of page contents
        self.state = DocumentState.READY

    def _newobj(self):
        # Begin a new object
        self.n += 1
        self.offsets[self.n] = len(self.buffer)
        self._out(f"{self.n} 0 obj")
        return self.n

    def _dounderline(self, x, y, txt):
        # Underline text
        up = self.current_font["up"]
        ut = self.current_font["ut"]
        w = self.get_string_width(txt, True) + self.ws * txt.count(" ")
        return (
            f"{x * self.k:.2f} "
            f"{(self.h - y + up / 1000 * self.font_size) * self.k:.2f} "
            f"{w * self.k:.2f} {-ut / 1000 * self.font_size_pt:.2f} re f"
        )

    def _out(self, s):
        if self.state == DocumentState.CLOSED:
            raise FPDFException(
                "Content cannot be added on a closed document, after calling output()"
            )
        if not isinstance(s, bytes):
            if not isinstance(s, str):
                s = str(s)
            s = s.encode("latin1")
        if self.state == DocumentState.GENERATING_PAGE:
            self.pages[self.page]["content"] += s + b"\n"
        else:
            self.buffer += s + b"\n"

    @check_page
    def interleaved2of5(self, txt, x, y, w=1, h=10):
        """Barcode I2of5 (numeric), adds a 0 if odd length"""
        narrow = w / 3
        wide = w

        # wide/narrow codes for the digits
        bar_char = {
            "0": "nnwwn",
            "1": "wnnnw",
            "2": "nwnnw",
            "3": "wwnnn",
            "4": "nnwnw",
            "5": "wnwnn",
            "6": "nwwnn",
            "7": "nnnww",
            "8": "wnnwn",
            "9": "nwnwn",
            "A": "nn",
            "Z": "wn",
        }

        self.set_fill_color(0)
        code = txt
        # add leading zero if code-length is odd
        if len(code) % 2 != 0:
            code = f"0{code}"

        # add start and stop codes
        code = f"AA{code.lower()}ZA"

        for i in range(0, len(code), 2):
            # choose next pair of digits
            char_bar = code[i]
            char_space = code[i + 1]
            # check whether it is a valid digit
            if char_bar not in bar_char:
                raise RuntimeError(f'Char "{char_bar}" invalid for I25:')
            if char_space not in bar_char:
                raise RuntimeError(f'Char "{char_space}" invalid for I25: ')

            # create a wide/narrow-seq (first digit=bars, second digit=spaces)
            seq = "".join(
                f"{cb}{cs}" for cb, cs in zip(bar_char[char_bar], bar_char[char_space])
            )

            for bar, char in enumerate(seq):
                # set line_width depending on value
                line_width = narrow if char == "n" else wide

                # draw every second value, the other is represented by space
                if bar % 2 == 0:
                    self.rect(x, y, line_width, h, "F")

                x += line_width

    @check_page
    def code39(self, txt, x, y, w=1.5, h=5):
        """Barcode 3of9"""
        dim = {"w": w, "n": w / 3}
        chars = {
            "0": "nnnwwnwnn",
            "1": "wnnwnnnnw",
            "2": "nnwwnnnnw",
            "3": "wnwwnnnnn",
            "4": "nnnwwnnnw",
            "5": "wnnwwnnnn",
            "6": "nnwwwnnnn",
            "7": "nnnwnnwnw",
            "8": "wnnwnnwnn",
            "9": "nnwwnnwnn",
            "A": "wnnnnwnnw",
            "B": "nnwnnwnnw",
            "C": "wnwnnwnnn",
            "D": "nnnnwwnnw",
            "E": "wnnnwwnnn",
            "F": "nnwnwwnnn",
            "G": "nnnnnwwnw",
            "H": "wnnnnwwnn",
            "I": "nnwnnwwnn",
            "J": "nnnnwwwnn",
            "K": "wnnnnnnww",
            "L": "nnwnnnnww",
            "M": "wnwnnnnwn",
            "N": "nnnnwnnww",
            "O": "wnnnwnnwn",
            "P": "nnwnwnnwn",
            "Q": "nnnnnnwww",
            "R": "wnnnnnwwn",
            "S": "nnwnnnwwn",
            "T": "nnnnwnwwn",
            "U": "wwnnnnnnw",
            "V": "nwwnnnnnw",
            "W": "wwwnnnnnn",
            "X": "nwnnwnnnw",
            "Y": "wwnnwnnnn",
            "Z": "nwwnwnnnn",
            "-": "nwnnnnwnw",
            ".": "wwnnnnwnn",
            " ": "nwwnnnwnn",
            "*": "nwnnwnwnn",
            "$": "nwnwnwnnn",
            "/": "nwnwnnnwn",
            "+": "nwnnnwnwn",
            "%": "nnnwnwnwn",
        }
        self.set_fill_color(0)
        for c in txt.upper():
            if c not in chars:
                raise RuntimeError(f'Invalid char "{c}" for Code39')
            for i, d in enumerate(chars[c]):
                if i % 2 == 0:
                    self.rect(x, y, dim[d], h, "F")
                x += dim[d]
            x += dim["n"]

    @check_page
    @contextmanager
    def rect_clip(self, x, y, w, h):
        self._out(
            (
                f"q {x * self.k:.2f} {(self.h - y - h) * self.k:.2f} {w * self.k:.2f} "
                f"{h * self.k:.2f} re W n\n"
            )
        )
        yield
        self._out("Q\n")

    @contextmanager
    def _trace_size(self, label):
        prev_size = len(self.buffer)
        yield
        LOGGER.debug("- %s.size: %s", label, _sizeof_fmt(len(self.buffer) - prev_size))

    @contextmanager
    def unbreakable(self):
        """
        Ensures that all rendering performed in this context appear on a single page
        by performing page break beforehand if need be.

        Note that using this method means to duplicate the FPDF `bytearray` buffer:
        when generating large PDFs, doubling memory usage may be troublesome.
        """
        prev_page, prev_y = self.page, self.y
        recorder = FPDFRecorder(self, accept_page_break=False)
        LOGGER.debug("Starting unbreakable block")
        yield recorder
        y_scroll = recorder.y - prev_y + (recorder.page - prev_page) * self.eph
        if prev_y + y_scroll > self.page_break_trigger:
            LOGGER.debug("Performing page jump due to unbreakable height")
            recorder.rewind()
            # pylint: disable=protected-access
            # Performing this call through .pdf so that it does not get recorded & replayed:
            assert recorder.pdf._perform_page_break_if_need_be(y_scroll)
            recorder.replay()
        LOGGER.debug("Ending unbreakable block")

    @check_page
    def insert_toc_placeholder(self, render_toc_function, pages=1):
        """
        Configure Table Of Contents rendering at the end of the document generation,
        and reserve some vertical space right now in order to insert it.

        Args:
            render_toc_function (function): a function that will be invoked to render the ToC.
                This function will receive 2 parameters: `pdf`, an instance of FPDF, and `outline`,
                a list of `OutlineSection`.
            pages (int): the number of pages that the Table of Contents will span,
                including the current one that will. As many page breaks as the value of this argument
                will occur immediately after calling this method.
        """
        if not callable(render_toc_function):
            raise TypeError(
                f"The first argument must be a callable, got: {type(render_toc_function)}"
            )
        if self._toc_placeholder:
            raise FPDFException(
                "A placeholder for the table of contents has already been defined"
                f" on page {self._toc_placeholder.start_page}"
            )
        self._toc_placeholder = ToCPlaceholder(
            render_toc_function, self.page, self.y, pages
        )
        for _ in range(pages):
            self.add_page()

    def set_section_title_styles(
        self,
        level0,
        level1=None,
        level2=None,
        level3=None,
        level4=None,
        level5=None,
        level6=None,
    ):
        """
        Defines a style for section titles.
        After calling this method, calls to `start_section` will render section names visually.

        Args:
            level0 (TitleStyle): style for the top level section titles
            level1 (TitleStyle): optional style for the level 1 section titles
            level2 (TitleStyle): optional style for the level 2 section titles
            level3 (TitleStyle): optional style for the level 3 section titles
            level4 (TitleStyle): optional style for the level 4 section titles
            level5 (TitleStyle): optional style for the level 5 section titles
            level6 (TitleStyle): optional style for the level 6 section titles
        """
        for level in (level0, level1, level2, level3, level4, level5, level6):
            if level and not isinstance(level, TitleStyle):
                raise TypeError(
                    f"Arguments must all be TitleStyle instances, got: {type(level)}"
                )
        self.section_title_styles = {
            0: level0,
            1: level1,
            2: level2,
            3: level3,
            4: level4,
            5: level5,
            6: level6,
        }

    @check_page
    def start_section(self, name, level=0):
        """
        Start a section in the document outline.
        If section_title_styles have been configured,
        render the section name visually as a title.

        Args:
            name (str): section name
            level (int): section level in the document outline. 0 means top-level.
        """
        if level < 0:
            raise ValueError('"level" mut be equal or greater than zero')
        if self._outline:
            if level > self._outline[-1].level + 1:
                raise ValueError(
                    f"Incoherent hierarchy: cannot start a level {level} section after a level {self._outline[-1].level} one"
                )
        internal_link = InternalLink(self._current_page_object_id(), self.y)
        struct_elem = None
        if self.section_title_styles:
            with self._marked_sequence(title=name) as marked_content:
                struct_elem = self.struct_builder.struct_elem_per_mc[marked_content]
                with self._apply_style(self.section_title_styles[level]):
                    self.multi_cell(w=self.epw, h=self.font_size, txt=name, ln=1)
        self._outline.append(
            OutlineSection(name, level, self.page, internal_link, struct_elem)
        )

    @contextmanager
    def _apply_style(self, title_style):
        prev_font = (self.font_family, self.font_style, self.font_size_pt)
        self.set_font(
            title_style.font_family, title_style.font_style, title_style.font_size_pt
        )
        prev_text_color = self.text_color
        if title_style.color is not None:
            if isinstance(title_style.color, Sequence):
                self.set_text_color(*title_style.color)
            else:
                self.set_text_color(title_style.color)
        prev_underline = self.underline
        self.underline = title_style.underline
        if title_style.t_margin:
            self.ln(title_style.t_margin)
        if title_style.l_margin:
            self.set_x(title_style.l_margin)
        yield
        if title_style.b_margin:
            self.ln(title_style.b_margin)
        self.set_font(*prev_font)
        self.text_color = prev_text_color
        self.underline = prev_underline


def _sizeof_fmt(num, suffix="B"):
    # Recipe from: https://stackoverflow.com/a/1094933/636849
    for unit in ["", "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi"]:
        if abs(num) < 1024:
            return f"{num:3.1f}{unit}{suffix}"
        num /= 1024
    return f"{num:.1f}Yi{suffix}"


sys.modules[__name__].__class__ = WarnOnDeprecatedModuleAttributes


__all__ = ["FPDF", "load_cache", "get_page_format", "TitleStyle", "PAGE_FORMATS"]
